! source file: /den/eby/UVic_ESCM/2.6/source/embm/insolation.F
      subroutine insolation (dayoyr, phi, solins)
!=======================================================================
!     calculate daily insolation
!     based on code by a.fanning and a.berger

!     input:
!       dayoyr = days + fractional days since start of calendar year
!       phi    = latitude in degrees
!     output:
!       solins = insolation (units as for solarconst)

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      common /insol_r/ ecc, perh, xob

      include "ndcon.h"
      include "cembm.h"

      integer nd

      real adelta, anm, anv, aphi, cosd, cosp, dayl, dayoyr, delta
      real dlamm, ecc, perh, phi, phid, pir, ranm, ranv, rdayl
      real rdelta, rphi, s, sind, sinp, solins, step, stp, test
      real tls, tp, ww, xec, xee, xl, xlam, xllp, xob, xse

      pir = pi/180.0
      step = 360.0/365.25
      test = 1.0e-4

      nd = nint(dayoyr)

!     longitude of perihelion relative to vernal equinox
      xl = perh + 180.0

!     calendar date  month (ma) and day (ja)
!     nd  number of this day in a year of 365 days
!     xlam = mean long. sun for true long. = 0
!     dlamm = mean long. sun for ma-ja

      xllp = xl*pir
      xee = ecc**2.0
      xse = sqrt(1.0-xee)
      xlam = (ecc/2.0 + ecc*xee/8.0)*(1.0+xse)*sin(xllp)
     &     - xee/4.0*(0.5 + xse)*sin(2.0*xllp)
     &     + ecc*xee/8.0*(1.0/3.0 + xse)*sin(3.0*xllp)
      xlam = 2.0*xlam/pir
      dlamm = xlam + (nd - 80)*step
      anm = dlamm - xl
      ranm = anm*pir
      xec = xee*ecc
      ranv = ranm + (2.0*ecc - xec/4.0)*sin(ranm)
     &     + 5.0/4.0*ecc**2*sin(2.0*ranm)
     &     + 13.0/12.0*xec*sin(3.0*ranm)
      anv = ranv/pir

!     true longitude of the earth
      tls = anv + xl

      rphi = phi*pir
      s = 1.0/(pi*((1.0-ecc*ecc)/(1.0+ecc*cos((tls-xl)*pir)))**2)
      sind = sin(xob*pir)*sin(tls*pir)
      cosd = sqrt(1.0-sind**2)
      rdelta = atan(sind/cosd)
      delta = rdelta/pir
      aphi = abs(phi)
      adelta = abs(delta)

!     singularity for aphi = 90 and delta = 0 => polar night
      if ((abs(aphi - 90.0) .gt. test) .or. (adelta .gt. test)) then
        if(adelta .le. test) then
!         equinoxes (delta = 0)
          dayl = 12.0
          ww = s*cos(rphi)
        else
          if(aphi .le. test) then
!           equator (phi = 0)
            dayl = 12.0
            ww = s*cos(rdelta)
          else
            phid = phi*delta
            if ((aphi .gt. 90.0 - adelta) .and. (phid .ne. 0.0)) then
              if (phid .gt. 0.0) then
!               polar continual day
                dayl = 24.00
                ww = s*sind*sin(rphi)*pi
              else
!               polar continual night
                dayl = 0.0
                ww = 0.0
              endif
            else
!             daily sunrise and sunset
              sinp = sind*sin(rphi)
              cosp = cosd*cos(rphi)
              tp = -sinp/cosp
              stp = sqrt(1.0-tp*tp)
              rdayl = acos(tp)
              dayl = 24.0*rdayl/pi
              ww = s*(rdayl*sinp+cosp*stp)
            endif
          endif
        endif
      else
!       polar continual night
        dayl = 0.0
        ww = 0.0
      endif

      solins = solarconst*ww

      return
      end

      subroutine berger (yr)
!=======================================================================
!     calculate eccentricity, precession and obliquity for any year
!     based on code by a.fanning and a.berger

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

!     note: this solution of berger 1978 is valid only for 1,000,000
!           years centered on present-day. for longer period the
!           solution 1990 must be used (contact Berger for this)

!     refer to :
!     Berger A. 1978. a simple algorithm to compute long term
!                     variations of daily or monthly insolation
!                     contr. 18  Inst of Astronomy and Geophysics
!                     Universite Catholique de Louvain
!                     Louvain-la-Neuve  Belgium

!     Berger A. 1978. long term variations of daily insolation
!                     and quaternary climatic changes
!                     J. of Atmospheric Sciences  35  2362-2367

!     nef  nob  nop have been reduced to  19  18  9

!     these are the expansion coefficients for minimal efficiency, they
!     can be increased by referring to Berger, 1978:
!     eccentricity           ecc   table 1
!     obliquity              xob   table 2
!     longitude perihelion   perh  table 3

      include "ndcon.h"
      include "stdunits.h"

      integer nef, nob, nop
      parameter (nef=19, nob=18, nop=9)

      integer i

      real arg, ecc, perh, pir, pirr, pre, prm, prg, rp, t
      real xec, xes, xob, xod, xop, yr

      common /insol_r/ ecc, perh, xob

      logical done

      real ae(nef), be(nef), ce(nef), ye(nef), ze(nef)
      real aob(nob), bob(nob), cob(nob), yob(nob), zob(nob)
      real aop(nop), bop(nop), cop(nop), yop(nop), zop(nop)

!     coefficients for solution of eccentricity

      data ae / 0.01860798, 0.01627522, -.01300660,
     &  0.00988829, -.00336700, 0.0033077, -.00235400,
     &  0.00140015, 0.00100700, 0.00085700, 0.00064990,
     &  0.00059900, 0.00037800, -.00033700, 0.00027600,
     &  0.00018200, -.00017400, -.00012400, 0.00001250 /

      data ye / 4.2072050, 7.3460910, 17.8572630,
     &  17.2205460, 16.8467330, 5.1990790, 18.2310760,
     &  26.2167580, 6.3591690, 16.2100160, 3.0651810,
     &  16.5838290, 18.4939800, 6.1909530, 18.8677930,
     &  17.4255670, 6.1860010, 18.4174410, 0.6678630 /

      data ze / 28.620089, 193.788772, 308.307024,
     &  320.199637, 279.376984, 87.195000, 349.129677,
     &  128.443387, 154.143880, 291.269597, 114.860583,
     &  332.092251, 296.414411, 145.769910, 337.237063,
     &  152.092288, 126.839891, 210.667199, 72.108838 /

!     coefficients for solution of obliquity

      data aob / -2462.2214466, -857.3232075, -629.3231835,
     &  -414.2804924, -311.7632587, 308.9408604, -162.5533601,
     &  -116.1077911, 101.1189923, -67.6856209, 24.9079067,
     &  22.5811241, -21.1648355, -15.6549876, 15.3936813,
     &  14.6660938, -11.7273029, 10.2742696 /

      data yob / 31.609974, 32.620504, 24.172203,
     &  31.983787, 44.828336, 30.973257, 43.668246,
     &  32.246691, 30.599444, 42.681324, 43.836462,
     &  47.439436, 63.219948, 64.230478, 1.010530,
     &  7.437771, 55.782177, 0.373813 /

      data zob / 251.9025, 280.8325, 128.3057,
     &  292.7252, 15.3747, 263.7951, 308.4258, 240.0099,
     &  222.9725, 268.7809, 316.7998, 319.6024, 143.8050,
     &  172.7351, 28.9300, 123.5968, 20.2082, 40.8226 /

!       coefficients for solution of precession

      data aop / 7391.0225890, 2555.1526947, 2022.7629188,
     &  -1973.6517951, 1240.2321818, 953.8679112, -931.7537108,
     &  872.3795383, 606.3544732 /

      data yop / 31.609974, 32.620504, 24.172203,
     &  0.636717, 31.983787, 3.138886, 30.973257,
     &  44.828336, 0.991874 /

      data zop / 251.9025, 280.8325, 128.3057, 348.1074,
     &  292.7252, 165.1686, 263.7951, 15.3747, 58.5749 /

      pir = pi/180.0
      pirr = pir/3600.0
      xod = 23.320556
      xop = 3.392506
      prm = 50.439273

!     eccentricity

!     number of terms to keep for expansion solution of eccentricity
      do i=1,nef
        be(i) = ye(i)*pirr
        ce(i) = ze(i)*pir
      enddo

!     obliquity relative to mean ecliptic of date

!     number of terms to keep for expansion solution of obliquity
      do i=1,nob
        bob(i) = yob(i)*pirr
        cob(i) = zob(i)*pir
      enddo

!     general precession in longitude

!     number of terms to keep in series expansion of solution of
!     general precession
      do i=1,nop
        bop(i) = yop(i)*pirr
        cop(i) = zop(i)*pir
      enddo

!     Berger's solution is referenced to 1950.
!     convert t such that zero is referenced about calendar year 0

      t = yr - 1950

!-----------------------------------------------------------------------
!      calculate eccentricity (ecc)
!-----------------------------------------------------------------------

      xes = 0.0
      xec = 0.0
      do i=1, nef
        arg = be(i)*t + ce(i)
        xes = xes + ae(i)*sin(arg)
        xec = xec + ae(i)*cos(arg)
      enddo

      ecc = sqrt(xes*xes + xec*xec)

!-----------------------------------------------------------------------
!     calculate precession (pre)
!-----------------------------------------------------------------------

      if (abs(xec) .gt. 1.0e-08) then
        rp = atan(xes/xec)
        if (xec .lt. 0.0) rp = rp + pi
        if (xec .gt. 0.0 .and. xes .lt. 0.0) rp = rp + 2.0*pi
      else
        if (xes .lt. 0.0) rp = 1.5*pi
        if (xes .eq. 0.0) rp = 0.0
        if (xes .gt. 0.0) rp = pi/2.0
      endif
      perh = rp/pir

      prg = prm*t
      do i=1, nop
        arg = bop(i)*t + cop(i)
        prg = prg + aop(i)*sin(arg)
      enddo
      prg = (prg/3600.0) + xop
      perh = perh + prg
      if (abs(perh) .gt. 360.) perh = perh-float(int(perh/360.0))*360.0
      if (perh .lt. 0.0) perh = perh + 360.0

      pre = ecc*sin(perh*pir)

!-----------------------------------------------------------------------
!     calculate obliquity (xob)
!-----------------------------------------------------------------------

      xob = xod
      do i=1,nob
        arg = bob(i)*t + cob(i)
        xob = xob+aob(i)/3600.0*cos(arg)
      enddo

!-----------------------------------------------------------------------
!     write summary
!-----------------------------------------------------------------------

      write(stdout,'(a,f10.1,a,f10.7,a,f10.6,a,f10.5)')
     &  ' orbital year =', t+1950.0, ' eccentricity =', ecc
     &, ' obliquity =', xob, ' longitude of perihelion =', perh+180.0

        return
        end
