! source file: /den/eby/UVic_ESCM/2.6/source/mom/ocn_rest.F
      subroutine ocn_rest_in (fname, ids, ide, jds, jde)
!=======================================================================
!     input routine for atmospheric restarts

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

!      implicit none

      include "param.h"

      include "coord.h"
      include "emode.h"
      include "grdvar.h"
      include "iounit.h"
      include "levind.h"
      include "mw.h"
      include "task_on.h"
      include "tmngr.h"
      include "switch.h"

      character(*) :: fname
      character(80) :: name
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, ids, ide, jds, jde, igs, ige, ig
      integer jgs, jge, jg, kgs, kge, kg, ils, ile, jls, jle, kls, kle
      integer is(10), ic(10), it(10), iu(10), id_time, id_xt, id_xu
      integer id_yt, id_yu, id_zt, id_zw, id_xt_e, id_xu_e, id_yt_e
      integer id_yu_e, id_zt_e, id_zw_e, id_part
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)
      real zt_e(km+1), zw_e(km+1), time, tmp(1), rkmt(ids:ide,jds:jde)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      time = 0.
      ntrec = 1
      call openfile (name, time, ntrec, iou)

!-----------------------------------------------------------------------
!     get global read domain size (may be less than global domain)
!-----------------------------------------------------------------------
      call getaxis('xt', iou, imt, xt, igs, ige, 1., 0.)
      call getaxis('xu', iou, imt, xu, igs, ige, 1., 0.)
      ig  = ige-igs+1
      call getaxis('yt', iou, jmt, yt, jgs, jge, 1., 0.)
      call getaxis('yu', iou, jmt, yu, igs, ige, 1., 0.)
      jg  = jge-jgs+1
      call getaxis('zt', iou, km, zt, kgs, kge, 0.01, 0.)
      call getaxis('zw', iou, km, zw, kgs, kge, 0.01, 0.)
      kg  = kge-kgs+1

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global read domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)
      kls = max(1,kgs)
      kle = min(km,kge)

!-----------------------------------------------------------------------
!     read 1d data (t)
!-----------------------------------------------------------------------
      call getvara ('itt', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      itt = tmp(1)
      call getvara ('irstdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      irstdy = tmp(1)
      call getvara ('msrsdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      msrsdy = tmp(1)

      call getvara ('year', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nyear = tmp(1)
      call getvara ('month', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nmonth = tmp(1)
      call getvara ('day', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nday = tmp(1)
      call getvara ('hour', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nhour = tmp(1)
      call getvara ('minute', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nmin = tmp(1)
      call getvara ('second', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nsec = tmp(1)
      call mkstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      if (init_time) then
        itt = 0
        irstdy = 0
        msrsdy = 0
        iday(imodeltime) = 0
        msday(imodeltime) = 0
        relyr = 0.0
        call mkstmp (stamp, year0, month0, day0, hour0, min0, sec0)
      endif

!-----------------------------------------------------------------------
!     read 2d data (x,y)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      ln = ic(1)*ic(2)
      call getvara ('kmt', iou, ln, is, ic, rkmt(ils:ile,jls:jle)
     &, 1., 0.)
      kmt(ils:ile,jls:jle) = rkmt(ils:ile,jls:jle)

!-----------------------------------------------------------------------
!     read 3d data (x,y,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      is(3) = 1
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)

      taum1disk = mod(itt+1,2) + 1
      taudisk   = mod(itt  ,2) + 1
      taup1disk = taum1disk

!     update pointers to tau-1, tau, & tau+1 data in the MW based on itt
      if (wide_open_mw) then
!       rotate time levels instead of moving data
        taum1 = mod(itt+0,3) - 1
        tau   = mod(itt+1,3) - 1
        taup1 = mod(itt+2,3) - 1
      endif

!     first do psi at "tau" then at "tau+1"
      call getvara ('psi1', iou, ln, is, ic, psi(ils:ile,jls:jle,1)
     &, 1., 0.)
      call getvara ('psi2', iou, ln, is, ic, psi(ils:ile,jls:jle,2)
     &, 1., 0.)
!     guess fields
      call getvara ('ptd1', iou, ln, is, ic, ptd(ils:ile,jls:jle)
     &, 1., 0.)
      call oput (kflds, nwds, nkflds-1, ptd)
      call getvara ('ptd2', iou, ln, is, ic, ptd(ils:ile,jls:jle)
     &, 1., 0.)
      call oput (kflds, nwds, nkflds, ptd)

!     construct depth arrays associated with "u" cells

      call depth_u (kmt, imt, jmt, zw, km, kmu, h, hr)

!-----------------------------------------------------------------------
!     read 4d data (x,y,z,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = 1
      is(3) = 1
      ic(3) = kle-jls+1
      is(4) = 1
      ic(4) = 1
      ln = ic(1)*ic(2)*ic(3)*ic(4)

!     read the "tau" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
        endif
        is(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          call getvara('tracer1_'//trim(a3), iou, ln, is, ic
     &,     t(ils:ile,kls:kle,j,n,tau), 1., 0.)
        enddo
        call getvara('u1', iou, ln, is, ic, u(ils:ile,kls:kle,j,1,tau)
     &,   1., 0.)
        call getvara('v1', iou, ln, is, ic, u(ils:ile,kls:kle,j,2,tau)
     &,   1., 0.)
!       initialize every latitude
        if (wide_open_mw) then
!         do nothing since "tau" data is in place in the MW
        else
          call putrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
        endif
      enddo
      if (wide_open_mw) then
!       Initialze 1st and last latitude row for tau-1 to prevent
!       use of uninitialized values on boundary row.
        do j=1,jmt,jmt-1
          do k=1,km
            do i=1,imt
              u(i,k,j,1,taum1) = u(i,k,j,1,tau)
              u(i,k,j,2,taum1) = u(i,k,j,2,tau)
              do n=1,nvar-2
                t(i,k,j,n,taum1) = t(i,k,j,n,tau)
              enddo
            enddo
          enddo
        enddo
      endif

!     read the "tau+1" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
        endif
        is(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          call getvara('tracer2_'//trim(a3), iou, ln, is, ic
     &,     t(ils:ile,kls:kle,j,n,taup1), 1., 0.)
        enddo
        call getvara('u2', iou, ln, is, ic, u(ils:ile,kls:kle,j,1,taup1)
     &,   1., 0.)
        call getvara('v2', iou, ln, is, ic, u(ils:ile,kls:kle,j,2,taup1)
     &,   1., 0.)
!       initialize every latitude
        if (wide_open_mw) then
!         do nothing since "tau+1" data is in place in the MW
        else
          call putrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif
      enddo

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)

      print*, '=> Ocn restart read from ',trim(fname),' on ', stamp

      return
      end

      subroutine ocn_rest_out (fname, ids, ide, jds, jde)
!=======================================================================
!     output routine for atmospheric restarts

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

!      implicit none

      include "param.h"

      include "coord.h"
      include "emode.h"
      include "grdvar.h"
      include "iounit.h"
      include "levind.h"
      include "mw.h"
      include "switch.h"
      include "task_on.h"
      include "tmngr.h"

      character(*) :: fname
      character(80) :: name
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, ids, ide, jds, jde, igs, ige, ig
      integer jgs, jge, jg, kgs, kge, kg, ils, ile, jls, jle, kls, kle
      integer is(10), ic(10), it(10), iu(10), id_time, id_xt, id_xu
      integer id_yt, id_yu, id_zt, id_zw, id_xt_e, id_xu_e, id_yt_e
      integer id_yu_e, id_zt_e, id_zw_e, id_part
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)
      real zt_e(km+1), zw_e(km+1), rkmt(ids:ide,jds:jde)
      real time, tmp(1), bufsl(imt,km,2), ext(imt,2)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      time = 0.
      ntrec = -1
      call openfile (name, time, ntrec, iou)

!-----------------------------------------------------------------------
!     set global write domain size
!-----------------------------------------------------------------------
      igs = 1
      ige = imt
      ig  = ige-igs+1
      jgs = 1
      jge = jmt
      jg  = jge-jgs+1
      kgs = 1
      kge = km
      kg  = kge-kgs+1

!-----------------------------------------------------------------------
!     start definitions
!-----------------------------------------------------------------------
      call redef (iou)

!-----------------------------------------------------------------------
!     define dimensions
!-----------------------------------------------------------------------
      call defdim ('time', iou, 0, id_time)
      call defdim ('xt', iou, ig, id_xt)
      call defdim ('yt', iou, jg, id_yt)
      call defdim ('zt', iou, kg, id_zt)
      call defdim ('xu', iou, ig, id_xu)
      call defdim ('yu', iou, jg, id_yu)
      call defdim ('zw', iou, kg, id_zw)
      call defdim ('xt_edges', iou, ig+1, id_xt_e)
      call defdim ('yt_edges', iou, jg+1, id_yt_e)
      call defdim ('zt_edges', iou, kg+1, id_zt_e)
      call defdim ('xu_edges', iou, ig+1, id_xu_e)
      call defdim ('yu_edges', iou, jg+1, id_yu_e)
      call defdim ('zw_edges', iou, kg+1, id_zw_e)

!-----------------------------------------------------------------------
!     define 1d data (t)
!-----------------------------------------------------------------------
      call defvar ('itt', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, ' ', ' ',' ')
      call defvar ('irstdy', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, ' ', ' ',' ')
      call defvar ('msrsdy', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, ' ', ' ',' ')

      call defvar ('year', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, 'year', ' ',' ')
      call defvar ('month', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, 'month', ' ',' ')
      call defvar ('day', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, 'day', ' ',' ')
      call defvar ('hour', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, 'hour', ' ',' ')
      call defvar ('minute', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, 'minute', ' ',' ')
      call defvar ('second', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &, 'second', ' ',' ')

!-----------------------------------------------------------------------
!     define 1d data (x, y or z)
!-----------------------------------------------------------------------
      call defvar ('xt', iou, 1, (/id_xt/), 0., 0., 'X', 'D'
     &, 'longitude of the t grid', 'grid_longitude', 'degrees_east')
      call defvar ('yt', iou, 1, (/id_yt/), 0., 0., 'Y', 'D'
     &, 'latitude of the t grid', 'grid_latitude', 'degrees_north')
      call defvar ('zt', iou, 1, (/id_zt/), 0., 0., 'Z', 'F'
     &, 'depth of the t grid', 'depth', 'm')
      call defvar ('xu', iou, 1, (/id_xu/), 0., 0., 'X', 'D'
     &, 'longitude of the u grid', 'grid_longitude', 'degrees_east')
      call defvar ('yu', iou, 1, (/id_yu/), 0., 0., 'Y', 'D'
     &, 'latitude of the u grid', 'grid_latitude', 'degrees_north')
      call defvar ('zw', iou, 1, (/id_zw/), 0., 0., 'Z', 'F'
     &, 'depth of the w grid', 'depth', 'm')
      call defvar ('xt_edges', iou, 1, (/id_xt_e/), 0., 0., 'X', 'D'
     &, 'longitude of t grid edges','grid_edge_longitude'
     &, 'degrees_east')
      call defvar ('yt_edges', iou, 1, (/id_yt_e/), 0., 0., 'Y', 'D'
     &, 'latitude of t grid edges', 'grid_edge_latitude'
     &, 'degrees_north')
      call defvar ('zt_edges', iou, 1, (/id_zt_e/), 0., 0., 'Z', 'F'
     &, 'depth of the t grid edges', 'edge_depth'
     &, 'degrees_north')
      call defvar ('xu_edges', iou, 1, (/id_xu_e/), 0., 0., 'X', 'D'
     &, 'longitude of u grid edges', 'grid_edge_longitude'
     &, 'degrees_east')
      call defvar ('yu_edges', iou, 1, (/id_yu_e/), 0., 0., 'Y', 'D'
     &, 'latitude of u grid edges', 'grid_edge_latitude'
     &, 'degrees_north')
      call defvar ('zw_edges', iou, 1, (/id_zw_e/), 0., 0., 'Z', 'F'
     &, 'depth of w grid edges', 'edge_depth'
     &, 'degrees_north')

!-----------------------------------------------------------------------
!       define 2d data (x,y)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      call defvar ('kmt', iou, 2, it, 0., 1000., ' ', 'I'
     &,  'ocean grid depth level', ' ' ,'level')

!-----------------------------------------------------------------------
!     define 3d data (x,y,t)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      it(3) = id_time
      iu(3) = id_time

      call defvar ('psi1', iou, 3, it,  -1.e18, 1.e18, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('psi2', iou, 3, it,  -1.e18, 1.e18, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('ptd1', iou, 3, it,  -1.e18, 1.e18, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('ptd2', iou, 3, it,  -1.e18, 1.e18, ' ', 'D'
     &, ' ', ' ', ' ')

!-----------------------------------------------------------------------
!     define 4d data (x,y,z,t)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      it(3) = id_zt
      iu(3) = id_zt
      it(4) = id_time
      iu(4) = id_time
      do n=1,nt
        if (n .lt. 1000) write(a3,'(i3)') n
        if (n .lt. 100) write(a3,'(i2)') n
        if (n .lt. 10) write(a3,'(i1)') n
        call defvar ('tracer1_'//trim(a3), iou , 4, it, -1.e6, 1.e6, ' '
     &,   'D', ' ', ' ', ' ')
        call defvar ('tracer2_'//trim(a3), iou , 4, it, -1.e6, 1.e6, ' '
     &,   'D', ' ', ' ', ' ')
      enddo
      call defvar ('u1', iou , 4, iu, -1.e6, 1.e6, ' '
     &, 'D', ' ', ' ', ' ')
      call defvar ('u2', iou , 4, iu, -1.e6, 1.e6, ' '
     &, 'D', ' ', ' ', ' ')
      call defvar ('v1', iou , 4, iu, -1.e6, 1.e6, ' '
     &, 'D', ' ', ' ', ' ')
      call defvar ('v2', iou , 4, iu, -1.e6, 1.e6, ' '
     &, 'D', ' ', ' ', ' ')

!-----------------------------------------------------------------------
!     end definitions
!-----------------------------------------------------------------------
      call enddef (iou)

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global write domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)
      kls = max(1,kgs)
      kle = min(km,kge)

!-----------------------------------------------------------------------
!     write 1d data (t)
!-----------------------------------------------------------------------
      tmp(1) = itt
      call putvara ('itt', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = iday(imodeltime)
      call putvara ('irstdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = msday(imodeltime)
      call putvara ('msrsdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)

      call rdstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      tmp(1) = nyear
      call putvara ('year', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nmonth
      call putvara ('month', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nday
      call putvara ('day', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nhour
      call putvara ('hour', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nmin
      call putvara ('minute', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nsec
      call putvara ('second', iou, 1, (/1/), (/1/), tmp, 1., 0.)

!-----------------------------------------------------------------------
!     write 1d data (x or y)
!-----------------------------------------------------------------------
      call putvara ('xt', iou, ig, (/1/), (/ig/), xt(igs:ige), 1., 0.)
      call putvara ('yt', iou, jg, (/1/), (/jg/), yt(jgs:jge), 1., 0.)
      call putvara ('zt', iou, kg, (/1/), (/kg/), zt(kgs:kge), 0.01
     &, 0.)
      call putvara ('xu', iou, ig, (/1/), (/ig/), xu(igs:ige), 1., 0.)
      call putvara ('yu', iou, jg, (/1/), (/jg/), yu(jgs:jge), 1., 0.)
      call putvara ('zw', iou, kg, (/1/), (/kg/), zw(kgs:kge), 0.01
     &, 0.)
      call edge_maker (1, xt_e, xt, dxt, xu, dxu, imt)
      call putvara ('xt_edges', iou, ig+1, (/1/), (/ig+1/)
     &, xt_e(igs:ige+1), 1., 0.)
      call edge_maker (1, yt_e, yt, dyt, yu, dyu, jmt)
      call putvara ('yt_edges', iou, jg+1, (/1/), (/jg+1/)
     &, yt_e(jgs:jge+1), 1., 0.)
      call edge_maker (1, zt_e, zt, dzt, zw, dzw, km)
      call putvara ('zt_edges', iou, kg+1, (/1/), (/kg+1/)
     &, zt_e(kgs:kge+1), 0.01, 0.)
      call edge_maker (2, xu_e, xt, dxt, xu, dxu, imt)
      call putvara ('xu_edges', iou, ig+1, (/1/), (/ig+1/)
     &, xu_e(igs:ige+1), 1., 0.)
      call edge_maker (2, yu_e, yt, dyt, yu, dyu, jmt)
      call putvara ('yu_edges', iou, jg+1, (/1/), (/jg+1/)
     &, yu_e(jgs:jge+1), 1., 0.)
      call edge_maker (2, zw_e, zt, dzt, zw, dzw, km)
      call putvara ('zw_edges', iou, kg+1, (/1/), (/kg+1/)
     &, zw_e(kgs:kge+1), 0.01, 0.)

!-----------------------------------------------------------------------
!     write 2d data (x,y)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      ln = ic(1)*ic(2)
      rkmt(ils:ile,jls:jle) = kmt(ils:ile,jls:jle)
      call putvara ('kmt', iou, ln, is, ic, rkmt(ils:ile,jls:jle)
     &, 1., 0.)

!-----------------------------------------------------------------------
!     write 3d data (x,y,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      is(3) = 1
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)

!     first do psi at "tau" then at "tau+1"
      call putvara ('psi1', iou, ln, is, ic, psi(ils:ile,jls:jle,1)
     &, 1., 0.)
      call putvara ('psi2', iou, ln, is, ic, psi(ils:ile,jls:jle,2)
     &, 1., 0.)
!     guess fields
      call oget (kflds, nwds, nkflds-1, ptd)
      call putvara ('ptd1', iou, ln, is, ic, ptd(ils:ile,jls:jle)
     &, 1., 0.)
      call oget (kflds, nwds, nkflds, ptd)
      call putvara ('ptd2', iou, ln, is, ic, ptd(ils:ile,jls:jle)
     &, 1., 0.)

!-----------------------------------------------------------------------
!     write 4d data (x,y,z,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = 1
      is(3) = 1
      ic(3) = kle-kls+1
      is(4) = 1
      ic(4) = 1
      ln = ic(1)*ic(2)*ic(3)*ic(4)

!     save the "tau" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
!         remove external mode from "tau". since psi has been updated
!         psi(,,2) is at "tau"
          if (jrow .lt. jmt) then

            do i=2,imt-1
              diag1 = psi(i+1,jrow+1,2) - psi(i  ,jrow,2)
              diag0 = psi(i  ,jrow+1,2) - psi(i+1,jrow,2)
              ext(i,1) = -(diag1+diag0)*dyu2r(jrow)*hr(i,jrow)
              ext(i,2) =  (diag1-diag0)*dxu2r(i)*hr(i,jrow)*csur(jrow)
             enddo
            do k=1,km
              do i=2,imt-1
                if (k .le. kmu(i,jrow)) then
                  bufsl(i,k,1) = (u(i,k,j,1,tau) - ext(i,1))
                  bufsl(i,k,2) = (u(i,k,j,2,tau) - ext(i,2))
                else
                  bufsl(i,k,1) = c0
                  bufsl(i,k,2) = c0
                endif
              enddo
            enddo

            call setbcx (bufsl(1,1,1), imt, km)
            call setbcx (bufsl(1,1,2), imt, km)
          else
            do k=1,km
              do i=1,imt
                bufsl(i,k,1) = c0
                bufsl(i,k,2) = c0
              enddo
            enddo
          endif
        else
          j = jmw
          call getrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
          do k=1,km
            do i=1,imt
              bufsl(i,k,1) = u(i,k,j,1,tau)
              bufsl(i,k,2) = u(i,k,j,2,tau)
            enddo
          enddo
        endif

        is(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          call putvara('tracer1_'//trim(a3), iou, ln, is, ic
     &,     t(ils:ile,kls:kle,j,n,tau), 1., 0.)
        enddo
        call putvara('u1', iou, ln, is, ic, bufsl(ils:ile,kls:kle,1)
     &,   1., 0.)
        call putvara('v1', iou, ln, is, ic, bufsl(ils:ile,kls:kle,2)
     &,   1., 0.)
      enddo

!     save the "tau+1" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
          call getrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif
        is(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          call putvara('tracer2_'//trim(a3), iou, ln, is, ic
     &,     t(ils:ile,kls:kle,j,n,taup1), 1., 0.)
        enddo
        call putvara('u2', iou, ln, is, ic, u(ils:ile,kls:kle,j,1,taup1)
     &,   1., 0.)
        call putvara('v2', iou, ln, is, ic, u(ils:ile,kls:kle,j,2,taup1)
     &,   1., 0.)
      enddo

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)

      print*, '=> Ocn restart written to ',trim(fname),' on ', stamp

      return
      end
