! source file: /den/eby/UVic_ESCM/2.6/source/embm/rivmodel.F
      subroutine rivmodel

!=======================================================================
!     river model for energy-moisture balance model
!     calculates river runoff from precipitation over land
!     based on code by e.wiebe

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include "param.h"
      include "atm.h"
      include "riv.h"
      include "cembm.h"
      include "grdvar.h"
      include "switch.h"

      include "ice.h"

      integer i, j, n, m

!-----------------------------------------------------------------------
!     sum (area integral) the precip over each basin
!-----------------------------------------------------------------------

      do n=1,nb
        psum(n) = 0.0
      enddo

      do m=1,nbp
        i = ibp(m,1)
        j = ibp(m,2)
        n = ibp(m,3)

        psum(n) = (precip(i,j) - psno(i,j))*dxt(i)*cst(j)*dyt(j)
     &          + psum(n)

      enddo

!-----------------------------------------------------------------------
!     calculate discharge for each discharge point in each basin
!     and add it to the fresh water flux
!-----------------------------------------------------------------------

      do m=1,ndp
        i = idp(m,1)
        j = idp(m,2)
        n = idp(m,3)
        if (addflux) flux(i,j,2) = flux(i,j,2) + dts*psum(n)*wdar(m)
      enddo

      return
      end

      subroutine rivinit

!=======================================================================
!     initializes river model for the energy-moisture balance model
!     based on code by e.wiebe

!     modified by:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include "param.h"
      include "riv.h"
      include "grdvar.h"
      include "levind.h"
      include "emode.h"

      character(1) :: label, line(imt), ocnchar
      character(15) :: rname

      integer i, ichk, im, ind(128,2), ior, ios, ip, j, jm, jp
      integer k, l, m, n, ndpb(maxnb), npdp, ni, np, nr

      real tba(maxnb), tda(maxnb), wt, wtdpb(maxnb), ztd

      logical        errorc

      write (stdout,*)
      write (stdout,*) '========== initializing river model =========='
      write (stdout,*)
      errorc = .false.
!     set "ocean character" (used to look for mapping errors)
      ocnchar = '#'

!-----------------------------------------------------------------------
!     zero ind array. ind(m,1) will hold the converted ASCII map labels
!     and ind(m,2) the equivalent basin number
!-----------------------------------------------------------------------

      do m=1,128
        ind(m,1) = 0
        ind(m,2) = 0
      enddo

!-----------------------------------------------------------------------
!     read in the river basin map. fill index array with indices
!     converted from ASCII map characters
!-----------------------------------------------------------------------

      call getunit (ior, 'rivers.map', 'f s r ieee')
      write (*,'(a,a)')
     &  ' ==> Reading river data from file ','rivers.map'
      nbp = 0

      read (ior,*)
      do j=jmtm1,2,-1
        read (ior,'(1000a)') (line(i),i=1,imt)
        do i=2,imtm1
          if (kmt(i,j) .eq. 0) then
            m = ichar(line(i)) + 1
            ind(m,1) = m
            if (line(i) .eq. ocnchar) then
              write (stdout,*) '==> Warning: land is labeled as ocean'
              write (stdout,*) '    at i,j ', i,j
            endif
            nbp = nbp + 1
            if (nbp .le. maxnbp) then
              ibp(nbp,1) = i
              ibp(nbp,2) = j
              ibp(nbp,3) = m
            else
              errorc = .true.
            endif
          else
            if (line(i) .ne. ocnchar) then
              write (stdout,*) '==> Warning: ocean is labeled as land'
              write (stdout,*) '    at i,j ', i,j
            endif
          endif
        enddo
      enddo
      read (ior,*)

!-----------------------------------------------------------------------
!     find the number of basins by looking at the valid basin indices
!     there are a possible 128 characters in the ASCII character set
!-----------------------------------------------------------------------

      nb = 0
      do m=1,128
        if (ind(m,1) .gt. 0) nb = nb + 1
      enddo
      write (stdout,*) '  ',nb,' river basins (nb)'
      write (stdout,*) '  ',nbp,' basin points (nbp)'
      if (nb .gt. maxnb) errorc = .true.

!-----------------------------------------------------------------------
!     read the river label, name and discharge points
!     n is a basin number, m is a basin label and l is a river number
!     (l should be the same as n unless extra rivers are defined)
!-----------------------------------------------------------------------

      n = 0
      ndp = 0
      ios = 0
      do while (ios .eq. 0)
        read (ior,'(a1,i3,F7.3,15a)',iostat=ios) label, np, wt, rname
        if (ios .eq. 0) then
          m = ichar(label) + 1
          if (ind(m,1) .eq. m .and. ind(m,2) .eq. 0) then
!           this is a valid basin (on map and not already defined)
            n = n + 1
            ind(m,2) = n
            ndp = ndp + np
            if (n .le. maxnb) then
              ndpb(n) = np
              wtdpb(n) = wt
              rivname(n) = rname
            else
              errorc = .true.
            endif
            do k=ndp-np+1,ndp
              read (ior,'(2i4)') i, j
              if (k .le. maxndp) then
                idp(k,1) = i
                idp(k,2) = j
                idp(k,3) = m
                wdar(k) = wt/np
              else
                errorc = .true.
              endif
              if (kmt(i,j) .eq. 0) then
                write (stdout,*) '==> Error: discharge point is on land'
                write (stdout,*) '    at i,j ', i,j
                errorc = .true.
              endif
            enddo
          else
!           this is not a valid basin (not on map or already defined)
            write (stdout,*) '==> Warning: extra river definition'
            write (stdout,*) '    ignoring river labeled ', label
            do k=1,np
              read (ior,'(2i4)') i, j
            enddo
          endif
        endif
      enddo

      call relunit (ior)

!-----------------------------------------------------------------------
!     check each basin has a discharge point (use perimeter if not)
!-----------------------------------------------------------------------

      do m=1,128
        np = 0
        npdp = ndp
        if (ind(m,2) .le. maxnb .and. ind(m,2) .ge. 1)
     &    np = ndpb(ind(m,2))
        if (ind(m,1) .ne. 0 .and. wtdpb(ind(m,2)) .lt. 1.0) then
          do np=1,min(nbp,maxnbp)
            if (ibp(np,3) .eq. ind(m,1)) then
              i = ibp(np,1)
              j = ibp(np,2)
              ni = map(i,j)
            endif
          enddo
          if (ind(m,2) .eq. 0) then
            n = n + 1
            ind(m,2) = n
          else
            n = ind(m,2)
          endif
          np = 0
          if (n .le. maxnb) then
            ndpb(n) = 0
            if (ind(m,2) .eq. 0) rivname(n) = '               '
          endif
          do k=iofs(ni)+1,iofs(ni)+nippts(ni)
            do np=1,min(nbp,maxnbp)
              if (ibp(np,3) .eq. ind(m,1)) then
                i = ibp(np,1)
                ip = min(imt,i+1)
                im = max(1,i-1)
                j = ibp(np,2)
                jp = min(jmt,j+1)
                jm = max(1,j-1)
                if ((ip .eq. iperm(k) .and. j .eq. jperm(k)) .or.
     &              (im .eq. iperm(k) .and. j .eq. jperm(k)) .or.
     &              (i .eq. iperm(k) .and. jp .eq. jperm(k)) .or.
     &              (i .eq. iperm(k) .and. jm .eq. jperm(k))) then
                  ichk = 0
                  do l=1,ndp
                    if (iperm(k) .eq. idp(l,1) .and.
     &                  jperm(k) .eq. idp(l,2)) ichk = 1
                    if (iperm(k) .eq. 1 .or.
     &                  iperm(k) .eq. imt) ichk = 1
                  enddo
                  if (ichk .eq. 0) then
                    ndp = ndp + 1
                    if (n .le. maxnb) ndpb(n) = ndpb(n) + 1
                    if (ndp .le. maxndp) then
                      idp(ndp,1) = iperm(k)
                      idp(ndp,2) = jperm(k)
                      idp(ndp,3) = ind(m,1)
                    else
                      errorc = .true.
                    endif
                  endif
                endif
              endif
            enddo
          enddo
          np = ndp - npdp
          if (np .gt. 0) then
            do k=npdp+1,ndp
              wdar(k) = (1.0 - wtdpb(n))/np
            enddo
          endif
        endif
      enddo
      write (stdout,*) '  ',ndp,' discharge points (ndp)'

!-----------------------------------------------------------------------
!     check discharge points are next to land
!-----------------------------------------------------------------------

      do m=1,ndp
        ichk = 0
        do ni=1,nisle
          do k=iofs(ni)+1,iofs(ni)+nippts(ni)
            if (iperm(k) .eq. idp(m,1) .and.
     &          jperm(k) .eq. idp(m,2)) ichk = 1
          enddo
        enddo
        if (ichk .eq. 0) write (stdout,*)
     &    '==> Warning: Discharge point not next to land:',
     &    idp(m,1),idp(m,2)
      enddo

!-----------------------------------------------------------------------
!     check for array bounds errors
!-----------------------------------------------------------------------

      if (errorc) then
        if (nb .gt. maxnb) then
          write (stdout,*) '==> Error: "nb" > "maxnb" in riv.h'
          write (stdout,*) '    nb = ',nb, ' and maxnb = ',maxnb
        endif
        if (nbp .gt. maxnbp) then
          write (stdout,*) '==> Error: "nbp" > "maxnbp" in riv.h'
          write (stdout,*) '    nbp = ',nbp, ' and maxnbp = ',maxnbp
        endif
        if (ndp .gt. maxndp) then
          write (stdout,*) '==> Error: "ndp" > "maxndp" in riv.h'
          write (stdout,*) '    ndp = ',ndp, ' and maxndp = ',maxndp
        endif
        stop '=>rivinit2'
      endif

!-----------------------------------------------------------------------
!     replace ASCII basin labels with basin numbers (indices)
!-----------------------------------------------------------------------

      do m=1,nbp
        ibp(m,3) = ind(ibp(m,3),2)
      enddo
      do m=1,ndp
        idp(m,3) = ind(idp(m,3),2)
      enddo

!-----------------------------------------------------------------------
!     calculate the total basin and discharge areas
!-----------------------------------------------------------------------

      do n=1,nb
        tda(n) = 0.0
        tba(n) = 0.0
      enddo
      do m=1,nbp
        i = ibp(m,1)
        j = ibp(m,2)
        n = ibp(m,3)
        tba(n) = tba(n) + dxt(i)*cst(j)*dyt(j)
      enddo
      do m=1,ndp
        i = idp(m,1)
        j = idp(m,2)
        n = idp(m,3)
        tda(n) = tda(n) + dxt(i)*cst(j)*dyt(j)
      enddo

!-----------------------------------------------------------------------
!     ensure discharge weights are normalized and factor in inverse area
!-----------------------------------------------------------------------

      do m=1,128
        if (ind(m,1) .ne. 0.0) then
          n = ind(m,2)
          wt = 0.0
          do k=1,ndp
            if (n .eq. idp(k,3)) wt = wt + wdar(k)
          enddo
          if (wt .gt. 0.0) then
            do k=1,ndp
              i = idp(k,1)
              j = idp(k,2)
              if (n .eq. idp(k,3))
     &          wdar(k) = wdar(k)/(wt*dxt(i)*cst(j)*dyt(j))
            enddo
          else
            write (stdout,*) '==> Error: Zero basin discharge weight: '
     &,       char(ind(m,1)-1)
            stop '=>rivinit3'
          endif
        endif
      enddo

!-----------------------------------------------------------------------
!     write summary
!-----------------------------------------------------------------------

      write (stdout,*)
      write (stdout,'(a10,a25,a25,a11)') 'summary:  ', 'basin name    '
     &, 'area (km**2)', 'ndp'
      write (stdout,'(1x,a8,a8,a32,a15)') 'number', 'symbol', 'basin'
     &, 'discharge'
      do m=1,128
        n = ind(m,2)
        label = char(ind(m,1)-1)
        if (ind(m,1) .ne. 0.0)
     &    write (stdout,'(i8,a8,a19,f15.1,f15.1,i6)') n, label
     &,     rivname(n), tba(n)/1.0e10, tda(n)/1.0e10, ndpb(n)
      enddo

      if (maxnb .gt. nb) write (stdout,*)
     &  'maxnb = ',maxnb,' but minimum memory used if maxnb = ',nb
      if (maxnbp .gt. nbp) write (stdout,*)
     &  'maxnbp = ',maxnbp,' but minimum memory used if maxnbp = ',nbp
      if (maxndp .gt. ndp) write (stdout,*)
     &  'maxndp = ',maxndp,' but minimum memory used if maxndp = ',ndp
      write (stdout,*)
      write (stdout,*) '======= end river model initialization ======='
      write (stdout,*)

      return
      end
