! source file: /usr/local/models/UVic_ESCM/2.6/source/common/UVic_ESCM.F
      program UVic_ESCM

!=======================================================================

!     UNIVERSITY OF VICTORIA EARTH SYSTEM CLIMATE MODEL

!     A climate model developed by researchers in the Climate Research
!     Group, in the School of Earth and Ocean Sciences, located at the
!     University of Victoria, Victoria, B.C., Canada.

!     Bug reports or questions:

!     Each section of code has a code author. Please direct questions
!     or problems to these authors

!     Many people have contributed to the development of this code
!     and attempts are made to indicate originators of code where
!     appropriate. Authorship here is meant to convey responsibility
!     for the current form of the code and does not necessarily
!     indicate that the author is responsible for all (or even
!     a major portion) of the code content.

!     Requirements:

!     Standard fortran 90 is used

!     Disclaimer:

!     The UVic Earth System Climate Model (UVic_ESCM) is a climate
!     modeling research tool developed at the University of Victoria.
!     Others may use it freely but we assume no responsibility for
!     problems or incorrect use. It is left to the user to ensure that
!     a particular configuration is working correctly.

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

!     This is the main driver. Integration time is divided into a number
!     of equal time segments and SBC are held fixed for each time
!     segment. When coupling, SBC are supplied each time segment (the
!     coupling period) and held fixed for that period.

!     based on code by: r. c. pacanowski and a. rosati
!=======================================================================

      include "param.h"

      include "atm.h"
      include "cembm.h"

      include "coord.h"
      include "csbc.h"
      include "iounit.h"
      include "levind.h"
      include "mapsbc.h"

      include "mw.h"
      include "task_on.h"

      include "scalar.h"
      include "switch.h"
      include "tmngr.h"

      print*, '== UNIVERSITY OF VICTORIA EARTH SYSTEM CLIMATE MODEL =='
      print*, '                                            '

!-----------------------------------------------------------------------
!     initialize i/o units
!-----------------------------------------------------------------------

      call ioinit

!-----------------------------------------------------------------------
!     setup file renaming
!-----------------------------------------------------------------------

      call file_names

!-----------------------------------------------------------------------
!     Initialize S.B.C. names and related items.
!-----------------------------------------------------------------------

      do n=1,numsbc
        sbcocn(:,:,n) = 0.0
        mapsbc(n) = n
      enddo

      m = 0
      m = m + 1
      itaux = m
      m = m + 1
      itauy = m
      m = m + 1
      ihflx = m
      m = m + 1
      ipme = m

      m = m + 1
      iws = m

      iwa = 0

      m = m + 1
      iwx = m
      m = m + 1
      iwy = m

      ipsw = 0

      m = m + 1
      isst = m
      m = m + 1
      isss = m

      m = m + 1
      isu = m
      m = m + 1
      isv = m
      m = m + 1
      igu = m
      m = m + 1
      igv = m

      if ( m .gt. numsbc) then
        print*, '==> Error: increase numsbc in csbc.h to ', m
        stop '=>driver'
      endif

!-----------------------------------------------------------------------
!     do the introductory ocean setup once per run
!-----------------------------------------------------------------------

      call setocn

!-----------------------------------------------------------------------
!     do the introductory atmosphere setup once per run
!-----------------------------------------------------------------------

      write (stdout,'(/a36/)') ' ==> Note: the atmos setup follows:'

!     "setatm" must do the following:
!       1) set up the atmospheric S.B.C. grid definition
!       2) define the atmosphere land/sea mask
!       3) set the atmosphere time step "dtatm" {seconds}

      call setembm

!-----------------------------------------------------------------------
!     compute the number of ocean time steps "numots" for this run and
!     the number of ocean time steps per ocean segment "ntspos".
!     compute the number of atmos time steps "numats" for this run and
!     the number of atmos time steps per atmos segment "ntspas".
!     divide the integration time "days" into "numseg" segments.
!     each will be length "segtim" days. Surface boundary conditions
!     are supplied every "segtim" days.
!-----------------------------------------------------------------------

      numots = nint(rundays/(dtts*secday))
      ntspos = nint(segtim/(dtts*secday))
      numats = nint(rundays/(dtatm*secday))
      ntspas = nint(segtim/(dtatm*secday))
      numseg = numots/ntspos

      write (stdout,8800) rundays, numseg, segtim, ntspos, ntspas, dtts
     &,                   dtatm

!-----------------------------------------------------------------------
!     check for consistency in the S.B.C. setup
!-----------------------------------------------------------------------

      call chkcpl (dtatm, dtts)

!-----------------------------------------------------------------------
!     S T A R T    S E G M E N T    L O O P
!-----------------------------------------------------------------------

      do n=1,numseg

!-----------------------------------------------------------------------
!       get the atmospheric S.B.C. from ocean
!-----------------------------------------------------------------------

        call gasbc

!-----------------------------------------------------------------------
!       call the atmospheric model once for each time step until one
!       segment of "segtim" days is complete. hold atmos S.B.C. fixed
!       during each segment and predict average S.B.C. for ocean
!-----------------------------------------------------------------------

        do loop=1,ntspas

          call embm

        enddo

!-----------------------------------------------------------------------
!       get ocean S.B.C.s from the atmosphere
!-----------------------------------------------------------------------

        call gosbc

!-----------------------------------------------------------------------
!       call the ocean model once for each time step until one
!       segment of "segtim" days is complete. hold ocean S.B.C. fixed
!       during each segment and predict average S.B.C. for atmos
!-----------------------------------------------------------------------

        do loop=1,ntspos
          call mom

          call embmout (1, imt, 1, jmt)

        enddo

      enddo

!-----------------------------------------------------------------------
!     E N D    S E G M E N T    L O O P
!-----------------------------------------------------------------------

      write (stdout,9300)

!     show files and close all units

      call showfiles
      call release_all

8800  format (/,/,1x,'The model has been configured to run for'
     &,g14.7,' days in ',i10,' segments of ',g14.7,' days each.'
     &,/1x,'There will be ',i6,' ocean time steps per segment'
     &,' and ',i6,' atmosphere time steps per segment.'/
     &,/1x,' The ocean "dtts" =', g14.7, ' seconds'
     &,', and the atmosphere "dtatm" =', g14.7,' seconds'
     &,/,/)

9300  format (/,10x,' ==>  M.O.M. integration is complete.')
      stop
      end

      subroutine chkcpl (dtatm, dtts)
      logical errorc
      include "param.h"
      include "csbc.h"
      include "switch.h"

!-----------------------------------------------------------------------
!     do consistency checks before allowing model to continue
!-----------------------------------------------------------------------

      errorc = .false.
      write (stdout,*) ' '
      write (stdout,*) '    (checking S.B.C. setup)'

      if (dtatm .eq. c0) then
          write (stdout,9000)
     & '==> Error: the atmospheric time step must be set in "setatm"  '
          errorc = .true.
          dtatm = 1.e-6
      endif
!      critv = 1.e-6
      critv = 1.e-4
      if (segtim .ne. c0) then
        r1 = rundays/segtim
      else
        r1 = 0.5
      endif
      r2 = segtim/(dtts*secday)
      r3 = segtim/(dtatm*secday)
      if (segtim .eq. c0) then
          write (stdout,9000)
     & '==> Error: coupling period "segtim" must be specified         '
          errorc = .true.
      elseif (abs(r1-nint(r1)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of segments       '
     &,'    "segtim"  within "rundays" (the length of the run)        '
          errorc = .true.
      elseif (abs(r2-nint(r2)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of density time   '
     &,'    steps "dtts"  within "segtim" (the segment time)          '
          errorc = .true.
      elseif (abs(r3-nint(r3)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of atmos time     '
     &,'    steps "dtatm"  within "segtim" (the segment time)         '
          errorc = .true.
      endif

      write (stdout,*) '    (End of S.B.C. checks) '
      write (stdout,*) ' '
      if (errorc) stop '=>chkcpl'

9000  format (/,(1x,a80))
      return
      end

