! source file: /usr/local/models/UVic_ESCM/2.6/source/common/uvic_netcdf.F
      subroutine openfile (fname, relyr, ntrec, ncid)
!=======================================================================
!     open file for reading or writing

!     input:
!       fname = file name to be opened
!       relyr = relative year
!       ntrec = number of time record (clobber if set less than 0)
!     output:
!       ntrec = number of time record
!       ncid  = iou unit

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: fname
      character(80) :: name

      integer i, id, iv, ln, n, ncid, ntrec

      real relyr
      real(kind=8), allocatable :: time(:)

      logical exists

      name = fname
      inquire (file=trim(name), exist=exists)
      if ( ntrec .lt. 0. ) exists = .false.
      ntrec = 1
      if (.not. exists) then
        i = nf_create (trim(name), nf_clobber, ncid)
        call checkerror (i,'openfile '//trim(name))
        i = nf_enddef (ncid)
        call checkerror (i,'openfile nf_enddef')
        ntrec = 0
      else
        i = nf_open (trim(name), nf_write, ncid)
        call checkerror (i,'openfile nf_open '//trim(name))
        i = nf_inq_varid (ncid, 'time', iv)
        if (i .ne. nf_noerr) return
        i = nf_inq_vardimid (ncid, iv, id)
        if (i .ne. nf_noerr) return
        i = nf_inq_dimlen (ncid, id, ln)
        if (i .ne. nf_noerr) return
        allocate (time(ln))
        i = nf_get_var_double (ncid, iv, time)
        if (i .ne. nf_noerr) then
          deallocate (time)
          return
        endif
!       find correct the record number for this time (within 32 sec)
        ntrec = ln + 1
        do n=1,ln
           if (abs(time(n) - relyr) .lt. 1.e-6) ntrec = n
        enddo
        deallocate (time)
      endif

      return
      end

      subroutine closefile (ncid)
!=======================================================================
!     close file

!     input:
!       ncid = iou unit

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      integer i, ncid

      i = nf_close (ncid)
      call checkerror (i,'closefile nf_close')

      return
      end

      subroutine redef (ncid)
!=======================================================================
!     redifine

!     input:
!       ncid = iou unit

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      integer i, ncid

      i = nf_redef(ncid)
      call checkerror (i,'redef nf_redef')

      return
      end

      subroutine enddef (ncid)
!=======================================================================
!     end definitions

!     input:
!       ncid = iou unit

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      integer i, ncid

      i = nf_enddef (ncid)
      call checkerror (i,' enddef nf_enddef')

      return
      end

      subroutine checkerror(i, trace)
!=======================================================================
!     check for any netcdf errors

!     input:
!       i     = netcdf error index
!       trace = trace string

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: trace

      integer i

      if (i .ne. nf_noerr) then
        print*, 'netcdf error: ', nf_strerror(i)
        print*, 'trace string: ', trace
        stop
      endif

      return
      end

      function inqvardef (name, ncid)
!=======================================================================
!     check if a variable is defined

!     input:
!       name       = variable name
!       ncid       = iou unit

!     output:
!        inqvardef = (true, false) = (defined, not defined)

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name
      integer i, iv, ncid
      logical inqvardef

      inqvardef = .false.
      i = nf_inq_varid (ncid, name, iv)
      if (i .eq. nf_noerr) inqvardef = .true.

      return
      end

      subroutine putglobal (ncid, name, title, expnam, timunit)
!=======================================================================
!     put global atributes

!     input:
!       ncid    = iou unit
!       name    = file name
!       title   = file title
!       expnam  = experiment name
!       timunit = timunit
!       rlapse  = atmospheric lapse rate

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name, title, expnam, timunit

      integer i, ncid, ntrec

      real timper, rlapse

      i = nf_put_att_text (ncid, nf_global, 'file_name'
     &, len(trim(name)), trim(name))
      call checkerror(i,'putglobal file_name '//trim(name))

      i = nf_put_att_text (ncid, nf_global, 'title'
     &, len(trim(title)), trim(title))
      call checkerror (i,'putglobal title '//trim(name))

      i = nf_put_att_text (ncid, nf_global, 'convention'
     &, len('COARDS'), 'COARDS')
      call checkerror(i,'putglobal convention '//trim(name))

      i = nf_put_att_text (ncid, nf_global, 'experiment_name'
     &, len(trim(expnam)), trim(expnam))
      call checkerror (i,'putglobal experiment_name '//trim(name))

      i = nf_put_att_text (ncid, nf_global, 'time_unit'
     &, len(trim(timunit)), trim(timunit))
      call checkerror (i,'putglobal time_unit '//trim(name))

      return
      end

      subroutine getglobal (ncid, name, title, expnam, timunit)
!=======================================================================
!     get global atributes

!     input:
!       ncid   = iou unit

!     output
!       name    = file name
!       title   = file title
!       expnam  = experiment name
!       timunit = timunit
!       rlapse  = atmospheric lapse rate

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name, title, expnam, timunit

      integer i, iv, ncid

      real rlapse
      real(kind=4) tmp

      i = nf_get_att_text (ncid, nf_global, 'file_name', name)
      if (i .ne. nf_noerr) print*,'getglobal: file_name not found'

      i = nf_get_att_text (ncid, nf_global, 'title', title)
      if (i .ne. nf_noerr) print*,'getglobal: title not found'

      i = nf_get_att_text (ncid, nf_global, 'experiment_name', expnam)
      if (i .ne. nf_noerr) print*,'getglobal: experiment_name not found'

      i = nf_get_att_text (ncid, nf_global, 'time_unit', timunit)
      if (i .ne. nf_noerr) print*,'getglobal: time_unit not found'

      return
      end

      subroutine defdim (name, ncid, ln, id)
!=======================================================================
!     define dimension

!     input:
!       name = name of variable to be defined
!       ncid = iou unit
!       ln   = length of axis (0 = unlimited)
!       id   = dimension id

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name

      integer i, id, ncid, ln

      i = nf_inq_dimid (ncid, name, id)
!     if dimension is already defined, return
      if (i .eq. nf_noerr) return

      if (ln .gt. 0) then
        i = nf_def_dim (ncid, name, ln, id)
      else
        i = nf_def_dim (ncid, name, nf_unlimited, id)
      endif
      call checkerror (i, 'defdim '//trim(name))

      return
      end

      subroutine getaxis (name, ncid, ln, dout, is, ie, s, o)
!=======================================================================
!     read a netcdf axis
!     the first value of the axis to be read must be found within the
!     global axis provided or the axis will be redefined. if the axis is
!     redefined, it will be centred in the global axis and padded with
!     nf_fill_double. if the read axis is larger than the global axis or
!     is defined outside of the global axis a stop error is generated.

!     input:
!       name = name of variable to be defined
!       ncid = iou unit
!       ln   = length of axis
!       dout = global axis
!       is   = starting index in global axis
!       ie   = ending index in global array
!       s    = data scalar
!       o    = data offset

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name

      integer i, id, ie, ncid, is, iv, len, ln

      real dout(ln), o, rs, s
      real(kind=4), allocatable :: din(:)

      i = nf_inq_varid (ncid, name, iv)
      call checkerror (i,'getaxis nf_inq_varid '//name)
      i = nf_inq_vardimid (ncid, iv, id)
      call checkerror (i,'getaxis nf_inq_varndimid '//name)
      i = nf_inq_dimlen (ncid, id, len)
      call checkerror (i,'getaxis nf_inq_dimlen '//name)
      allocate (din(len))
      i = nf_get_vara_real (ncid, iv, 1, len, din)
      call checkerror(i,'getaxis nf_get_vara_real '//name)
      is = 0
      do i=ln,1,-1
        if (abs(dout(i)-din(1)) .lt. 1.e-5) is = i
      enddo
      if (is .eq. 0) then
        dout(:) = nf_fill_double
        is = 1
        if (len .lt. ln) is = is + (ln - len)/2
      endif
      if (len + is - 1 .gt. ln) then
        print*, 'error in getaxis => read axis not within global axis'
        stop
      endif
      ie = is - 1 + len
      rs = 0.0
      if (s .ne. 0.) rs = 1.0/s
      do i=1,len
        dout(i + is - 1) = (din(i) - o)*rs
      enddo
      deallocate (din)

      return
      end

      subroutine defvar (name, ncid, nd, id, rmin, rmax, axis
     &,                  type, lname, sname, units)
!=======================================================================
!     define data

!     input:
!       name  = name of variable to be defined
!       ncid  = iou unit
!       nd    = number dimensions of data
!       id    = data id
!       rmin  = minimum range (default real)
!       rmax  = maximum range (default real)
!       axis  = axis type
!       type  = data type (D=double,F=float,I=integer,Tn=char*n)
!       lname = long name
!       sname = standard name
!       units = data units

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name, axis, lname, sname, type, units

      integer i, nd, id(nd), idt(nd+1), iv, ncid, ln, rvari(2)

      real rmax, rmin
      real(kind=4) rvarf(2)
      real(kind=8) rvard(2)

      i = nf_inq_varid (ncid, name, iv)
!     if variable is already defined, return
      if (i .eq. nf_noerr) return

      if (type .eq. 'D') then
        i = nf_def_var (ncid, name, nf_double, nd, id, iv)
        call checkerror (i,'defvar double '//trim(name))
        if (rmin .ne. rmax) then
          rvard(1) = dble(rmin)
          rvard(2) = dble(rmax)
          i = nf_put_att_double(ncid,iv,'valid_range',nf_double,2,rvard)
          call checkerror(i,'defvar valid_range double '//trim(name))
        endif
        i = nf_put_att_double (ncid,iv,'fill_value',nf_double,1
     &,   nf_fill_double)
        call checkerror (i,'defvar missing_value double '//trim(name))
        i = nf_put_att_double (ncid,iv,'missing_value',nf_double,1
     &,   nf_fill_double)
        call checkerror (i,'defvar fill_value double '//trim(name))

      elseif (type .eq. 'F') then
        i = nf_def_var (ncid, name, nf_real, nd, id, iv)
        call checkerror (i,'defvar real '//name)
        if (rmin .ne. rmax) then
          rvarf(1) = real(rmin)
          rvarf(2) = real(rmax)
          i = nf_put_att_real (ncid,iv,'valid_range',nf_real,2,rvarf)
          call checkerror (i,'defvar valid_range real '//trim(name))
        endif
        i = nf_put_att_double (ncid,iv,'fill_value',nf_real,1
     &,   nf_fill_double)
        call checkerror (i,'defvar fill_value real '//trim(name))
        i = nf_put_att_double (ncid,iv,'missing_value',nf_real,1
     &,   nf_fill_double)
        call checkerror (i,'defvar missing_value real '//trim(name))

      elseif (type .eq. 'I') then
        i = nf_def_var (ncid, name, nf_int, nd, id, iv)
        call checkerror (i,'defvar integer '//trim(name))
        if (rmin .ne. rmax) then
          rvari(1) = int(rmin)
          rvari(2) = int(rmax)
          i = nf_put_att_int (ncid,iv,'valid_range',nf_int,2,rvari)
          call checkerror (i,'defvar valid_range integer '//trim(name))
        endif
        i = nf_put_att_int (ncid,iv,'fill_value',nf_int,1
     &,   nf_fill_int)
        call checkerror (i,'defvar fill_value integer '//trim(name))
        i = nf_put_att_int (ncid,iv,'missing_value',nf_int,1
     &,   nf_fill_int)
        call checkerror (i,'defvar missing_value integer '//trim(name))

      elseif (type(1:1) .eq. 'T') then
        ln = 0
        do i=2,len(type)
          ln = ln*10.0 +  ichar(type(i:i)) - 48
        enddo
        if (ln .le. 0 .or. ln .ge. 1000) ln = 80
        do i=1,nd
         idt(i+1) = id(i)
        enddo
        call defdim (type, ncid, ln, idt(1))
        i = nf_def_var (ncid, name, nf_char, 2, idt, iv)
        call checkerror (i,'defvar text '//trim(name))
      endif

      if (axis .ne. ' ') then
        i = nf_put_att_text (ncid,iv,'axis',len(axis),axis)
        call checkerror (i,'defvar axis '//trim(name))
      endif
      if (lname .ne. ' ') then
        i = nf_put_att_text (ncid,iv,'long_name',len(lname),lname)
        call checkerror (i,'defvar long_name '//trim(name))
      endif
      if (sname .ne. ' ') then
        i = nf_put_att_text (ncid,iv,'standard_name',len(sname),sname)
        call checkerror(i,'defvar standard_name '//trim(name))
      endif
      if (units .ne. ' ') then
        i = nf_put_att_text (ncid,iv,'units',len(units),units)
        call checkerror(i,'defvar units '//trim(name))
      endif

      return
      end

      subroutine putvaramsk (name, ncid, ln, is, ic, din, dm, s, o)
!=======================================================================
!     write data

!     input:
!       name = name of variable to be written
!       ncid = iou unit
!       ln   = length of data
!       is   = starting point for write in each dimension
!       ic   = count (or length) for write in each dimension
!       din  = data to be written (default real)
!       dm   = data mask (>0.5:dout=din, <0.5:dout=fv)
!       s    = data scalar
!       o    = data offset

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name

      integer i, ncid, iv, is(10), ic(10), ln, nd

      real din(ln), dm(ln), o, s
      real(kind=8) dout(ln)

      i = nf_inq_varid (ncid, name, iv)
      call checkerror (i,'putvaramsk nf_inq_varid '//name)

      do i=1,ln
        if (dm(i) .ge. 0.5) then
          dout(i) = din(i)*s + o
        else
          dout(i) = nf_fill_double
        endif
      enddo
      i = nf_inq_varid (ncid, name, iv)
      call checkerror (i,'putvaramsk nf_inq_varid '//name)
      i = nf_inq_varndims(ncid, iv, nd)
      call checkerror (i,'putvaramsk nf_inq_varndims '//name)
      i = nf_put_vara_double (ncid, iv, is(1:nd), ic(1:nd), dout)
      call checkerror(i,'putvaramsk '//name)

      return
      end

      subroutine putvara (name, ncid, ln, is, ic, din, s, o)
!=======================================================================
!     write data

!     input:
!       name = name of variable to be written
!       ncid = iou unit
!       ln   = length of data
!       is   = starting point for write in each dimension
!       ic   = count (or length) for write in each dimension
!       din  = data to be written (default real)
!       s    = data scalar
!       o    = data offset

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name

      integer i, ncid, iv, is(10), ic(10), ln, nd

      real din(ln), o, s
      real(kind=8) dout(ln)

      do i=1,ln
        dout(i) = din(i)*s + o
      enddo
      i = nf_inq_varid (ncid, name, iv)
      call checkerror (i,'putvara nf_inq_varid '//name)
      i = nf_inq_varndims(ncid, iv, nd)
      call checkerror (i,'putvara nf_inq_varndims '//name)
      i = nf_put_vara_double (ncid, iv, is(1:nd), ic(1:nd), dout)
      call checkerror(i,'putvara '//name)

      return
      end

      subroutine getvara (name, ncid, ln, is, ic, dout, s, o)
!=======================================================================
!     read real data

!     input:
!       name = name of variable to be written
!       ncid = iou unit
!       ln   = length of data
!       is   = starting point for write in each dimension
!       ic   = count (or length) for write in each dimension
!       dout = data (default real)
!       s    = data scalar
!       o    = data offset

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      include 'netcdf.inc'

      character(*) :: name

      integer i, ncid, iv, is(10), ic(10), ln, nd

      real(kind=8) din(ln)
      real dout(ln), o, rs, s

      i = nf_inq_varid (ncid, name, iv)
!     return zero for data if variable is not found
      if (i .ne. nf_noerr) then
        dout(:) = 0.
        print*, '==> Warning: netcdf variable ',trim(name), ' not found'
        return
      endif
      i = nf_inq_varndims(ncid, iv, nd)
      call checkerror (i,'getvara nf_inq_varndims '//name)
      i = nf_get_vara_double (ncid, iv, is(1:nd), ic(1:nd), din)
      call checkerror(i,'getvara '//name)
      rs = 0.0
      if (s .ne. 0.) rs = 1.0/s
      do i=1,ln
        dout(i) = (din(i)- o)*rs
      enddo

      return
      end

      subroutine edge_maker (it, edges, xt, dxt, xu, dxu, imt)
!=======================================================================
!     make edges for grid cells

!     input:
!       it    = flag for grid (t=1, u=2)
!       xt    = t grid poistion array
!       dxt   = t grid width
!       xu    = u grid poistion array
!       dxu   = u grid width
!       imt   = array size
!     output:
!       edges = edge array

!     based on GFDL MOM3 code

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      integer i, imt, it
      real edges(0:imt), xt(imt), dxt(imt), xu(imt), dxu(imt)

      if (it .eq. 1) then
!       make edges for T cells
        edges(0) = xu(1) - dxt(1)
        do i=1,imt
          edges(i) = xu(i)
        enddo
      elseif (it .eq. 2) then
!       make edges for U cells
        edges(imt) = xt(imt) + dxu(imt)
        do i=1,imt
          edges(i-1) = xt(i)
        enddo
      else
        write (*,*) 'Error:  it = ',it, ' in edge_maker'
        stop
      endif

      return
      end
