!=======================================================================
!                      UBC Basic Ice Sheet Model

!     Program bism models ice flow over a 2D vertically-integrated ice
!     sheet, using the biconjugate gradient method on a spherical grid.
!     Thermomechanical coupling is optional. When invoked, the vertical
!     dimension is added in the dynamic solution and 3D velocity fields
!     are calculated.  Thermodynamics are solved after Jenssen (1977) in
!     the style of Huybrechts (1988) and followers. Dynamics and
!     thermodynamics are solved on vertically-transformed co-ordinates
!     with linear interpolation between grids.

!     This is a stream-lined, pure ice-sheet version of the general
!     mixture model which includes ice-stream dynamics.

!     MODULE global_vars declares global arrays and variables.
!     MODULE global_param defines global model and physical constants.
!     All code is implicit none; though I usually use (a-h,o-z)
!     explicitly for double precision variables and (i-n) explicitly for
!     integers.
!     External to MAIN bism, all variables are locally defined.
!     Global variables are passed as arguments in the subroutine calls,
!     including explicit array dimensions. Global constants are passed
!     through MODULE global_param.

!     original author:     s.marshall
!     additional author:   m.yoshimori
!     additional author:   a.schmittner
!=======================================================================

#if defined uvic_embm
      SUBROUTINE cidm(e_timein,e_timeout,avg_Hice,avg_hi,avg_calv,&
                      avg_aice)
#else
      SUBROUTINE cidm(e_timein,e_timeout)
#endif

#if defined ubc_cidm
!-----------------------------------------------------------------------
!     declare constants and variables
!-----------------------------------------------------------------------

! ... global
      USE global_vars
      USE subgrid_arrays
# if !defined uvic_embm
      USE deg_day_vars
# endif
      USE hydrol_arrays
      USE geol_arrays
      USE icetd_arrays
      USE bedtd_arrays
      USE shelf_arrays
# if defined uvic_embm
      USE cidm_mod_embm
# endif
      IMPLICIT NONE

# if defined uvic_embm
! ... subroutine arguments
      REAL(KIND=dp),INTENT(IN) :: e_timein,e_timeout
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:iimt,1:jjmt) :: avg_Hice,&
                                            avg_hi,avg_calv,avg_aice

! ... local
      REAL(KIND=dp),DIMENSION(1:nlon,1:nlat) :: fwater,sum_Hice,sum_hi,&
                                                   bal,sum_calv,sum_aice
      REAL(KIND=dp),DIMENSION(1) :: vol1, vol2, bal2, calv2
# endif
      INTEGER :: i,j,ndcur,iprint,isloop,ieloop,icloop,nloop
      LOGICAL :: ltdcur,ldiverge
      REAL(KIND=dp) :: res

!-----------------------------------------------------------------------
!     set integration time and number of loops
!-----------------------------------------------------------------------

      stime = e_timein
      etime = e_timeout
      isloop = INT(stime/delt)
      ieloop = INT(etime/delt)
      nloop = ieloop-isloop

# if defined uvic_embm
!-----------------------------------------------------------------------
!     initialize SBC arrays for other components of the climate system
!     model
!-----------------------------------------------------------------------

      sum_Hice = c0
      sum_hi   = c0
      sum_calv = c0

# endif
      vol1 = c0 ! global ice volume before update
      DO j=1,nlat
        DO i=2,nlon-1
          vol1 = vol1 + Hice(i,j)*area(j)
        END DO
      END DO
!-----------------------------------------------------------------------
!     S T A R T   CIDM   M A I N   L O O P
!-----------------------------------------------------------------------

      ctime = stime
      DO icloop = isloop+1,ieloop
        ctime = ctime + delt

!-----------------------------------------------------------------------
!       isostatic adjustment hg(t)
!-----------------------------------------------------------------------

        IF (irebound == 1) THEN
          CALL cidm_isos(hg0,hg,hi,Hice,Hicekt,dhgdt,hgk,hgkt,hik,hikt,&
                    load,vert,loadscale,alinear,aquad,bquad,curvature,&
                    time_maxwell,hg_maxwell,iglacial,ifloating,iland)
        END IF

# if defined uvic_embm
!-----------------------------------------------------------------------
!       surface mass balance, balance (m yr^{-1} i.e.)
!       (accumuration by snowfall and ablation by surface melting)
!-----------------------------------------------------------------------

        CALL cidm_mass_acc
        CALL cidm_mass_smelt
        CALL cidm_mass_balance
        bal = balance
# else
!-----------------------------------------------------------------------
!       calculate glacial index, based on GRIP del18 record
!-----------------------------------------------------------------------

        IF ((iclim >= 6).and.(iclim <= 10)) THEN
          IF ((MOD(time,100)==0).or.(time==(stime+idelt))) THEN
            CALL glacindex(glacial,tsea,precip,hi,hi0,hg,iglacial,&
                           delsea,sealevel,time)
          END IF
        END IF

!-----------------------------------------------------------------------
!       monthly albedo
!-----------------------------------------------------------------------

        IF (ialbedo == 1) THEN                ! SM albedo code
          CALL surfalb(albedo,bckgrnd,soot,snowfrac,snowdep,snowage,&
                        snowrad,coszrs,hi)
        END IF

        ! Calculate current climate variables
        ! Surface temperature
        CALL airtemp(tsea,tseaC,hi,hg,hi0,tair,Hice,delO,time)

        ! Surface mass balance
        CALL massbal(acc,smelt,balance,Hice,hi,hg,slope,slope0,tair,&
                     tsea,tseaC,precip,rainfall,pddnet,annamp,delO,&
                     delsea,hi0,hg0,sealevel,iland,iglacial,icelevel,&
                     ideep,ifloating,igreen,landfrac,time)

        ! Treat mass loss from calving if enabled
        IF (ishelf >= 2) THEN
          CALL calving(calve,Hocean,Hice,Tbar,hg,landfrac,sealevel,iland)
          balance = balance - calve                ! Adjust mass balance
        END IF

        ! Write initial mass balance data
        IF (time==(stime+idelt)) THEN
          CALL writmass(balance,acc,smelt,bmelt,pddnet,annamp,tsea,tair,&
                        precip,time)
        END IF
# endif

!-----------------------------------------------------------------------
!       flag ltdcur tracks time for asynchronous T-D runs
!-----------------------------------------------------------------------

        IF ((itherm == 1).and.(MOD((ctime-stime),deltt) == c0)) THEN
          ltdcur = .true.
        ELSE
          ltdcur = .false.
        END IF

!-----------------------------------------------------------------------
!       solve dynamic equations
!-----------------------------------------------------------------------

        IF (itdcoup == 1) THEN   ! thermodynamics option
          ! Load interface temperatures for flux integration
          CALL interface_temps(ticev,tt,tp,hi,Hice)
        END IF

        ! Conjugate gradient 2D solution
        CALL is_dyn (hg,hgk,hi,hik,dhgdt,Hice,Hicek,wvel,vt,vp,tiso,&
                      ub,vb,ubk,vbk,iub,ivb,Bbar,Tbar,Hocean,balance,&
                      bmelt,acc,smelt,calve,dhidt,fpshtk,ftshtk,fpsht,&
                      ftsht,pice,icover,iland,ifloating,&
                      ltdcur,ldiverge,res,itime,fwater)

# if !defined uvic_embm
        IF ((isubtop == 2).or.(isubtop == 3)) THEN
          ! Subgrid ice nucleation and decay
          CALL classify_glacial_cells(hi,hg,Hice,Hicek,sealevel,&
                                      ifloating,iglacial,icelevel,igreen)
        END IF

!-----------------------------------------------------------------------
!       solve surface hydrology evolution at intervals hydnest
!-----------------------------------------------------------------------

        IF ((igingembre > 0).and.(MOD((time-stime),ihydnest) == 0)) THEN

          IF (igingembre == 1) THEN
            CALL surf_hydrol(hi,hg,hg0,Hice,smelt,rainfall,iland,&
                             ifloating,igreen,time)

          ELSE IF (igingembre == 2) THEN
            CALL shydrol_newt(hi,hg,hg0,Hice,smelt,rainfall,iland,&
                              ifloating,igreen,time)
          END IF
        END IF

!-----------------------------------------------------------------------
!       particle tracking
!-----------------------------------------------------------------------

        IF ((itrack == 1).and.(MOD((time-stime),ideltg) == 0)) THEN

          ! Particle tracking, spawn subglacial ducks
          CALL spawn(ndcur,time)
          CALL track(ndcur,vt,vp,hi,hg,Hice,time)

          ! Erode the bed
          CALL striae(vt,vp,wvel,tsole,hg,Hice,time)

        END IF

        IF ((isilt==1).and.(MOD(time,ideltg)==0)) THEN
          CALL debris(vt,vp,hi,hg,tsole,bmelt,time)

        END IF

# endif
!-----------------------------------------------------------------------
!       solve thermal balance at current time step
!-----------------------------------------------------------------------

        IF (ltdcur) THEN

          ! Transform velocity field to temperature grid
          CALL bktrans(vt,vp,wvel,vtt,vpt,wvelt,hi,Hice,itime,iland)

          ! Solve temperature equation
          CALL tfield(tair,ub,vb,hi,hikt,hg,hgkt,Hice,Hicekt,&
                      pice,geo,rest,dtdt,itime,coimp,coexp,cocon,&
                      tiso,tsea,tsole,Tbar,bmelt,Hocean,ifloating)

# if !defined uvic_embm
          ! Write temperature/velocity fields
          iprint = 10000
          IF ((MOD((time-stime),iprint)==0).or.(time==etime)) THEN
            CALL writt(time)
          END IF

# endif
          ! Transform temperature field to dynamical grid
          CALL fdtrans(tice,ticev,hi,Hice)

# if !defined uvic_embm
!-----------------------------------------------------------------------
!         solve bed thermal balance at current time step
!-----------------------------------------------------------------------

          IF ((ibedtd > 0).and.(MOD((time-stime),ideltb)==0)) THEN

            CALL bedtd(tsole,tair,hg,hgkt,Hice,geo,icover,iland,time)

            ! Write results
            CALL writb(hg,time)

          END IF

# endif
        END IF      ! End itherm ifblock

# if !defined uvic_embm
!-----------------------------------------------------------------------
!       convergence or iterout; write current fields
!-----------------------------------------------------------------------

        IF (ntime <= 20000) iprint = 500
        IF (ntime > 20000) iprint = 10000        ! Maintenance output
        IF ((MOD(time,iprint)==0).or.(time==etime).or.(time==-12000) &
                .or. (time==-10000) .or. (time==-9000)) THEN
          CALL writ(hg,Hice,time)
          CALL writv(vt,vp,wvel,fpsht,ftsht,time)
          CALL writmass(balance,acc,smelt,bmelt,pddnet,annamp,tsea,tair,&
                        precip,time)
          IF (igingembre > 0) THEN
            CALL writhyd(hg,hg0,hi,smelt,bmelt,rainfall,iland,time)
          END IF
        END IF
        IF ((MOD(time,100)==0).and.(nlon==63)) THEN       ! EISMINT output
          CALL writeis(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,time)
        ELSE IF ((MOD(time,100)==0).and.(nlon==120)) THEN         ! N America
          IF (isubtop==0) THEN
            CALL writna(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                        calve,tair,tsea,precip,igreen,time)
          ELSE IF (isubtop == 2) THEN                ! subgrid diagnostics
            CALL writna_sub(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                            calve,tair,tsea,precip,igreen,time)
          END IF
        ELSE IF ((MOD(time,200)==0).and.(nlon==30)) THEN          ! Cold Bay
          CALL writeis(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,time)
        END IF

# endif
!-----------------------------------------------------------------------
!       store current as previous solution for next timestep
!-----------------------------------------------------------------------

        hgk   = hg
        Hicek = Hice
        hik   = hi
        pice  = rhoi*grav*Hice

# if defined uvic_embm
!-----------------------------------------------------------------------
!       sum handshake values for subroutine "bism_embm"
!-----------------------------------------------------------------------

        sum_Hice = sum_Hice + Hice
        sum_hi   = sum_hi   + hi
        sum_calv = sum_calv + fwater

!-------for output of global error in moisture conservation due to
!       ice sheet dynamics uncomment following lines
!        vol2 = c0 ! global ice volume after update
!        bal2 = c0 ! global mass balance
!        calv2 = c0 ! global calving rate
!        DO j=1,nlat
!                 DO i=2,nlon-1
!            vol2 = vol2 + Hice(i,j)*area(j)
!            bal2 = bal2 + bal(i,j)*area(j)
!            calv2 = calv2 + sum_calv(i,j)*area(j)
!          END DO
!              END DO
!        write(*,*)'vol1,vol2,bal2,calv2'
!        write(*,1) vol1,vol2,bal2,calv2
!    1   FORMAT(4E16.8)
!        write(*,2)'        vol2-vol1 = ',vol2-vol1
!    2   FORMAT(A20,E16.8)
!        write(*,2)'(bal2-calv2)*delt = ',(bal2-calv2)*delt
# endif

!-----------------------------------------------------------------------
!       calculate and write ice volume
!-----------------------------------------------------------------------

        IF (MOD(icloop,idtvol) == 0) THEN
          CALL cidm_vol_calc(Hice,vol_SH,vol_NH)
          CALL cidm_vol_writ(vol_SH,vol_NH)
        END IF

!-----------------------------------------------------------------------
!       write snapshot
!-----------------------------------------------------------------------

        IF (MOD(icloop,idtsnap) == 0) THEN
          CALL cidm_snap_writ(sum_calv)
        END IF

!-----------------------------------------------------------------------
!     E N D   BISM   M A I N   L O O P
!-----------------------------------------------------------------------

      END DO

# if defined uvic_embm
!-----------------------------------------------------------------------
!     average surface boundary conditions over the coupling period
!-----------------------------------------------------------------------

      sum_Hice  = sum_Hice / DBLE(nloop)
      sum_hi    = sum_hi   / DBLE(nloop)
      sum_calv  = sum_calv / (etime-stime)

!-----------------------------------------------------------------------
!     convert SBC grids for other components of the climate system model
!-----------------------------------------------------------------------

      CALL cidm_sbc_hl(sum_Hice,avg_Hice)
      CALL cidm_sbc_hl(sum_hi,avg_hi)
      CALL cidm_sbc_hl(sum_calv,avg_calv)
      sum_aice = sum_Hice*rhoi/rhos
      sum_aice = MAX(sum_aice,c0)
      sum_aice = MIN(sum_aice,c1)
      CALL cidm_sbc_hl(sum_aice,avg_aice)
      avg_aice = MAX(avg_aice,c0)
      avg_aice = MIN(avg_aice,c1)

# endif
!-----------------------------------------------------------------------
!     write restart
!-----------------------------------------------------------------------

      IF (MOD(ieloop,idtrest) == 0) THEN
        CALL cidm_rest_writ(avg_Hice,avg_hi,avg_calv,avg_aice)
      END IF

#endif
!=======================================================================
      END SUBROUTINE cidm
!=======================================================================
