!=======================================================================

!            Initialization routines, basic ice sheet model

!   Program bism models ice flow over a 2D vertically-integrated
!   ice sheet, using the Mahaffy (1976) ADI scheme on a spherical grid.
!   Thermomechanical coupling is optional.  When invoked, the vertical
!   dimension is added in the dynamic solution and 3D velocity fields
!   are calculated.  Thermodynamics are solved after Jenssen (1977) in
!   the style of Huybrechts (1988) and followers.  Dynamics and
!   thermodynamics are solved on vertically-transformed co-ordinates
!   with linear interpolation between grids.

!   This is a stream-lined, pure ice-sheet version of the general mixture
!   model which includes ice-stream dynamics.  User inputs are required
!   for: numerical/physical option flags, set in script file "bism_opt",
!        model integration limits: timein, timeout, delt, deltt
!        model dimensions and spatial discretization: phimin, phimax,
!                themin, themax, delphid, delthed (in "bism_par")
!         bed topography: load in "loadtopo.f90"
!         intital ice thickness: load in "loadice.f90"
!         temperature and precipitation: used each timestep in
!                                       "degday.f90" and "surfbal.f90"

!   Script file bism_opt sets on/off flags and bism_par sets global
!   parameters and dimensions at run-time.
!   MODULE global_vars declares global arrays and variables.
!   MODULE global_param defines global model and physical constants.
!   All code is implicit none; though I usually use (a-h,o-z) explicitly
!   for double precision variables and (i-n) explicitly for integers.
!   External to MAIN bism, all variables are locally defined.
!   Global variables are passed as arguments in the subroutine calls,
!   including explicit array dimensions.  Global constants are passed
!   through MODULE global_param.

!   author:    s.marshall
!=======================================================================

#if defined uvic_embm
      SUBROUTINE cidm_init(avg_Hice,avg_hi,avg_calv,avg_aice)
#else
      SUBROUTINE cidm_init
#endif

#if defined ubc_cidm
!-----------------------------------------------------------------------
!     declare constants and variables
!-----------------------------------------------------------------------

! ... global
      USE global_vars
      USE subgrid_arrays
# if !defined uvic_embm
      USE deg_day_vars
# endif
      USE hydrol_arrays
      USE geol_arrays
      USE icetd_arrays
      USE bedtd_arrays
      USE shelf_arrays
# if defined uvic_embm
      USE cidm_mod_embm
# endif
      IMPLICIT NONE

# if defined uvic_embm
! ... subroutine arguments
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:iimt,1:jjmt) :: avg_Hice,&
                                            avg_hi,avg_calv,avg_aice
# endif

! ... local
# if defined uvic_embm
      INTEGER :: i,j,k
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:,:) :: sum_aice
# else
      INTEGER :: iyear,isumriv
# endif
      LOGICAL :: ltdcur
      REAL(KIND=dp),EXTERNAL :: B

      WRITE(*,*) '******************* SETCIDM START *******************'

!-----------------------------------------------------------------------
!     load basic model parameters
!-----------------------------------------------------------------------

      WRITE(*,*) 'loading basic model parameters...'
      CALL cidm_input

!-----------------------------------------------------------------------
!     number of tracer grids
!-----------------------------------------------------------------------

      nlon = NINT((phimax-phimin)/delphid)
      nlat = NINT((themax-themin)/delthed)

!-----------------------------------------------------------------------
!     allocate array dimensions and initialize arrays
!-----------------------------------------------------------------------

      WRITE(*,*) 'allocating and initializing arrays...'
      CALL is_arrays

!-----------------------------------------------------------------------
!     calculate grid-cell geometric attributes
!-----------------------------------------------------------------------

      WRITE(*,*) 'calculating geometric constatnts...'
      CALL cidm_geom
# if defined uvic_embm
      CALL cidm_geom_embm
# endif

!-----------------------------------------------------------------------
!     load in vertical grid transforms for ice dynamics
!     (zeta), thermodynamics (xi) and the bed model (xibf, xibc)
!-----------------------------------------------------------------------

      WRITE(*,*) 'setting grid parameters...'
      CALL is_grids

!-----------------------------------------------------------------------
!     sea level, sealevel (m); bed topography, hg (m), and
!     land mask, iland [1/0/-1]
!-----------------------------------------------------------------------

      sealevel = c0

      WRITE(*,*) 'loading bed topography...'
      CALL cidm_get_bed(hg0,hg,iland)
      hgk  = hg
      hgkt = hg

!-----------------------------------------------------------------------
!     ice thickness, Hice (m); pressure, pice (Nm^-2)
!-----------------------------------------------------------------------

      WRITE(*,*) 'loading ice thickness...'
      CALL cidm_get_ice(iland,Hice)
      Hicek  = Hice
      Hicekt = Hice

      pice = rhoi*grav*Hice

!-----------------------------------------------------------------------
!     get surface boundary conditions for atmosphere component
!-----------------------------------------------------------------------

      IF (init == 2) THEN   ! read restart file

!-----------------------------------------------------------------------
!       read restart
!-----------------------------------------------------------------------

        CALL cidm_rest_read(avg_Hice,avg_hi,avg_calv,avg_aice)

!       this was sofar in cidm_get_ice A.S. 04/25/01 cAAA
        IF (ishelf > 0) THEN    ! shelf-ice option
           WHERE (iland == -1) Hice = c0
        ELSE
           WHERE (iland ==  0) Hice = c0
        END IF

        WHERE (Hice < c0) Hice = c0
        WHERE (idomain == -1) Hice = c0

        CALL cidm_lbc_2d_real(hg)
        CALL cidm_lbc_2d_real(Hice)

      ELSE

!-----------------------------------------------------------------------
!       convert grids
!-----------------------------------------------------------------------

        CALL cidm_sbc_hl(Hice,avg_Hice)
        CALL cidm_sbc_hl(hi,avg_hi)
        avg_calv = c0
        ALLOCATE(sum_aice(1:nlon,1:nlat))
        sum_aice = Hice*rhoi/rhos
        sum_aice = MIN(sum_aice,c1)
        sum_aice = MAX(sum_aice,c0)
        CALL cidm_sbc_hl(sum_aice,avg_aice)
        DEALLOCATE(sum_aice)
        avg_aice = MAX(avg_aice,c0)
        avg_aice = MIN(avg_aice,c1)

!-----------------------------------------------------------------------
!       write initial snapshot
!-----------------------------------------------------------------------

        isnap = 0
        CALL cidm_snap_writ(avg_calv)

      END IF

      hgk  = hg
      hgkt = hg

      Hicek  = Hice
      Hicekt = Hice

      pice = rhoi*grav*Hice ! pressure, pice (Nm^-2)

!-----------------------------------------------------------------------
!     ice surface elevation, hi (m)
!     floating ice or grounded ice over the continental shelf,
!     ifloating [1/0]
!-----------------------------------------------------------------------

      WRITE(*,*) 'calculating surface elevation...'
      CALL cidm_get_surf(iland,hg,Hice,ifloating,hi)
      hik  = hi
      hikt = hi
#if !defined uvic_embm
      hi0  = hi
#endif

# if !defined uvic_embm
!-----------------------------------------------------------------------
!     initial topography and land mask
!-----------------------------------------------------------------------

      SELECT CASE(igrid)
      CASE (1,2,4,5,6,7,8)              ! Read in bed topography
        CALL loadtopo(hg0,hgk,hg,hgkt,igreen,landfrac)
      CASE (3)                          ! Synthetic topography
        CALL loadmask(hg0,hgk,hg,hgkt)
      CASE DEFAULT
        WRITE(45,*) 'invalid igrid case in bism: igrid = ', igrid
        STOP
      END SELECT

!-----------------------------------------------------------------------
!     ice thickness
!-----------------------------------------------------------------------

      CALL loadice(hg,hg0,hi,hik,hikt,Hice,Hicek,Hicekt,pice,landfrac,&
                   slope,slope0,hi0,Hocean,iland,ifloating,igreen)
# endif

!-----------------------------------------------------------------------
!     load initial temperature field
!-----------------------------------------------------------------------

      IF (itherm == 0) THEN
        ltdcur = .false.
        tiso = 258.15_dp   ! Isothermal calculation temperature
        Tbar = tiso        ! Array assignment, used for calving
        Biso = B(tiso)
      ELSE IF (itherm == 1) THEN
        ltdcur = .true.                                  ! Interpolate TD --> Dyn
        CALL loadtfld(tiso,tsole,hi,Hice)
        IF (itdcoup == 1) THEN                        ! Thermomechanical flow
          ! Load interface temperatures for flux integration
          CALL fdtrans(tice,ticev,hi,Hice)
        END IF
        Biso = B(tiso)
      END IF

!-----------------------------------------------------------------------
!     Load table lookup for visco-elastic earth resposne
!-----------------------------------------------------------------------

      IF (irebound == 2) THEN            ! Maxwell Earth model, table lookup
        CALL load_earth(vert,Hice,load,loadscale)
      END IF

!-----------------------------------------------------------------------
!     load geothermal flux
!-----------------------------------------------------------------------

      IF (itherm == 1) CALL loadgeo(geo)

# if !defined uvic_embm
      ! Load surface geology
      IF (isurf > 0) THEN
        CALL loadsurf
        CALL sgprop
      END IF

      ! Load bedrock geology
      IF (ibedr > 0) THEN
        CALL loadbedr
      END IF

      ! Sediment transport
      IF (isilt > 0) THEN                 ! Load erodabilities
        CALL loaderode
        CALL loadsilt
      END IF

      ! Sub-grid topographic information (pre-processed)
      IF (isubtop > 0) THEN
        CALL loadterr(hg)
        IF ((isubtop == 2).or.(isubtop == 3)) THEN
          CALL calhypso(hg,hg0,hgk,hgkt,igreen)
          WHERE ((igreen==1).or.(icemask==1)) iglacial = 1
        END IF
      END IF

      ! Bed thermal/groundwater specifications
      IF (ibedtd > 0) THEN
        CALL loadbed
        CALL loadpf(hg,iland)
        CALL loadcov(Hice,icover)
      END IF

      ! Climatological data
      IF (imass < 5) THEN                  ! Parameterized climate
        CALL snowfall(precip)
      ELSE                                ! Climate data
        iyear = 0
        CALL loadsst(tsea,hi0,hg0,delO,delsea,glacial,iyear)
        CALL loadsnow(precip,iyear)
      END IF
      sealevel = c0

      ! Read in river basin map
      IF (igingembre > 0) THEN
        ! Used in shydrol to sum river fluxes
        isumriv = 1
        IF (nlat == 45) THEN
          CALL loadriver(hi,hg,hg0,Hice,iland,igreen,isumriv)
        ELSE IF (nlat == 100) THEN
          CALL loadriver_fine(hi,hg,hg0,Hice,iland,igreen,isumriv)
        END IF
      END IF

      ! Write initial fields
      time = stime
      CALL writ(hg,Hice,time)
      IF (nlon==120) THEN        ! Save N Amer startup information
        IF (isubtop == 0) THEN
          CALL writna(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                      calve,tair,tsea,precip,igreen,time)
        ELSE IF (isubtop == 2) THEN                ! subgrid diagnostics
          CALL writna_sub(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                          calve,tair,tsea,precip,igreen,time)
        END IF
      END IF

# endif

!-----------------------------------------------------------------------
!     calculate global ice volume
!-----------------------------------------------------------------------

      istart = 0
      IF (init == 2) THEN   ! read previous ice volume data
        CALL cidm_vol_read(vol_SH,vol_NH)
      ELSE
        stime  = c0
        CALL cidm_vol_calc(Hice,vol_SH,vol_NH)
        CALL cidm_vol_writ(vol_SH,vol_NH)
      END IF

!-----------------------------------------------------------------------
!     diagnostics
!-----------------------------------------------------------------------

      idtvol  = INT(dtvol/delt)
      idtsnap = INT(dtsnap/delt)
      idtrest = INT(dtrest/delt)

      CALL cidm_summary
      WRITE(*,*) '*******************  SETCIDM END  *******************'

#endif
!=======================================================================
      END SUBROUTINE cidm_init
!=======================================================================
