!=======================================================================
!     This subroutine gives surface accumulation (m yr^{-1}).
!       imass = 1: annual mean snow accumulation
!               2: monthly mean precipitation and temperature
!                  (degree-day method)
!               3: annual mean precipitation and temperature
!                  (degree-day method) --- under construction

!     author:   m.yoshimori
!=======================================================================

      SUBROUTINE cidm_mass_acc

#if defined ubc_cidm
! ... global
      USE cidm_mod_mass   ! s_ann,p_mon,t_mon
      USE global_vars     ! hi,iland,acc
      IMPLICIT NONE

! ... local
      INTEGER :: i,j,k,ipdd
      REAL(KIND=dp) :: a,b,tmean,ss,accfact,elcor

! ... initialization
      acc = c0

      IF (imass == 1) THEN

!-----------------------------------------------------------------------
!       direct method
!-----------------------------------------------------------------------

        WHERE (iland == 1) acc = s_ann
# if defined ubc_cidm_desert
        WHERE ((iland == 1).AND.(hi > 2000._dp)) &
          acc = acc*c2**(-hi/1000._dp+c2)
# endif

      ELSE IF (imass == 2) THEN

!-----------------------------------------------------------------------
!       degree-day method bsed on monthly climatology

!       Specify accumulation from temperature and precip data
!       Numerically integrate for fraction of precip to fall as snow
!       Following Johanneson et al, monthly data
!       Perform 1D quadrature of the deg day function
!       Numerical Recipes Gaussian quadrature routine, is_qgausx
!       Integrate over temperature (T) in deg C
!-----------------------------------------------------------------------

! ..... integration limits over temperature (T) in deg C
        a = -c100   ! -(infinity)
        b = c1      ! snowfall limit

        ipdd = 1    ! function flag
        DO j=2,nlat-1
          DO i=2,nlon-1
            IF (iland(i,j) == 1) THEN   ! over land
              elcor = lapse*MAX(hi(i,j),c0)
              DO k=1,nmonth
                tmean = t_mon(i,j,k) - triplept - elcor   ! deg C
                IF (tmean < (b - cutoff)) THEN
                  accfact = c1
                ELSE IF (tmean > (b + cutoff)) THEN
                  accfact = c0
                ELSE
                  CALL cidm_qgaus(ipdd,a,b,tmean,ss)
                  accfact = ss/(tvar*SQRT(c2*pi))
                  IF (accfact < c0) accfact = c0
                  IF (accfact > c1) accfact = c1
                END IF
                acc(i,j) = acc(i,j) + accfact*p_mon(i,j,k)
              END DO
            END IF
          END DO
        END DO

      END IF

      acc = acc*rhow/rhoi
      CALL cidm_lbc_2d_real(acc)

#endif
!=======================================================================
      END SUBROUTINE cidm_mass_acc
!=======================================================================

!=======================================================================
!     This subroutine gives surface melting.
!       imass = 1: annual snow melting
!               2: monthly temperature (positive degree-day method)

!     author:   m.yoshimori
!=======================================================================

      SUBROUTINE cidm_mass_smelt

#if defined ubc_cidm
! ... global
      USE cidm_mod_mass   ! m_ann,p_mon,t_mon
      USE global_vars     ! hi,iland,acc,smelt,pddnet
      IMPLICIT NONE

! ... local parameters
      REAL(KIND=dp),PARAMETER :: c28 = 28._dp
      REAL(KIND=dp),PARAMETER :: c30 = 30._dp
      REAL(KIND=dp),PARAMETER :: c31 = 31._dp

! ... local variables
      INTEGER :: i,j,k,ipdd
      REAL(KIND=dp) :: a,b,tmean,ss,elcor,&
                       dh,pddice,pddsnow,snowmelt,icemelt,refreeze
      REAL(KIND=dp),DIMENSION(1:nlon,1:nlat) :: pddtmp
      REAL(KIND=dp),DIMENSION(1:nlon,1:nlat,1:nmonth) :: pdd
      REAL(KIND=dp),DIMENSION(1:nmonth) :: dayinmonth = &
      (/c31,c28,c31,c30,c31,c30,c31,c31,c30,c31,c30,c31/)

!-----------------------------------------------------------------------
!     Numerically integrate for positive degree days
!     Following Johannesson et al (1995), use monthly temperature
!     data with Gaussian pdf, SD of 4.5\degC
!     Perform 1D quadrature of the deg day function
!     Numerical Recipes Gaussian quadrature routine, is_qgausx
!     Integrate over temperature (T) in deg C
!-----------------------------------------------------------------------

! ... initialization
      smelt  = c0

      IF (imass == 1) THEN

        WHERE (iland == 1) smelt = m_ann

      ELSE IF (imass == 2) THEN

        pddnet = c0
        pdd    = c0

! ..... integration limits over temperature (T) in deg C
        a = c0
        b = c100   ! (infinity)

        ipdd = 2   ! function flag
        DO j=1,nlat
          DO i=1,nlon
            elcor = lapse*MAX(hi(i,j),c0)
            DO k=1,nmonth
              tmean = t_mon(i,j,k) - triplept - elcor   ! deg C
              IF (tmean < -cutoff) THEN
                pdd(i,j,k) = c0
              ELSE
                CALL cidm_qgaus(ipdd,a,b,tmean,ss)
                IF (ss < c0) ss = c0
                pdd(i,j,k) =  ss*dayinmonth(k)/(tvar*SQRT(c2*pi))
              END IF
              pddnet(i,j) =  pddnet(i,j) + pdd(i,j,k)
            END DO
          END DO
        END DO

!-----------------------------------------------------------------------
!       Calculate effective PDD based on interface elevations,
!       assuming linear piecewise slope
!-----------------------------------------------------------------------

        pddtmp = pddnet
        DO j=2,nlat-1
          DO i=2,nlon-1
            pddnet(i,j) = r6*(pddtmp(i+1,j) + pddtmp(i-1,j) &
                        + pddtmp(i,j-1) + pddtmp(i,j+1) &
                        + c4*pddtmp(i,j))
          END DO
        END DO

!-----------------------------------------------------------------------
!       calculate ablation from annual degree day factors
!       and the amount of new snow (vs. ice).  Snow is melted first,
!       from this year's accumulation, then ice.
!-----------------------------------------------------------------------

        DO j=2,nlat-1
          DO i=2,nlon-1
            IF (iland(i,j) == 1) THEN
              dh = hi(i,j) &
                 - p25*(hi(i+1,j)+hi(i-1,j)+hi(i,j+1)+hi(i,j-1))
              refreeze = MAX(refreeze_max-ABS(dh)/c100,refreeze_min)
              snowmelt = pddnet(i,j)*ddfsnow*(c1-refreeze)
              IF (snowmelt > acc(i,j)) THEN
                snowmelt = acc(i,j)
                pddsnow = snowmelt/(ddfsnow*(c1-refreeze))
                pddice = pddnet(i,j) - pddsnow
                icemelt = pddice*ddfice*(c1-refreeze)
                smelt(i,j) = snowmelt + icemelt
              ELSE
                smelt(i,j) = snowmelt
              END IF
            END IF
          END DO
        END DO

      END IF

      smelt = smelt*rhow/rhoi
      CALL cidm_lbc_2d_real(smelt)

#endif
!=======================================================================
      END SUBROUTINE cidm_mass_smelt
!=======================================================================

!=======================================================================
!     This subroutine gives surface accumulation (m yr^{-1}).
!       imass = 1: annual mean snow accumulation
!               2: monthly mean precipitation and temperature
!                  (degree-day method)
!               3: annual mean precipitation and temperature
!                  (degree-day method) --- under construction

!     author:   m.yoshimori
!=======================================================================

      SUBROUTINE cidm_mass_balance

#if defined ubc_cidm
! ... global
      USE cidm_mod_mass   ! s_ann,m_ann,r_ann
      USE global_vars     ! acc,smelt,balance
      IMPLICIT NONE

! ... local
      REAL(KIND=dp),DIMENSION(1:nlon,1:nlat) :: esc

      IF (imass == 1) THEN

# if defined ubc_cidm_refreeze
        WHERE (Hice > c0)   ! neglect evaporation
          esc = MAX(c0,MIN(c1,(m_ann/s_ann-0.7_dp)/0.3_dp))
          balance = (s_ann + r_ann - esc*(m_ann + r_ann))*rhow/rhoi
        ELSEWHERE
          balance = acc - smelt
        END WHERE
# else
        balance = acc - smelt
# endif

      ELSE IF (imass == 2) THEN

        balance = acc - smelt

      END IF

#endif
!=======================================================================
      END SUBROUTINE cidm_mass_balance
!=======================================================================

!=======================================================================
!     Numerical recipes routine to perform 1D quadrature
!     (C) Copr. 1986-92 Numerical Recipes Software 5,29#(.

!     author:   m.yoshimori
!=======================================================================

      SUBROUTINE cidm_qgaus(ipdd,a,b,tmean,ss)

#if defined ubc_cidm
      IMPLICIT NONE

! ... local parameter
      INTEGER,PARAMETER :: dp = 8
      REAL(KIND=dp),PARAMETER :: c0 = 0._dp
      REAL(KIND=dp),PARAMETER :: c1 = 1._dp
      REAL(KIND=dp),PARAMETER :: c2 = 2._dp
      REAL(KIND=dp),PARAMETER :: r2 = c1/c2

      INTEGER,INTENT(IN) :: ipdd
      REAL(KIND=dp),INTENT(IN) :: a,b,tmean
      REAL(KIND=dp),INTENT(OUT) :: ss

      INTEGER :: j
      REAL(KIND=dp) :: dx,xm,xr
      REAL(KIND=dp),DIMENSION(5) :: w = (/ 0.2955242247, 0.2692667193, &
                          0.2190863625, 0.1494513491, 0.0666713443 /), &
                                    x = (/ 0.1488743389, 0.4333953941, &
                          0.6794095682, 0.8650633666, 0.9739065285 /)
      REAL(KIND=dp),EXTERNAL :: cidm_qgaus_func

      xm = (b+a)*r2
      xr = (b-a)*r2
      ss = c0
      DO j=1,5
        dx = xr*x(j)
        ss = ss+w(j)*(cidm_qgaus_func(ipdd,xm+dx,tmean) &
                     +cidm_qgaus_func(ipdd,xm-dx,tmean))
      END DO
      ss = xr*ss

#endif
!=======================================================================
      END SUBROUTINE cidm_qgaus
!=======================================================================

!=======================================================================
!     Function for inner integrand of degree-day model and
!     precip factor model.  Called by is_qgausy.

!     author:   m.yoshimori
!=======================================================================

      FUNCTION cidm_qgaus_func(ipdd,T,tmean) RESULT(res)

#if defined ubc_cidm
! ... global
      USE cidm_mod_mass   ! tvar
      IMPLICIT NONE

! ... subroutine arguments
      INTEGER,INTENT(IN) :: ipdd
      REAL(KIND=dp),INTENT(IN) :: T,tmean
      REAL(KIND=dp) :: res

      SELECT CASE (ipdd)
      CASE (1)   ! snow fraction
        res = EXP(-(T-tmean)**2/(c2*tvar**2))
      CASE (2)   ! positive degree factor
        res = T*EXP(-(T-tmean)**2/(c2*tvar**2))
      CASE DEFAULT
        WRITE(*,*) 'invalid value for ipdd: ipdd = ', ipdd
        STOP
      END SELECT

#endif
!=======================================================================
      END FUNCTION cidm_qgaus_func
!=======================================================================
