      subroutine cidm_snap_in (fname, ids, ide, jds, jde, imt, jmt
     &,                        xt, yt, ntrec, timunit
     &,                        expnam, avgper, relyr, stamp
     &,                        hg, hice, hi, balance
#if defined rot_grid
     &,                         tlat, tlon, ulat, ulon
#endif
     &                         )
!=======================================================================
!     output routine for atmospheric time averages or snapshots

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain
!     imt, jmt ...       = global array dimensions
!     xt, yt ...         = global axes
!     dxt, dyt ...       = grid widths
!     ntrec              = number of time record (zero if not defined)

!   outputs:
!     timunit            = time unit
!     expnam             = experiment name
!     avgper             = length of averaging period
!     relyr              = time in years
!     stamp              = time stamp
!     hg, ...            = data to be read

!   local variables
!     igs, ige, jgs, jge = global read domain start and end indicies
!     ig, jg             = global read domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for read on each axis (x,y,t default)
!     ic                 = count for read on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm && defined uvic_netcdf

      implicit none

      character(*) :: fname, timunit, expnam, stamp
      character(80) :: name, title, file_stamp
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, imt, jmt
      integer ids, ide, jds, jde, igs, ige, ig, jgs, jge, jg
      integer ils, ile, jls, jle, is(10), ic(10), it(10)
      integer iu(10), id_time, id_xt, id_xu, id_yt, id_yu
      integer id_xt_e, id_xu_e, id_yt_e, id_yu_e
      integer iyear, imonth, iday, ihour, imin, isec

      real xt(imt), xu(imt), yt(jmt), yu(jmt)
      real dxt(imt), dxu(imt), dyt(jmt), dyu(jmt), avgper
      real hg(ids:ide,jds:jde), hice(ids:ide,jds:jde)
      real hi(ids:ide,jds:jde), balance(ids:ide,jds:jde)
# if defined rot_grid
      real tlat(ids:ide,jds:jde), tlon(ids:ide,jds:jde)
      real ulat(ids:ide,jds:jde), ulon(ids:ide,jds:jde)
# endif
      real relyr, ryear, rmonth, rday, rhour, rmin, rsec
      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      i = 0
      call openfile (name, relyr, i, iou)

!-----------------------------------------------------------------------
!     get global attributes
!-----------------------------------------------------------------------
      call getglobal (iou, fname, title, expnam, timunit, 0.)

!-----------------------------------------------------------------------
!     get global read domain size (may be less than global domain)
!-----------------------------------------------------------------------
      call getaxis('cidm_xt', iou, imt, xt, igs, ige, 1., 0.)
!      call getaxis('cidm_xu', iou, imt, xu, igs, ige, 1., 0.)
      ig  = ige-igs+1
      call getaxis('cidm_yt', iou, jmt, yt, jgs, jge, 1., 0.)
!      call getaxis('cidm_yu', iou, jmt, yu, igs, ige, 1., 0.)
      jg  = jge-jgs+1

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global read domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)

!-----------------------------------------------------------------------
!     set start and count arrays (default for 3d is x,y,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      is(3) = ntrec
      ic(3) = 1
      is(4) = ntrec
      ic(4) = 1
      ln = ic(1)*ic(2)

!-----------------------------------------------------------------------
!     read 1d data (t)
!-----------------------------------------------------------------------
      call getvara ('time', iou, 1, ntrec, 1, relyr, 1., 0.)
      call getvara ('year', iou, 1, ntrec, 1, ryear, 1., 0.)
      call getvara ('month', iou, 1, ntrec, 1, rmonth, 1., 0.)
      call getvara ('day', iou, 1, ntrec, 1, rday, 1., 0.)
      iyear = int(ryear)
      imonth = int(rmonth)
      iday = int(rday)
      rhour = (rday - real(iday))*24.
      ihour = int(rhour)
      rmin = (rhour - real(ihour))*60.
      imin = int(rmin)
      rsec = (rmin - real(imin))*60.
      isec = int(rsec)
      call mkstmp (stamp, iyear, imonth, iday, ihour, imin, isec)
      call getvara ('period', iou, 1, ntrec, 1, avgper, 1., 0.)

!-----------------------------------------------------------------------
!     read 2d data (x,y)
!-----------------------------------------------------------------------
# if defined rot_grid
      call getvara ('cidm_tlat', iou, ln, is, ic, tlat(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('cidm_tlon', iou, ln, is, ic, tlon(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('cidm_ulat', iou, ln, is, ic, ulat(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('cidm_ulon', iou, ln, is, ic, ulon(ils:ile,jls:jle)
     &, 1., 0.)
# endif

!-----------------------------------------------------------------------
!     read 3d data (x,y,t)
!-----------------------------------------------------------------------
      call getvara ('cidm_hg', iou, ln, is, ic
     &, hg(ils:ile,jls:jle), 1., 1.)
      call getvara ('cidm_hice', iou, ln, is, ic
     &, hice(ils:ile,jls:jle), 1., 1.)
      call getvara ('cidm_hi', iou, ln, is, ic
     &, hi(ils:ile,jls:jle), 1., 1.)
      call getvara ('cidm_balance', iou, ln, is, ic
     &, balance(ils:ile,jls:jle), 1., 1.)

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)
#endif

      return
      end

      subroutine cidm_snap_out (fname, ids, ide, jds, jde, imt, jmt
     &,                         xt, yt, ntrec, timunit
     &,                         expnam, avgper, relyr, stamp
     &,                         hg, hice, hi, balance
#if defined rot_grid
     &,                          tlat, tlon, ulat, ulon
#endif
     &                          )
!=======================================================================
!     output routine for atmospheric time averages or snapshots

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain
!     imt, jmt ...       = global array dimensions
!     xt, yt ...         = global axes
!     dxt, dyt ...       = grid widths
!     ntrec              = number of time record (zero if not defined)
!     timunit            = time unit
!     expnam             = experiment name
!     avgper             = length of averaging period
!     relyr              = time in years
!     stamp              = time stamp
!     hg, ...            = data to be written

!   outputs:
!     ntrec              = number of time record in file

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm && defined uvic_netcdf

      implicit none

      character(*) :: fname, timunit, expnam, stamp
      character(80) :: name, title, file_stamp
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, imt, jmt
      integer ids, ide, jds, jde, igs, ige, ig, jgs, jge, jg
      integer ils, ile, jls, jle, is(10), ic(10), it(10)
      integer iu(10), id_time, id_xt, id_xu, id_yt, id_yu
      integer id_cat, id_xt_e, id_xu_e, id_yt_e, id_yu_e
      integer iyear, imonth, iday, ihour, imin, isec

      real xt(imt), xu(imt), yt(jmt), yu(jmt)
      real dxt(imt), dxu(imt), dyt(jmt), dyu(jmt), avgper
      real hg(ids:ide,jds:jde), hice(ids:ide,jds:jde)
      real hi(ids:ide,jds:jde), balance(ids:ide,jds:jde)
# if defined rot_grid
      real tlat(ids:ide,jds:jde), tlon(ids:ide,jds:jde)
      real ulat(ids:ide,jds:jde), ulon(ids:ide,jds:jde)
# endif
      real relyr, ryear, rmonth, rday, rhour, rmin, rsec
      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)

      logical defined

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      defined = .true.
      name = fname
      if (ntrec .eq. 0) defined = .false.
      call openfile (name, relyr, ntrec, iou)

!-----------------------------------------------------------------------
!     set global write domain size (may be less than global domain)
!-----------------------------------------------------------------------
# if defined cyclic
      igs = 2
      ige = imt-1
# else
      igs = 1
      ige = imt
# endif
      ig  = ige-igs+1
      jgs = 1
      jge = jmt
      do j=2,jmt
        if (yt(j-1) .lt. -90. .and. yt(j) .gt. -90.) jgs = j
        if (yt(j-1) .lt.  90. .and. yt(j) .gt. 90.) jge = j-1
      enddo
      jg  = jge-jgs+1

      if (.not. defined .or. ntrec .eq. 0) then

!-----------------------------------------------------------------------
!       start definitions
!-----------------------------------------------------------------------
        call redef (iou)

!-----------------------------------------------------------------------
!       set global attributes
!-----------------------------------------------------------------------
        if (avgper .gt. 1.e-6) then
          title = 'Time Average'
        else
          title = 'Snapshot'
        endif
        call putglobal (iou, name, title, expnam, timunit, 0.)

!-----------------------------------------------------------------------
!       define dimensions
!-----------------------------------------------------------------------
        call defdim ('time', iou, 0, id_time)
        call defdim ('cidm_xt', iou, ig, id_xt)
        call defdim ('cidm_yt', iou, jg, id_yt)
        call defdim ('cidm_xu', iou, ig, id_xu)
        call defdim ('cidm_yu', iou, jg, id_yu)
        call defdim ('cidm_xt_edges', iou, ig+1, id_xt_e)
        call defdim ('cidm_yt_edges', iou, jg+1, id_yt_e)
        call defdim ('cidm_xu_edges', iou, ig+1, id_xu_e)
        call defdim ('cidm_yu_edges', iou, jg+1, id_yu_e)
!       define output axis id's for arrays (default for 3d is x,y,t)
        it(1) = id_xt
        iu(1) = id_xu
        it(2) = id_yt
        iu(2) = id_yu
        it(3) = id_time
        iu(3) = id_time
        it(4) = id_time
        iu(4) = id_time

!-----------------------------------------------------------------------
!       define 1d data (t)
!-----------------------------------------------------------------------
        call defvar ('time', iou, 1, id_time, 0., 0., 'T', 'D'
     &,   'time since initial condition', 'time', trim(timunit))

        call defvar ('year', iou, 1, id_time, 0., 0., ' ', 'F'
     &,   'year', ' ',' ')

        call defvar ('month', iou, 1, id_time, 0., 0., ' ', 'F'
     &,   'month', ' ',' ')

        call defvar ('day', iou, 1, id_time, 0., 0., ' ', 'F'
     &,   'day', ' ',' ')

        if (avgper .gt. 1.e-6) then
          call defvar ('period', iou, 1, id_time, 0., 0., ' ', 'F'
     &,     'averaging period', ' ',' ')
        endif

!-----------------------------------------------------------------------
!       define 1d data (x or y)
!-----------------------------------------------------------------------
        call defvar ('cidm_xt', iou, 1, id_xt, 0., 0., 'X', 'F'
     &,   'longitude of the t grid', 'grid_longitude', 'degrees_east')

        call defvar ('cidm_yt', iou, 1, id_yt, 0., 0., 'Y', 'F'
     &,   'latitude of the t grid', 'grid_latitude', 'degrees_north')

!        call defvar ('cidm_xu', iou, 1, id_xu, 0., 0., 'X', 'F'
!     &,   'longitude of the u grid', 'grid_longitude', 'degrees_east')

!        call defvar ('cidm_yu', iou, 1, id_yu, 0., 0., 'Y', 'F'
!     &,   'latitude of the u grid', 'grid_latitude', 'degrees_north')

!        call defvar ('cidm_xt_edges', iou, 1, id_xt_e, 0., 0., 'X', 'F'
!     &,   'longitude of t grid edges','grid_edge_longitude'
!     &,   'degrees_east')

!        call defvar ('cidm_yt_edges', iou, 1, id_yt_e, 0., 0., 'Y', 'F'
!     &,   'latitude of t grid edges', 'grid_edge_latitude'
!     &,   'degrees_north')

!        call defvar ('cidm_xu_edges', iou, 1, id_xu_e, 0., 0., 'X', 'F'
!     &,   'longitude of u grid edges', 'grid_edge_longitude'
!     &,   'degrees_east')

!        call defvar ('cidm_yu_edges', iou, 1, id_yu_e, 0., 0., 'Y', 'F'
!     &,   'latitude of u grid edges', 'grid_edge_latitude'
!     &,   'degrees_north')

!-----------------------------------------------------------------------
!       define 2d data (x,y)
!-----------------------------------------------------------------------
# if defined rot_grid
        call defvar ('cidm_tlat', iou, 2, it, -1.e4, 1.e4, ' ', 'F'
     &,   'tracer grid latitude', 'latitude', 'deg')
        call defvar ('cidm_tlon', iou, 2, it, -1.e4, 1.e4, ' ', 'F'
     &,   'tracer grid longitude', 'longitude', 'deg')
        call defvar ('cidm_ulat', iou, 2, iu, -1.e4, 1.e4, ' ', 'F'
     &,   'velocity grid latitude', 'latitude', 'deg')
        call defvar ('cidm_ulon', iou, 2, iu, -1.e4, 1.e4, ' ', 'F'
     &,   'velocity grid longitude', 'longitude', 'deg')

# endif
!-----------------------------------------------------------------------
!       define 3d data (x,y,t)
!-----------------------------------------------------------------------
        call defvar ('cidm_hg', iou, 3, it, -1.e6, 1.e6, ' ', 'F'
     &,   'bed topography', '', 'm')
        call defvar ('cidm_hice', iou, 3, it, -1.e6, 1.e6, ' ', 'F'
     &,   'ice sheet thickness', '', 'm')
        call defvar ('cidm_hi', iou, 3, it, -1.e6, 1.e6, ' ', 'F'
     &,   'ice sheet surface', '', 'm')
        call defvar ('cidm_balance', iou, 3, it, -1.e6, 1.e6, ' ', 'F'
     &,   'ice sheet mass balance rate', '', 'm/a')

!-----------------------------------------------------------------------
!       end definitions
!-----------------------------------------------------------------------
        call enddef (iou)
        if (ntrec .eq. 0 ) ntrec = 1

      endif

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global write domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)

!-----------------------------------------------------------------------
!     set start and count arrays (default for 3d is x,y,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      is(3) = ntrec
      ic(3) = 1
      is(4) = ntrec
      ic(4) = 1
      ln = ic(1)*ic(2)

!-----------------------------------------------------------------------
!     write 1d data (t)
!-----------------------------------------------------------------------
      call putvara ('time', iou, 1, ntrec, 1, relyr, 1., 0.)
      call rdstmp (stamp, iyear, imonth, iday, ihour, imin, isec)
      rday = real(iday) + real(ihour)/24. + real(imin)/3600.
     &     + real(isec)/86400.
      call putvara ('year', iou, 1, ntrec, 1, real(iyear), 1., 0.)
      call putvara ('month', iou, 1, ntrec, 1, real(imonth), 1., 0.)
      call putvara ('day', iou, 1, ntrec, 1, rday, 1., 0.)
      if (avgper .gt. 1.e-6) then
        call putvara ('period', iou, 1, ntrec, 1, avgper, 1., 0.)
      endif

      if (ntrec .eq. 1 .or. .not. defined) then

!-----------------------------------------------------------------------
!       write 1d data (x or y)
!-----------------------------------------------------------------------
        call putvara ('cidm_xt', iou, ig, 1, ig, xt(igs:ige), 1., 0.)
        call putvara ('cidm_yt', iou, jg, 1, jg, yt(jgs:jge), 1., 0.)
!        call putvara ('cidm_xu', iou, ig, 1, ig, xu(igs:ige), 1., 0.)
!        call putvara ('cidm_yu', iou, jg, 1, jg, yu(jgs:jge), 1., 0.)
!        call edge_maker (1, xt_e, xt, dxt, xu, dxu, imt)
!        call putvara ('cidm_xt_edges', iou, ig+1, 1, ig+1
!     &,   xt_e(igs:ige+1), 1., 0.)
!        call edge_maker (1, yt_e, yt, dyt, yu, dyu, jmt)
!        call putvara (cidm_'yt_edges', iou, jg+1, 1, jg+1
!     &,   yt_e(jgs:jge+1), 1., 0.)
!        call edge_maker (2, xu_e, xt, dxt, xu, dxu, imt)
!        call putvara ('cidm_xu_edges', iou, ig+1, 1, ig+1
!     &,   xu_e(igs:ige+1), 1., 0.)
!        call edge_maker (2, yu_e, yt, dyt, yu, dyu, jmt)
!        call putvara ('cidm_yu_edges', iou, jg+1, 1, jg+1
!     &,   yu_e(jgs:jge+1), 1., 0.)

!-----------------------------------------------------------------------
!       write 2d data (x,y)
!-----------------------------------------------------------------------
#  if defined rot_grid
        call putvara ('cidm_tlat', iou, ln, is, ic
     &,   tlat(ils:ile,jls:jle), 1., 0.)
          call putvara ('cidm_tlon', iou, ln, is, ic
     &, tlon(ils:ile,jls:jle), 1., 0.)
          call putvara ('cidm_ulat', iou, ln, is, ic
     &, ulat(ils:ile,jls:jle), 1., 0.)
        call putvara ('cidm_ulon', iou, ln, is, ic
     &,   ulon(ils:ile,jls:jle), 1., 0.)
#  endif

      endif

!-----------------------------------------------------------------------
!     write 3d data (x,y,t)
!-----------------------------------------------------------------------
      call putvara ('cidm_hg', iou, ln, is, ic
     &, hg(ils:ile,jls:jle), 1., 1.)
      call putvara ('cidm_hice', iou, ln, is, ic
     &, hice(ils:ile,jls:jle), 1., 1.)
      call putvara ('cidm_hi', iou, ln, is, ic
     &, hi(ils:ile,jls:jle), 1., 1.)
      call putvara ('cidm_balance', iou, ln, is, ic
     &, balance(ils:ile,jls:jle), 1., 1.)

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)
#endif

      return
      end
