!*****************************************************************************
!   Flux control module: makes sure no more ice leaves a cell than
!   is available.  General routine to check for both sheet or stream
!   fluxes (isheet=1,0).  If fluxes must be corrected, volume-conservative
!   values are returned in the original flux-variables.

!   Must limit ice flux out of cell based on mass conservation
!   vtotala = available volume flux currently in the cell (m^3/yr)
!   vtotalc = calculated volume flux in/out
!   vtotin/vtotout break vtotalc down
!   vtotoutp = total available ice in cell (influx+previous)

!   Note that sheet fluxes and stream fluxes have opposite sign
!   convention; stream fluxes are intuitive (positive right, down).

    SUBROUTINE flux_control(ub,vb,iub,ivb,Hvimp,balance,bmelt,fpsht,&
                            ftsht,scalert,scalehi,Hrt,Hhi)

!*****************************************************************************
      USE shelf_arrays
      IMPLICIT NONE
      INTEGER, INTENT(IN), DIMENSION(nlon-1,nlatp) :: iub
      INTEGER, INTENT(IN), DIMENSION(nlonp,nlat-1) :: ivb
      REAL(dp), INTENT(IN), DIMENSION(ndsys) :: Hvimp
      REAL(dp), INTENT(IN), DIMENSION(nlon,nlat) :: bmelt
      REAL(dp), INTENT(IN), DIMENSION(nlon-1,nlatp) :: Hrt
      REAL(dp), INTENT(IN), DIMENSION(nlonp,nlat-1) :: Hhi
      REAL(dp), INTENT(INOUT), DIMENSION(nlon,nlat) :: balance
      REAL(dp), INTENT(INOUT), DIMENSION(nlon-1,nlatp) :: ub,fpsht,scalert
      REAL(dp), INTENT(INOUT), DIMENSION(nlonp,nlat-1) :: vb,ftsht,scalehi

      ! Local
      INTEGER :: i,j,ij,jj,ip,jp,k
      REAL(dp) :: vfhi,vflo,vfrt,vflt,vtotala,vtotout,vtotin,vtotoutp,&
                  vfrac,fluxhi,fluxlo,fluxrt,fluxlt

!******************************************************************************
    ! Loop over the interior (physical) model grid, correcting all
    ! outgoing interface fluxes on a cell-by-cell basis

    DO j=2,nlat-1
      jj = j-1
      jp = j+1

      DO i=2,nlon-1
        ij = i-1
        ip = i+1
        k = i + nlon*jj                  ! vector

!*****************************************************************************
        ! Initialize
        fluxrt = fpsht(i,jj)
        fluxlt = fpsht(ij,jj)
        fluxhi = ftsht(ij,j)
        fluxlo = ftsht(ij,jj)

        ! Interface transport volumes (m^3/a)
        vfhi = fluxhi*delxhi(j)
        vflo = fluxlo*delxhi(jj)
        vfrt = fluxrt*dy
        vflt = fluxlt*dy

!*****************************************************************************
        ! Adjust fluxes out of current cell based on available ice
        vtotala = area(j)*(Hvimp(k)/delt + balance(i,j) - bmelt(i,j))
                                     ! Volume currently in cell (m^3/a)

        vtotout = MAX(c0,vfhi) + MAX(c0,vflt) - MIN(c0,vflo) - MIN(c0,vfrt)
        vtotin = -MIN(c0,vfhi) - MIN(c0,vflt) + MAX(c0,vflo) + MAX(c0,vfrt)
                                     ! Volume flux out/in of cell (m^3/a)

        vtotoutp = vtotala + vtotin  ! Maximum possible export volume (m^3/a)

        IF (vtotoutp < c0) THEN                      ! Excess melting
          ! Adjust mass balance rate to represent true ablation
!          balance(i,j) = -vtotin/area(j) - Hvimp(k)/delt + bmelt(i,j)
          vtotoutp = c0
        END IF

        ! Jump to next point when no efflux from the cell
        IF (vtotout < zerom) CYCLE

        IF (vtotoutp < zerom) THEN            ! No ice whatsoever available
          IF (vfhi > c0) fluxhi = c0
          IF (vflo < c0) fluxlo = c0
          IF (vflt > c0) fluxlt = c0
          IF (vfrt < c0) fluxrt = c0

        ELSE IF (vtotout > vtotoutp) THEN
          vfrac = vtotoutp/vtotout            ! Limited ice available

          IF (vfhi > c0) THEN
            fluxhi = fluxhi*vfrac
            IF ((ivb(ij,j)==1) .or. (ifloathi(ij,j)==1)) &     ! Basal flow
                                           vb(ij,j) = vb(ij,j)*vfrac
          END IF
          IF (vflo < c0) THEN
            fluxlo = fluxlo*vfrac
            IF ((ivb(ij,jj)==1) .or. (ifloathi(ij,jj)==1)) &   ! Basal flow
                                           vb(ij,jj) = vb(ij,jj)*vfrac
          END IF
          IF (vflt > c0) THEN
            fluxlt = fluxlt*vfrac
            IF ((iub(ij,jj)==1) .or. (ifloatrt(ij,jj)==1)) &   ! Basal flow
                                           ub(ij,jj) = ub(ij,jj)*vfrac
          END IF
          IF (vfrt < c0) THEN
            fluxrt = fluxrt*vfrac
            IF ((iub(i,jj)==1) .or. (ifloatrt(i,jj)==1)) &     ! Basal flow
                                           ub(i,jj) = ub(i,jj)*vfrac
          END IF
        END IF

!*****************************************************************************
        ! Reassign fluxes and assign scaling factors for the Jacobian matrix
        IF (ABS(fluxrt) < ABS(fpsht(i,jj))) THEN
          IF (ifloatrt(i,jj) == 0)  scalert(i,jj) = fluxrt/fpsht(i,jj)
          fpsht(i,jj) = fluxrt
        END IF

        IF (ABS(fluxlt) < ABS(fpsht(ij,jj))) THEN
          IF (ifloatrt(ij,jj) == 0) scalert(ij,jj) = fluxlt/fpsht(ij,jj)
          fpsht(ij,jj) = fluxlt
        END IF

        IF (ABS(fluxhi) < ABS(ftsht(ij,j))) THEN
          IF (ifloathi(ij,j) == 0)  scalehi(ij,j) = fluxhi/ftsht(ij,j)
          ftsht(ij,j) = fluxhi
        END IF

        IF (ABS(fluxlo) < ABS(ftsht(ij,jj))) THEN
          IF (ifloathi(ij,jj) == 0) scalehi(ij,jj) = fluxlo/ftsht(ij,jj)
          ftsht(ij,jj) = fluxlo
        END IF

        ! Stupidity check
        IF ((scalert(i,jj) < c0) .or. (scalert(i,jj) > c1)) THEN
          WRITE(45,*) 'Err - east interface scaling = ', scalert(i,jj)
          WRITE(45,*) '   Point ', i,j, ' vtotout, vtotin ', vtotout,vtotin
          WRITE(45,*) '   fluxes ', fluxrt, fluxhi, fluxlt, fluxlo
          WRITE(45,*) '   available ice : ', vtotala, 'm^3, H =', Hvimp(k)
        END IF

      END DO
    END DO

!***************************************************************************
  END SUBROUTINE flux_control
!***************************************************************************
