!=======================================================================
!     Define physical and model constants
!     Declare model options and dimensions, for global accessibility
!=======================================================================

      MODULE global_param

      IMPLICIT NONE
      SAVE

!-----------------------------------------------------------------------
!     numerical precision
!-----------------------------------------------------------------------

      INTEGER, PARAMETER :: sp = SELECTED_REAL_KIND(P=6)
#if defined uvic_embm
      INTEGER, PARAMETER :: dp = 8
#else
      INTEGER, PARAMETER :: dp = SELECTED_REAL_KIND(PRECISION(1.0d0))
#endif

!-----------------------------------------------------------------------
!     numerical constants
!-----------------------------------------------------------------------

      REAL(KIND=dp),PARAMETER :: c0 = 0.0_dp
      REAL(KIND=dp),PARAMETER :: c1 = 1.0_dp
      REAL(KIND=dp),PARAMETER :: c2 = 2.0_dp
      REAL(KIND=dp),PARAMETER :: c3 = 3.0_dp
      REAL(KIND=dp),PARAMETER :: c4 = 4.0_dp
      REAL(KIND=dp),PARAMETER :: c5 = 5.0_dp
      REAL(KIND=dp),PARAMETER :: c6 = 6.0_dp
      REAL(KIND=dp),PARAMETER :: c7 = 7.0_dp
      REAL(KIND=dp),PARAMETER :: c8 = 8.0_dp
      REAL(KIND=dp),PARAMETER :: c9 = 9.0_dp
      REAL(KIND=dp),PARAMETER :: c10 = 10.0_dp
      REAL(KIND=dp),PARAMETER :: c12 = 12.0_dp
      REAL(KIND=dp),PARAMETER :: c90 = 90.0_dp
      REAL(KIND=dp),PARAMETER :: c100 = 100.0_dp
      REAL(KIND=dp),PARAMETER :: r2 = c1/c2
      REAL(KIND=dp),PARAMETER :: r3 = c1/c3
      REAL(KIND=dp),PARAMETER :: r6 = c1/c6
      REAL(KIND=dp),PARAMETER :: r12 = c1/c12
      REAL(KIND=dp),PARAMETER :: p25 = c1/c4
      REAL(KIND=dp),PARAMETER :: p75 = c3/c4
      REAL(KIND=dp),PARAMETER :: spval = 1.E38_dp

      ! Universal physical constants
      REAL(KIND=dp), PARAMETER :: grav = 9.81_dp
      REAL(KIND=dp), PARAMETER :: erad = 6.370e6_dp !cAAA 04/25/01
      REAL(KIND=dp), PARAMETER :: pi = 3.14159265358979_dp
      REAL(KIND=dp), PARAMETER :: minute = 60._dp
      REAL(KIND=dp), PARAMETER :: hour = 3600._dp
      REAL(KIND=dp), PARAMETER :: day = hour*24._dp
      REAL(KIND=dp), PARAMETER :: month = day*365.24_dp/c12
      REAL(KIND=dp), PARAMETER :: year = day*365.24_dp
      REAL(KIND=dp), PARAMETER :: degtrad = pi/180._dp
      REAL(KIND=dp), PARAMETER :: shelfd = -120._dp    ! m below sea level
      INTEGER, PARAMETER :: nmonth = 12                       ! Months
      INTEGER, PARAMETER :: ncoord = 2                 ! Co-ordinate components
      INTEGER, PARAMETER :: nface = 4                  ! Co-ordinate components

      ! Ice properties (modified SI - annum)
#if defined uvic_embm
      REAL(KIND=dp), PARAMETER :: rhoi = 913._dp
      REAL(KIND=dp), PARAMETER :: rhos = 330._dp
#else
      REAL(KIND=dp), PARAMETER :: rhoi = 910._dp
#endif
      REAL(KIND=dp), PARAMETER :: htci = 2009._dp
      REAL(KIND=dp), PARAMETER :: tcondi = 2.1_dp*year
      REAL(KIND=dp), PARAMETER :: tdiffi = tcondi/(rhoi*htci)
      REAL(KIND=dp), PARAMETER :: tlat = 3.35e5_dp
      INTEGER, PARAMETER :: gn  = 3
      INTEGER, PARAMETER :: gnp1 = gn + 1
      INTEGER, PARAMETER :: gnp2 = gn + 2
      INTEGER, PARAMETER :: gnm1 = gn - 1
      INTEGER, PARAMETER :: gnm2 = gn - 2
      INTEGER, PARAMETER :: exponent = gnm1/2
      INTEGER, PARAMETER :: derivexp = (gn-3)/2
      INTEGER, PARAMETER :: strnhtexp = gnp1/2
      REAL(KIND=dp), PARAMETER :: strinvexp = r3     ! (n-1)/2n
      REAL(KIND=dp), PARAMETER :: Rgas  = 8.314_dp
      REAL(KIND=dp), PARAMETER :: Qactwarm = 139000._dp
      REAL(KIND=dp), PARAMETER :: Qactcold = 60000._dp
      REAL(KIND=dp), PARAMETER :: Qactiso = 60700._dp
      REAL(KIND=dp), PARAMETER :: Bowarm = 5.47e10_dp*c3
      REAL(KIND=dp), PARAMETER :: Bocold = 1.14e-5_dp*c3
      REAL(KIND=dp), PARAMETER :: Boiso = 4.0562e-5_dp*c3
      REAL(KIND=dp), PARAMETER :: tnoice = 260.0_dp
      REAL(KIND=dp), PARAMETER :: glen_min = 208.15_dp
      REAL(KIND=dp), PARAMETER :: glen_max = 273.15_dp
      REAL(KIND=dp), PARAMETER :: glen_inc = 0.01_dp

      ! Bed properties (modified SI - annum)
      REAL(KIND=dp), PARAMETER :: rhoe = 3300._dp
!      REAL(KIND=dp), PARAMETER :: tau = 5000._dp
      REAL(KIND=dp), PARAMETER :: tau = 3000._dp
      REAL(KIND=dp), PARAMETER :: Hbed = 2000._dp

      ! Water properties (modified SI - annum)
#if defined uvic_embm
      REAL(KIND=dp), PARAMETER :: rhow = 1035._dp
#else
      REAL(KIND=dp), PARAMETER :: rhow = 1028._dp
#endif
      REAL(KIND=dp), PARAMETER :: htcw = 4183._dp
      REAL(KIND=dp), PARAMETER :: tcondw = 1._dp*year
      REAL(KIND=dp), PARAMETER :: tdiffw = tcondw/(rhow*htcw)
      REAL(KIND=dp), PARAMETER :: iceberg = c1 - rhoi/rhow

      ! Mass balance model constants
#if defined uvic_embm
      REAL(KIND=dp), PARAMETER :: lapse = .0065_dp      ! Deg C/m
#else
      REAL(KIND=dp), PARAMETER :: lapse = .0075_dp      ! Deg C/m
#endif
      REAL(KIND=dp), PARAMETER :: plapse = 1000.0_dp   ! m
      REAL(KIND=dp), PARAMETER :: ablapse = 0.002_dp   ! (m/a) m^{-1}
      REAL(KIND=dp) :: kcalf,expcalf                   ! Calving parameters

      ! Define second order `constants'
      !   k(T) = 9.828*exp(-5.7d-3 T)
      !   dkdT(T) = -0.05602 exp(-5.7d-3 T)
      !   C(T) = 152.5 + 7.122 T
      REAL(KIND=dp), PARAMETER :: triplept = 273.15_dp
      REAL(KIND=dp), PARAMETER :: fptdep = 8.7e-4_dp   ! Deg C/metre of ice
      REAL(KIND=dp), PARAMETER :: kice1 = 9.828_dp
      REAL(KIND=dp), PARAMETER :: kice2 = -5.7e-3_dp
      REAL(KIND=dp), PARAMETER :: kice3 = kice1*kice2
      REAL(KIND=dp), PARAMETER :: cice1 = 152.5_dp
      REAL(KIND=dp), PARAMETER :: cice2 = 7.122_dp

      ! Define model zero, physical zero
      REAL(KIND=dp), PARAMETER :: zerop = 0.1_dp           ! physical zero
      REAL(KIND=dp), PARAMETER :: zerom = 1.e-15_dp        ! machine zero
      REAL(KIND=dp), PARAMETER :: fimp = c1                   ! TD implicitness
      REAL(KIND=dp), PARAMETER :: afimp = c1 - fimp        ! " implicitness
      REAL(KIND=dp), PARAMETER :: fimpd = r2
      REAL(KIND=dp), PARAMETER :: afimpd = c1 - fimpd

      ! Define vertical co-ordinate system for dynamics
      ! zeta = EXP(aa(hi-z)/H);   *Use negative sense*
      REAL(KIND=dp), PARAMETER :: zetatop = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: zeta

      ! Define vertical co-ordinate system for thermodynamics
      ! xi = (hi-z)/H;   *Use positive sense*
      REAL(KIND=dp), PARAMETER :: xitop = c0
      REAL(KIND=dp), PARAMETER :: xibot = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: xi

      ! Define vertical co-ordinate system for lower bed (unfrozen)
      ! xibc = (hg-Hbedf-z)/(Hbed-Hbedf);  *Use positive sense*
      REAL(KIND=dp), PARAMETER :: xibctop = c0
      REAL(KIND=dp), PARAMETER :: xibcbot = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: xibc

      ! Define vertical co-ordinate system for upper bed (frozen zone)
      ! xibhi = (hg-z)/Hbedf;    *Use positive sense*
      REAL(KIND=dp), PARAMETER :: xibftop = c0
      REAL(KIND=dp), PARAMETER :: xibfbot = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: xibf

      ! Grid transformation dimensions
      REAL(KIND=dp) :: delzeta,delxi,delxibc,delxibf
      REAL(KIND=dp) :: aa          ! Dynamical grid transformation parameter

!***************************************************************************
      ! Declare grid properties and input parameters
      INTEGER :: ntime,istart         ! Integration limits
      INTEGER :: idelt,ideltt,idelth,ideltg,ideltb         ! Time steps
      INTEGER :: ihydnest                                   ! Nested soln times
      REAL(KIND=dp) :: delt,deltt,delth,deltg,deltb        ! Time steps
      REAL(KIND=dp) :: hydnest                                   ! Nested soln times
      REAL(KIND=dp) :: themin,themax,phimin,phimax         ! Grid limits
      REAL(KIND=dp) :: delthed,delphid,deltheta,delphi     ! Grid resolution
      REAL(KIND=dp) :: dx,dy,dxy,dxdx,dydy,Biso            ! Grid parameters
      INTEGER, PARAMETER :: nglen_table = 6500
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: Btable   ! Glen parameter

!-----------------------------------------------------------------------
!     mask
!-----------------------------------------------------------------------

      INTEGER,ALLOCATABLE,DIMENSION(:,:) :: idomain
#if !defined uvic_embm
      INTEGER,ALLOCATABLE,DIMENSION(:,:) :: icemask
#endif

!-----------------------------------------------------------------------
!     geometry
!-----------------------------------------------------------------------

      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: lon      ! longitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: lat      ! latitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: area     ! Cell area (m^2)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: delx     ! Cell width  (m)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: delxhi   ! Interface width (m)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: sine     ! sine of latitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: sinehi   ! sine of inter. lat.
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: clat     ! colatitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: clathi   ! interface colat.
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: Bjvect   ! Bj (integration)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: Cjvect   ! Cj (integration)
#if !defined uvic_embm
      REAL(KIND=dp),ALLOCATABLE, DIMENSION(:,:) :: latitude  ! Cell latitude
#endif

!     REAL(KIND=dp), PARAMETER :: themin = 7.644234_dp    ! EISMINT tests
!     REAL(KIND=dp), PARAMETER :: themax = 7.644234_dp    ! Equatorial block
!     REAL(KIND=dp), PARAMETER :: phimin = -7.644234_dp
!     REAL(KIND=dp), PARAMETER :: phimax = 7.644234_dp
!     REAL(KIND=dp), PARAMETER :: delthed = .449661_dp    ! 50 km
!     REAL(KIND=dp), PARAMETER :: delphid = .449661_dp

!     REAL(KIND=dp), PARAMETER :: themin = 35._dp         ! Laurentide
!     REAL(KIND=dp), PARAMETER :: themax = 80._dp
!     REAL(KIND=dp), PARAMETER :: phimin = 195._dp
!     REAL(KIND=dp), PARAMETER :: phimax = 315._dp
!     REAL(KIND=dp), PARAMETER :: delthed = 1._dp         ! 1 degree
!     REAL(KIND=dp), PARAMETER :: delphid = 1._dp

!     REAL(KIND=dp), PARAMETER :: themin = 53.0_dp     ! Alaskan peninsula
!     REAL(KIND=dp), PARAMETER :: themax = 56.0_dp
!     REAL(KIND=dp), PARAMETER :: phimin = -163.5_dp
!     REAL(KIND=dp), PARAMETER :: phimax = -161._dp
!     REAL(KIND=dp), PARAMETER :: delthed = 1.0_dp/24.0_dp    ! 4.6125 km
!     REAL(KIND=dp), PARAMETER :: delphid = 1.0_dp/12.0_dp    ! 5.3715 km

!***************************************************************************
      ! Declare integer option flags, globally-accesible
      INTEGER :: igrid           ! Grid type  1/2/3
      INTEGER :: init            ! Initial field treatment 1/2/3
      INTEGER :: imass           ! Mass balance treatment  1/2/3/4/5/6
      INTEGER :: iclim           ! Climate data type/source  1/2/3/4/5/6
      INTEGER :: itherm          ! Thermodynamic treatment  0/1
      INTEGER :: itdcoup         ! Coupled dynamics-TD?  1/0
      INTEGER :: itdptvar        ! T-dept ice properties?  1/0
      INTEGER :: istrn           ! Strain heating discretization  0/1
      INTEGER :: idisc           ! Discretization for TD (Up/centred) 3/4/5
      INTEGER :: idyn            ! Dynamical treatment  1/2
      INTEGER :: islid           ! Sliding?  1/0
      INTEGER :: itrack          ! Particle tracking?  1/0
      INTEGER :: isilt           ! Englacial debris?  1/0
      INTEGER :: iheat           ! Geothermal heat treatment 1/2
      INTEGER :: ibedtd          ! Bed TD treatment   0/1/2
      INTEGER :: ibedmo          ! Bed TD solution method   1/2
      INTEGER :: isurf           ! Surface geology treatment  0/1/2
      INTEGER :: ibedr           ! Bed geology treatment  0/1
      INTEGER :: isubtop         ! Subgrid topo treatment  0/1
      INTEGER :: igingembre      ! Surface hydrological treatment  0/1/2
      INTEGER :: isurfhyd        ! Surface water flux treatment  1/2
      INTEGER :: irebound         ! Model isostatic compensation? 0/1
      INTEGER :: ishelf                 ! Model floating ice and calving? 0/1
      INTEGER :: ialbedo         ! Apply Susan Marshall albedo code? 0/1
      INTEGER :: itest           ! result output test 0/1

      ! Array dimensions
      INTEGER :: nlon,nlat,nvert        ! Grid dimensions
      INTEGER :: nlonp,nlatp,nvertp     ! Physical grid dimensions
      INTEGER :: nxi,nzeta              ! Transformed vertical grid dimensions
      INTEGER :: njac1,njac2                 ! Jacobian matrix dimensions
      INTEGER :: ncore                  ! Ice core climate proxy
      INTEGER :: nducks                 ! Particle tracers
      INTEGER :: ntrack                 ! r-t tracking variables
      INTEGER :: nsilt                  ! Geologic debris units
      INTEGER :: nsurf,nbedr            ! Surface, bedrock materials
      INTEGER :: nsedtype               ! Sediment `type' divisions
      INTEGER :: nvertb                 ! Crustal grid dimension
      INTEGER :: nxibc,nxibf            ! Lower, upper crust grid size
      INTEGER :: nbin                   ! Directional bins for striae
      INTEGER :: ntable                 ! Number of entries, PDD lookup table
      INTEGER :: npddamp                ! Cross-entries, PDD lookup table
      INTEGER :: nhyps                  ! Number of hypsometric bins
      INTEGER :: ndepw                  ! Dependencies in water sheet eqns
      INTEGER :: njacw                  ! Subglacial water unknowns
      INTEGER :: njw                    ! Subglacial water unknowns
      INTEGER :: nwsys                  ! Sparse matrix dimension, water sheet
      INTEGER :: ndepd                  ! Dependencies in dynamical eqns
      INTEGER :: njacd                  ! Dynamical system unknowns
      INTEGER :: ndsys                  ! Sparse matrix dimension, dynamics
      INTEGER :: ntsys                        ! Full ice/bed TD system size
      INTEGER :: ntfrz                        ! Vertical level of permafrost front
      INTEGER :: ntisos

      INTEGER :: itime                  ! current time

      REAL(KIND=dp) :: sealevel

! ... time
      REAL(KIND=dp) :: stime    ! start time
      REAL(KIND=dp) :: ctime    ! current time
      REAL(KIND=dp) :: etime    ! end time

! ... snapshot output
      INTEGER :: isnap          ! current number of snapshot output
      INTEGER :: nsnap          ! maximum number of snapshot output
      REAL(KIND=dp) :: dtsnap   ! snapshot output interval
      INTEGER :: idtsnap

! ... ice volume output
      INTEGER :: nvol           ! maximum number of ice volume output
      REAL(KIND=dp) :: dtvol    ! ice volume output interval
      INTEGER :: idtvol

! ... restart
      REAL(KIND=dp) :: dtrest   ! restart file output interval
      INTEGER :: idtrest

!=======================================================================
      END MODULE global_param
!=======================================================================
