!=======================================================================
!     Calculate current ice sheet fluxes over the entire grid
!     Fluxes computed at the cell interface, Crank-Nicholson weighting
!     on the explicit and implicit contributions from ice thickness

!     Save a great deal of information for use in the Jacobian
!     calculation.

!     *_e  : * on the east face of a cell
!     ub   : eastward basal sliding velocity (2D)
!     u    : eastward velocity (3D)
!     flux : eastward advective flux
!     Hice : ice thickness

!     N.B. General sign convention: eastward positive, southward
!          positive
!=======================================================================

      SUBROUTINE is_dyn_eflux(time,ifloating,tiso,Dcoeff,Hvimp,hivimp, &
                              ub_e,scale_e,u_e,flux_e,Hice_e,gradx_e,  &
                              grady_e,div_e,ganma_e,dbar_e)

      USE icetd_arrays
      USE shelf_arrays
      IMPLICIT NONE

      INTEGER,INTENT(IN) :: time
      INTEGER,INTENT(IN), DIMENSION(nlon,nlat) :: ifloating
      REAL(KIND=dp),INTENT(IN) :: tiso,Dcoeff
      REAL(KIND=dp),INTENT(IN),DIMENSION(ndsys)      :: Hvimp,hivimp
      REAL(KIND=dp),INTENT(IN),DIMENSION(nlon-1,nlatp) :: ub_e
      REAL(KIND=dp),INTENT(INOUT),DIMENSION(nlon-1,nlatp) :: scale_e
      REAL(KIND=dp),INTENT(INOUT),DIMENSION(nvertp,nlon-1,nlatp) :: u_e
      REAL(KIND=dp),INTENT(OUT),DIMENSION(nlon-1,nlatp) :: flux_e,&
                      Hice_e,gradx_e,grady_e,div_e,ganma_e,dbar_e

      ! Local variables
      INTEGER :: i,j,jm,jp,k,kz,ktabrt
      REAL(KIND=dp) :: zsumrt,zetacur,urtcur,Bintrt,hi_e
      REAL(KIND=dp),DIMENSION(nvert) :: Bzrt

      REAL(KIND=dp),EXTERNAL :: B

!-----------------------------------------------------------------------
!     Loop over the interior (physical) model grid.
!     Calculate eastern (rt) and northern (hi) interface fluxes
!-----------------------------------------------------------------------

      DO j=2,nlat-1
        jm = j-1
        jp = j+1

long:   DO i=2,nlon-1
          k = i + nlon*(j-1)

!-----------------------------------------------------------------------
!         ice thickness at the east cell interface: Hice(i+1/2,j)
!-----------------------------------------------------------------------

          Hice_e(i,jm) = (Hvimp(k)+Hvimp(k+1))/c2

!-----------------------------------------------------------------------
!         jump to next point when no ice flux available
!-----------------------------------------------------------------------

          IF (Hice_e(i,jm) < zerop) THEN
            flux_e(i,jm) = c0
            CYCLE long
          END IF

!-----------------------------------------------------------------------
!         if both floating, no vertical shear flux, i.e., basal sliding
!         only
!-----------------------------------------------------------------------

          IF (ishelf > 0) THEN   ! floating ice option
            IF ((ifloating(i,j) == 1) .AND. (ifloatrt(i,jm) == 1)) THEN
              flux_e (i,jm) = -ub_e(i,jm)*Hice_e(i,jm)
              scale_e(i,jm) = c0
              CYCLE long
            END IF
          END IF

!-----------------------------------------------------------------------
!         gradient in  x direction: 1/sin(theta)*(d_hi/d_ramuda)i+1/2,j
!-----------------------------------------------------------------------

          gradx_e(i,jm) = (hivimp(k+1) - hivimp(k))/(sine(j)*delphi)

!-----------------------------------------------------------------------
!         gradient in -y direction: (d_hi/d_theta)i+1/2,j
!-----------------------------------------------------------------------

          IF (((j == 2).OR.(j == nlat-1)).AND.(i == nlon-1)) THEN
            grady_e(i,jm) = c0
          ELSE
            grady_e(i,jm) = - ((hivimp(k+nlon+1) + hivimp(k+nlon)) -   &
                               (hivimp(k-nlon+1) + hivimp(k-nlon)))    &
                            / (c4*deltheta)
          END IF

!-----------------------------------------------------------------------
!         divergence
!-----------------------------------------------------------------------

          div_e(i,jm) = gradx_e(i,jm)**2 + grady_e(i,jm)**2

!-----------------------------------------------------------------------
!         ((ganma)i+1/2,j)**(n-1)
!-----------------------------------------------------------------------

          ganma_e(i,jm) = (SQRT(div_e(i,jm)))**(gn-1)

!-----------------------------------------------------------------------
!         Sheet Fluxes out of control volume
!         out:  fluxhi, fluxlt (i.e. positive fluxhi = out) (follow slopes)
!         in:   fluxlo, fluxrt

!         Numerical integration over ice thickness to calculate
!           velocities and fluxes;
!         use scaled vertical co-ordinate zeta=exp(a(hi-z)/Hice), with
!           zeta=exp(a) at the bed and zeta=1 at the surface;
!         delzeta(nvert) is fixed, as defined in global_param

!         N.B. Flux sign convention: counter-intuitive (following the
!                slope), such that velocity signs need reversed
!-----------------------------------------------------------------------

          IF ((itherm == 0) .OR. (itdcoup ==0)) THEN

!-----------------------------------------------------------------------
!           isothermal treatment
!-----------------------------------------------------------------------

            ! column ice stiffness Bbar
            dbar_e(i,jm) = Dcoeff*Biso/DBLE(gnp2)

            ! westward flux at east cell interface
            flux_e(i,jm) = sine(j)*(-ub_e(i,jm)*Hice_e(i,jm)           &
           +dbar_e(i,jm)*ganma_e(i,jm)*gradx_e(i,jm)*Hice_e(i,jm)**gnp2)

            IF (ctime == etime) THEN   ! for output

              ! velocities
              hi_e = (hivimp(k)+hivimp(k+1))/c2
              u_e(1,i,jm) = ub_e(i,jm)
              DO kz=1,nvert
                zetacur = (zeta(kz+1)+zeta(kz))/c2
                u_e(kz+1,i,jm) = u_e(1,i,jm)                           &
                             + dbar_e(i,jm)*DBLE(gnp2)/DBLE(gnp1)      &
                               *ganma_e(i,jm)*gradx_e(i,jm)            &
                               *((Hice_e(i,jm)*LOG(zetacur)/aa)**gnp1  &
                                 -Hice_e(i,jm)**gnp1)
              END DO
            END IF

          ELSE

!-----------------------------------------------------------------------
!           polythermal treatment
!-----------------------------------------------------------------------

            ! First integration: velocities
            u_e(1,i,jm) = ub_e(i,jm)
            Bintrt = c0
            DO kz=1,nvert
              zetacur = (zeta(kz+1)+zeta(kz))/c2
              Bzrt(kz) = B(tp(kz,i,jm))*1.67_dp
              Bintrt = Bintrt &
                     - LOG(zetacur)**gn*Bzrt(kz)*delzeta/zetacur
              u_e(kz+1,i,jm) = u_e(1,i,jm) - Dcoeff*gradx_e(i,jm) &
                          *(Hice_e(i,jm)/aa)**gnp1*ganma_e(i,jm)*Bintrt
            END DO

            ! Second integration: fluxes
            ! Take linear average of variables in cell centre
            zsumrt = c0
            DO kz=1,nvert
              zetacur = (zeta(kz+1)+zeta(kz))/c2
              urtcur = (u_e(kz,i,jm)+u_e(kz+1,i,jm))/c2
              zsumrt = zsumrt + delzeta/zetacur*urtcur
            END DO

            flux_e(i,jm) = sine(j)*Hice_e(i,jm)*(-ub_e(i,jm) + zsumrt/aa)

            ! Vertically-integrated ice stiffness coefficient
            dbar_e(i,jm) = Bzrt(3)*Dcoeff/gnp2*aa**gnp2

          END IF    ! End itdcoup if block

        END DO long   ! End longitude loop
      END DO          ! End latitude loop

      u_e(1:nvertp,1,1:nlatp) = u_e(1:nvertp,nlon-1,1:nlatp)
      scale_e(1,1:nlatp) = scale_e(nlon-1,1:nlatp)
      flux_e (1,1:nlatp) = flux_e (nlon-1,1:nlatp)
      Hice_e (1,1:nlatp) = Hice_e (nlon-1,1:nlatp)
      gradx_e(1,1:nlatp) = gradx_e(nlon-1,1:nlatp)
      grady_e(1,1:nlatp) = grady_e(nlon-1,1:nlatp)
      div_e  (1,1:nlatp) = div_e  (nlon-1,1:nlatp)
      dbar_e (1,1:nlatp) = dbar_e (nlon-1,1:nlatp)
      ganma_e(1,1:nlatp) = ganma_e(nlon-1,1:nlatp)

!=======================================================================
      END SUBROUTINE is_dyn_eflux
!=======================================================================
