      SUBROUTINE is_PCHEZ(N,X,F,D,SPLINE,WK,LWK,IERR)
!***BEGIN PROLOGUE  PCHEZ
!***DATE WRITTEN   870821   (YYMMDD)
!***REVISION DATE  870908   (YYMMDD)
!***CATEGORY NO.  E1B
!***KEYWORDS  CUBIC HERMITE MONOTONE INTERPOLATION, SPLINE
!             INTERPOLATION, EASY TO USE PIECEWISE CUBIC INTERPOLATION
!***AUTHOR  KAHANER, D.K., (NBS)
!             SCIENTIFIC COMPUTING DIVISION
!             NATIONAL BUREAU OF STANDARDS
!             GAITHERSBURG, MARYLAND 20899
!             (301) 975-3808
!***PURPOSE  Easy to use spline or cubic Hermite interpolation.
!***DESCRIPTION

!          PCHEZ:  Piecewise Cubic Interpolation, Easy to Use.

!     From the book "Numerical Methods and Software"
!          by  D. Kahaner, C. Moler, S. Nash
!               Prentice Hall 1988

!     Sets derivatives for spline (two continuous derivatives) or
!     Hermite cubic (one continuous derivative) interpolation.
!     Spline interpolation is smoother, but may not "look" right if the
!     data contains both "steep" and "flat" sections.  Hermite cubics
!     can produce a "visually pleasing" and monotone interpolant to
!     monotone data. This is an easy to use driver for the routines
!     by F. N. Fritsch in reference (4) below. Various boundary
!     conditions are set to default values by PCHEZ. Many other choices
!     are available in the subroutines PCHIC, PCHIM and PCHSP.

!     Use PCHEV to evaluate the resulting function and its derivative.

! ----------------------------------------------------------------------

!  Calling sequence:   CALL  PCHEZ (N, X, F, D, SPLINE, WK, LWK, IERR)

!     INTEGER  N, IERR,  LWK
!     REAL  X(N), F(N), D(N), WK(*)
!     LOGICAL SPLINE

!   Parameters:

!     N -- (input) number of data points.  (Error return if N.LT.2 .)
!           If N=2, simply does linear interpolation.

!     X -- (input) real array of independent variable values.  The
!           elements of X must be strictly increasing:
!                X(I-1) .LT. X(I),  I = 2(1)N.
!           (Error return if not.)

!     F -- (input) real array of dependent variable values to be inter-
!           polated.  F(I) is value corresponding to X(I).

!     D -- (output) real array of derivative values at the data points.

!     SPLINE -- (input) logical variable to specify if the interpolant
!           is to be a spline with two continuous derivaties
!           (set SPLINE=.TRUE.) or a Hermite cubic interpolant with one
!           continuous derivative (set SPLINE=.FALSE.).
!        Note: If SPLINE=.TRUE. the interpolating spline satisfies the
!           default "not-a-knot" boundary condition, with a continuous
!           third derivative at X(2) and X(N-1). See reference (3).
!              If SPLINE=.FALSE. the interpolating Hermite cubic will be
!           monotone if the input data is monotone. Boundary conditions are
!           computed from the derivative of a local quadratic unless this
!           alters monotonicity.

!     WK -- (scratch) real work array, which must be declared by the calling
!           program to be at least 2*N if SPLINE is .TRUE. and not used
!           otherwise.

!     LWK -- (input) length of work array WK. (Error return if
!           LWK.LT.2*N and SPLINE is .TRUE., not checked otherwise.)

!     IERR -- (output) error flag.
!           Normal return:
!              IERR = 0  (no errors).
!           Warning error:
!              IERR.GT.0  (can only occur when SPLINE=.FALSE.) means that
!                 IERR switches in the direction of monotonicity were detected.
!                 When SPLINE=.FALSE.,  PCHEZ guarantees that if the input
!                 data is monotone, the interpolant will be too. This warning
!                 is to alert you to the fact that the input data was not
!                 monotone.
!           "Recoverable" errors:
!              IERR = -1  if N.LT.2 .
!              IERR = -3  if the X-array is not strictly increasing.
!              IERR = -7  if LWK is less than 2*N and SPLINE is .TRUE.
!             (The D-array has not been changed in any of these cases.)
!               NOTE:  The above errors are checked in the order listed,
!                   and following arguments have **NOT** been validated.

! ----------------------------------------------------------------------
!***REFERENCES  1. F.N.FRITSCH AND R.E.CARLSON, 'MONOTONE PIECEWISE
!                 CUBIC INTERPOLATION,' SIAM J.NUMER.ANAL. 17, 2 (APRIL
!                 1980), 238-246.
!               2. F.N.FRITSCH AND J.BUTLAND, 'A METHOD FOR CONSTRUCTING
!                 LOCAL MONOTONE PIECEWISE CUBIC INTERPOLANTS,' LLNL
!                 PREPRINT UCRL-87559 (APRIL 1982).
!               3. CARL DE BOOR, A PRACTICAL GUIDE TO SPLINES, SPRINGER-
!                 VERLAG (NEW YORK, 1978).  (ESP. CHAPTER IV, PP.49-62.)
!               4. F.N.FRITSCH, 'PIECEWISE CUBIC HERMITE INTERPOLATION
!                 PACKAGE, FINAL SPECIFICATIONS', LAWRENCE LIVERMORE
!                 NATIONAL LABORATORY, COMPUTER DOCUMENTATION UCID-30194,
!                 AUGUST 1982.
!***ROUTINES CALLED  PCHIM,PCHSP
!***END PROLOGUE  PCHEZ
      INTEGER  N, LWK, IERR
      REAL  X(N), F(N), D(N), WK(LWK)
      LOGICAL SPLINE

!  DECLARE LOCAL VARIABLES.

      INTEGER IC(2), INCFD
      REAL  VC(2)
      DATA IC(1) /0/
      DATA IC(2) /0/
      DATA INCFD /1/

!***FIRST EXECUTABLE STATEMENT  PCHEZ

      IF ( SPLINE ) THEN
        CALL  IS_PCHSP (IC, VC, N, X, F, D, INCFD, WK, LWK, IERR)
      ELSE
        CALL  IS_PCHIM (N, X, F, D, INCFD, IERR)
      ENDIF

!  ERROR CONDITIONS ALREADY CHECKED IN PCHSP OR PCHIM

      RETURN
!------------- LAST LINE OF PCHEZ FOLLOWS ------------------------------
      END
      SUBROUTINE IS_PCHIM(N,X,F,D,INCFD,IERR)
!***BEGIN PROLOGUE  PCHIM
!     THIS PROLOGUE HAS BEEN REMOVED FOR REASONS OF SPACE
!     FOR A COMPLETE COPY OF THIS ROUTINE CONTACT THE AUTHORS
!     From the book "Numerical Methods and Software"
!          by  D. Kahaner, C. Moler, S. Nash
!               Prentice Hall 1988
!***END PROLOGUE  PCHIM

!  DECLARE ARGUMENTS.

      INTEGER  N, INCFD, IERR
      REAL  X(N), F(INCFD,N), D(INCFD,N)

!  DECLARE LOCAL VARIABLES.

      INTEGER  I, NLESS1
      REAL  DEL1, DEL2, DMAX, DMIN, DRAT1, DRAT2, DSAVE,
     *      H1, H2, HSUM, HSUMT3, THREE, W1, W2, ZERO
      REAL  IS_PCHST
      DATA  ZERO /0./,  THREE /3./

!  VALIDITY-CHECK ARGUMENTS.

!***FIRST EXECUTABLE STATEMENT  PCHIM
      IF ( N.LT.2 )  GO TO 5001
      IF ( INCFD.LT.1 )  GO TO 5002
      DO 1  I = 2, N
         IF ( X(I).LE.X(I-1) )  GO TO 5003
    1 CONTINUE

!  FUNCTION DEFINITION IS OK, GO ON.

      IERR = 0
      NLESS1 = N - 1
      H1 = X(2) - X(1)
      DEL1 = (F(1,2) - F(1,1))/H1
      DSAVE = DEL1

!  SPECIAL CASE N=2 -- USE LINEAR INTERPOLATION.

      IF (NLESS1 .GT. 1)  GO TO 10
      D(1,1) = DEL1
      D(1,N) = DEL1
      GO TO 5000

!  NORMAL CASE  (N .GE. 3).

   10 CONTINUE
      H2 = X(3) - X(2)
      DEL2 = (F(1,3) - F(1,2))/H2

!  SET D(1) VIA NON-CENTERED THREE-POINT FORMULA, ADJUSTED TO BE
!     SHAPE-PRESERVING.

      HSUM = H1 + H2
      W1 = (H1 + HSUM)/HSUM
      W2 = -H1/HSUM
      D(1,1) = W1*DEL1 + W2*DEL2
      IF ( IS_PCHST(D(1,1),DEL1) .LE. ZERO)  THEN
         D(1,1) = ZERO
      ELSE IF ( IS_PCHST(DEL1,DEL2) .LT. ZERO)  THEN
!        NEED DO THIS CHECK ONLY IF MONOTONICITY SWITCHES.
         DMAX = THREE*DEL1
         IF (ABS(D(1,1)) .GT. ABS(DMAX))  D(1,1) = DMAX
      ENDIF

!  LOOP THROUGH INTERIOR POINTS.

      DO 50  I = 2, NLESS1
         IF (I .EQ. 2)  GO TO 40

         H1 = H2
         H2 = X(I+1) - X(I)
         HSUM = H1 + H2
         DEL1 = DEL2
         DEL2 = (F(1,I+1) - F(1,I))/H2
   40    CONTINUE

!        SET D(I)=0 UNLESS DATA ARE STRICTLY MONOTONIC.

         D(1,I) = ZERO
         IF ( IS_PCHST(DEL1,DEL2) )  42, 41, 45

!        COUNT NUMBER OF CHANGES IN DIRECTION OF MONOTONICITY.

   41    CONTINUE
         IF (DEL2 .EQ. ZERO)  GO TO 50
         IF ( IS_PCHST(DSAVE,DEL2) .LT. ZERO)  IERR = IERR + 1
         DSAVE = DEL2
         GO TO 50

   42    CONTINUE
         IERR = IERR + 1
         DSAVE = DEL2
         GO TO 50

!        USE BRODLIE MODIFICATION OF BUTLAND FORMULA.

   45    CONTINUE
         HSUMT3 = HSUM+HSUM+HSUM
         W1 = (HSUM + H1)/HSUMT3
         W2 = (HSUM + H2)/HSUMT3
         DMAX = AMAX1( ABS(DEL1), ABS(DEL2) )
         DMIN = AMIN1( ABS(DEL1), ABS(DEL2) )
         DRAT1 = DEL1/DMAX
         DRAT2 = DEL2/DMAX
         D(1,I) = DMIN/(W1*DRAT1 + W2*DRAT2)

   50 CONTINUE

!  SET D(N) VIA NON-CENTERED THREE-POINT FORMULA, ADJUSTED TO BE
!     SHAPE-PRESERVING.

      W1 = -H2/HSUM
      W2 = (H2 + HSUM)/HSUM
      D(1,N) = W1*DEL1 + W2*DEL2
      IF ( IS_PCHST(D(1,N),DEL2) .LE. ZERO)  THEN
         D(1,N) = ZERO
      ELSE IF ( IS_PCHST(DEL1,DEL2) .LT. ZERO)  THEN
!        NEED DO THIS CHECK ONLY IF MONOTONICITY SWITCHES.
         DMAX = THREE*DEL2
         IF (ABS(D(1,N)) .GT. ABS(DMAX))  D(1,N) = DMAX
      ENDIF

!  NORMAL RETURN.

 5000 CONTINUE
      RETURN

!  ERROR RETURNS.

 5001 CONTINUE
!     N.LT.2 RETURN.
      IERR = -1
      CALL IS_XERROR ('PCHIM -- NUMBER OF DATA POINTS LESS THAN TWO'
     *           , 44, IERR, 1)
      RETURN

 5002 CONTINUE
!     INCFD.LT.1 RETURN.
      IERR = -2
      CALL IS_XERROR ('PCHIM -- INCREMENT LESS THAN ONE'
     *           , 32, IERR, 1)
      RETURN

 5003 CONTINUE
!     X-ARRAY NOT STRICTLY INCREASING.
      IERR = -3
      CALL IS_XERROR ('PCHIM -- X-ARRAY NOT STRICTLY INCREASING'
     *           , 40, IERR, 1)
      RETURN
!------------- LAST LINE OF PCHIM FOLLOWS ------------------------------
      END
      REAL FUNCTION IS_PCHST(ARG1,ARG2)
!***BEGIN PROLOGUE  PCHST
!***REFER TO  PCHCE,PCHCI,PCHCS,PCHIM
!***END PROLOGUE  PCHST
      REAL  ARG1, ARG2

!  DECLARE LOCAL VARIABLES.

      REAL  ONE, ZERO
      DATA  ZERO /0./,  ONE /1./

!  PERFORM THE TEST.

!***FIRST EXECUTABLE STATEMENT  PCHST
      IS_PCHST = SIGN(ONE,ARG1) * SIGN(ONE,ARG2)
      IF ((ARG1.EQ.ZERO) .OR. (ARG2.EQ.ZERO))  IS_PCHST = ZERO

      RETURN
!------------- LAST LINE OF PCHST FOLLOWS ------------------------------
      END
      SUBROUTINE IS_PCHSP(IC,VC,N,X,F,D,INCFD,WK,NWK,IERR)
!***BEGIN PROLOGUE  PCHSP
!     THIS PROLOGUE HAS BEEN REMOVED FOR REASONS OF SPACE
!     FOR A COMPLETE COPY OF THIS ROUTINE CONTACT THE AUTHORS
!     From the book "Numerical Methods and Software"
!          by  D. Kahaner, C. Moler, S. Nash
!               Prentice Hall 1988
!***END PROLOGUE  PCHSP

!  DECLARE ARGUMENTS.

      INTEGER  IC(2), N, INCFD, NWK, IERR
      REAL  VC(2), X(N), F(INCFD,N), D(INCFD,N), WK(2,N)

!  DECLARE LOCAL VARIABLES.

      INTEGER  IBEG, IEND, INDEX, J, NM1
      REAL  G, HALF, ONE, STEMP(3), THREE, TWO, XTEMP(4), ZERO
      REAL  IS_PCHDF

      DATA  ZERO /0./,  HALF /0.5/,  ONE /1./,  TWO /2./,  THREE /3./

!  VALIDITY-CHECK ARGUMENTS.

!***FIRST EXECUTABLE STATEMENT  PCHSP
      IF ( N.LT.2 )  GO TO 5001
      IF ( INCFD.LT.1 )  GO TO 5002
      DO 1  J = 2, N
         IF ( X(J).LE.X(J-1) )  GO TO 5003
    1 CONTINUE

      IBEG = IC(1)
      IEND = IC(2)
      IERR = 0
      IF ( (IBEG.LT.0).OR.(IBEG.GT.4) )  IERR = IERR - 1
      IF ( (IEND.LT.0).OR.(IEND.GT.4) )  IERR = IERR - 2
      IF ( IERR.LT.0 )  GO TO 5004

!  FUNCTION DEFINITION IS OK -- GO ON.

      IF ( NWK .LT. 2*N )  GO TO 5007

!  COMPUTE FIRST DIFFERENCES OF X SEQUENCE AND STORE IN WK(1,.). ALSO,
!  COMPUTE FIRST DIVIDED DIFFERENCE OF DATA AND STORE IN WK(2,.).
      DO 5  J=2,N
         WK(1,J) = X(J) - X(J-1)
         WK(2,J) = (F(1,J) - F(1,J-1))/WK(1,J)
    5 CONTINUE

!  SET TO DEFAULT BOUNDARY CONDITIONS IF N IS TOO SMALL.

      IF ( IBEG.GT.N )  IBEG = 0
      IF ( IEND.GT.N )  IEND = 0

!  SET UP FOR BOUNDARY CONDITIONS.

      IF ( (IBEG.EQ.1).OR.(IBEG.EQ.2) )  THEN
         D(1,1) = VC(1)
      ELSE IF (IBEG .GT. 2)  THEN
!        PICK UP FIRST IBEG POINTS, IN REVERSE ORDER.
         DO 10  J = 1, IBEG
            INDEX = IBEG-J+1
!           INDEX RUNS FROM IBEG DOWN TO 1.
            XTEMP(J) = X(INDEX)
            IF (J .LT. IBEG)  STEMP(J) = WK(2,INDEX)
   10    CONTINUE
!                 --------------------------------
         D(1,1) = IS_PCHDF (IBEG, XTEMP, STEMP, IERR)
!                 --------------------------------
         IF (IERR .NE. 0)  GO TO 5009
         IBEG = 1
      ENDIF

      IF ( (IEND.EQ.1).OR.(IEND.EQ.2) )  THEN
         D(1,N) = VC(2)
      ELSE IF (IEND .GT. 2)  THEN
!        PICK UP LAST IEND POINTS.
         DO 15  J = 1, IEND
            INDEX = N-IEND+J
!           INDEX RUNS FROM N+1-IEND UP TO N.
            XTEMP(J) = X(INDEX)
            IF (J .LT. IEND)  STEMP(J) = WK(2,INDEX+1)
   15    CONTINUE
!                 --------------------------------
         D(1,N) = IS_PCHDF (IEND, XTEMP, STEMP, IERR)
!                 --------------------------------
         IF (IERR .NE. 0)  GO TO 5009
         IEND = 1
      ENDIF

! --------------------( BEGIN CODING FROM CUBSPL )--------------------

!  **** A TRIDIAGONAL LINEAR SYSTEM FOR THE UNKNOWN SLOPES S(J) OF
!  F  AT X(J), J=1,...,N, IS GENERATED AND THEN SOLVED BY GAUSS ELIM-
!  INATION, WITH S(J) ENDING UP IN D(1,J), ALL J.
!     WK(1,.) AND WK(2,.) ARE USED FOR TEMPORARY STORAGE.

!  CONSTRUCT FIRST EQUATION FROM FIRST BOUNDARY CONDITION, OF THE FORM
!             WK(2,1)*S(1) + WK(1,1)*S(2) = D(1,1)

      IF (IBEG .EQ. 0)  THEN
         IF (N .EQ. 2)  THEN
!           NO CONDITION AT LEFT END AND N = 2.
            WK(2,1) = ONE
            WK(1,1) = ONE
            D(1,1) = TWO*WK(2,2)
         ELSE
!           NOT-A-KNOT CONDITION AT LEFT END AND N .GT. 2.
            WK(2,1) = WK(1,3)
            WK(1,1) = WK(1,2) + WK(1,3)
            D(1,1) =((WK(1,2) + TWO*WK(1,1))*WK(2,2)*WK(1,3)
     *                        + WK(1,2)**2*WK(2,3)) / WK(1,1)
         ENDIF
      ELSE IF (IBEG .EQ. 1)  THEN
!        SLOPE PRESCRIBED AT LEFT END.
         WK(2,1) = ONE
         WK(1,1) = ZERO
      ELSE
!        SECOND DERIVATIVE PRESCRIBED AT LEFT END.
         WK(2,1) = TWO
         WK(1,1) = ONE
         D(1,1) = THREE*WK(2,2) - HALF*WK(1,2)*D(1,1)
      ENDIF

!  IF THERE ARE INTERIOR KNOTS, GENERATE THE CORRESPONDING EQUATIONS AND
!  CARRY OUT THE FORWARD PASS OF GAUSS ELIMINATION, AFTER WHICH THE J-TH
!  EQUATION READS    WK(2,J)*S(J) + WK(1,J)*S(J+1) = D(1,J).

      NM1 = N-1
      IF (NM1 .GT. 1)  THEN
         DO 20 J=2,NM1
            IF (WK(2,J-1) .EQ. ZERO)  GO TO 5008
            G = -WK(1,J+1)/WK(2,J-1)
            D(1,J) = G*D(1,J-1)
     *                  + THREE*(WK(1,J)*WK(2,J+1) + WK(1,J+1)*WK(2,J))
            WK(2,J) = G*WK(1,J-1) + TWO*(WK(1,J) + WK(1,J+1))
   20    CONTINUE
      ENDIF

!  CONSTRUCT LAST EQUATION FROM SECOND BOUNDARY CONDITION, OF THE FORM
!           (-G*WK(2,N-1))*S(N-1) + WK(2,N)*S(N) = D(1,N)

!     IF SLOPE IS PRESCRIBED AT RIGHT END, ONE CAN GO DIRECTLY TO BACK-
!     SUBSTITUTION, SINCE ARRAYS HAPPEN TO BE SET UP JUST RIGHT FOR IT
!     AT THIS POINT.
      IF (IEND .EQ. 1)  GO TO 30

      IF (IEND .EQ. 0)  THEN
         IF (N.EQ.2 .AND. IBEG.EQ.0)  THEN
!           NOT-A-KNOT AT RIGHT ENDPOINT AND AT LEFT ENDPOINT AND N = 2.
            D(1,2) = WK(2,2)
            GO TO 30
         ELSE IF ((N.EQ.2) .OR. (N.EQ.3 .AND. IBEG.EQ.0))  THEN
!           EITHER (N=3 AND NOT-A-KNOT ALSO AT LEFT) OR (N=2 AND *NOT*
!           NOT-A-KNOT AT LEFT END POINT).
            D(1,N) = TWO*WK(2,N)
            WK(2,N) = ONE
            IF (WK(2,N-1) .EQ. ZERO)  GO TO 5008
            G = -ONE/WK(2,N-1)
         ELSE
!           NOT-A-KNOT AND N .GE. 3, AND EITHER N.GT.3 OR  ALSO NOT-A-
!           KNOT AT LEFT END POINT.
            G = WK(1,N-1) + WK(1,N)
!           DO NOT NEED TO CHECK FOLLOWING DENOMINATORS (X-DIFFERENCES).
            D(1,N) = ((WK(1,N)+TWO*G)*WK(2,N)*WK(1,N-1)
     *                  + WK(1,N)**2*(F(1,N-1)-F(1,N-2))/WK(1,N-1))/G
            IF (WK(2,N-1) .EQ. ZERO)  GO TO 5008
            G = -G/WK(2,N-1)
            WK(2,N) = WK(1,N-1)
         ENDIF
      ELSE
!        SECOND DERIVATIVE PRESCRIBED AT RIGHT ENDPOINT.
         D(1,N) = THREE*WK(2,N) + HALF*WK(1,N)*D(1,N)
         WK(2,N) = TWO
         IF (WK(2,N-1) .EQ. ZERO)  GO TO 5008
         G = -ONE/WK(2,N-1)
      ENDIF

!  COMPLETE FORWARD PASS OF GAUSS ELIMINATION.

      WK(2,N) = G*WK(1,N-1) + WK(2,N)
      IF (WK(2,N) .EQ. ZERO)   GO TO 5008
      D(1,N) = (G*D(1,N-1) + D(1,N))/WK(2,N)

!  CARRY OUT BACK SUBSTITUTION

   30 CONTINUE
      DO 40 J=NM1,1,-1
         IF (WK(2,J) .EQ. ZERO)  GO TO 5008
         D(1,J) = (D(1,J) - WK(1,J)*D(1,J+1))/WK(2,J)
   40 CONTINUE
! --------------------(  END  CODING FROM CUBSPL )--------------------

!  NORMAL RETURN.

      RETURN

!  ERROR RETURNS.

 5001 CONTINUE
!     N.LT.2 RETURN.
      IERR = -1
      CALL IS_XERROR ('PCHSP -- NUMBER OF DATA POINTS LESS THAN TWO'
     *           , 44, IERR, 1)
      RETURN

 5002 CONTINUE
!     INCFD.LT.1 RETURN.
      IERR = -2
      CALL IS_XERROR ('PCHSP -- INCREMENT LESS THAN ONE'
     *           , 32, IERR, 1)
      RETURN

 5003 CONTINUE
!     X-ARRAY NOT STRICTLY INCREASING.
      IERR = -3
      CALL IS_XERROR ('PCHSP -- X-ARRAY NOT STRICTLY INCREASING'
     *           , 40, IERR, 1)
      RETURN

 5004 CONTINUE
!     IC OUT OF RANGE RETURN.
      IERR = IERR - 3
      CALL IS_XERROR ('PCHSP -- IC OUT OF RANGE'
     *           , 24, IERR, 1)
      RETURN

 5007 CONTINUE
!     NWK TOO SMALL RETURN.
      IERR = -7
      CALL IS_XERROR ('PCHSP -- WORK ARRAY TOO SMALL'
     *           , 29, IERR, 1)
      RETURN

 5008 CONTINUE
!     SINGULAR SYSTEM.
!   *** THEORETICALLY, THIS CAN ONLY OCCUR IF SUCCESSIVE X-VALUES   ***
!   *** ARE EQUAL, WHICH SHOULD ALREADY HAVE BEEN CAUGHT (IERR=-3). ***
      IERR = -8
      CALL IS_XERROR ('PCHSP -- SINGULAR LINEAR SYSTEM'
     *           , 31, IERR, 1)
      RETURN

 5009 CONTINUE
!     ERROR RETURN FROM PCHDF.
!   *** THIS CASE SHOULD NEVER OCCUR ***
      IERR = -9
      CALL IS_XERROR ('PCHSP -- ERROR RETURN FROM IS_PCHDF'
     *           , 32, IERR, 1)
      RETURN
!------------- LAST LINE OF PCHSP FOLLOWS ------------------------------
      END
      REAL FUNCTION IS_PCHDF(K,X,S,IERR)
!***BEGIN PROLOGUE  PCHDF
!***REFER TO  PCHCE,PCHSP
!***END PROLOGUE  PCHDF
      INTEGER  K, IERR
      REAL  X(K), S(K)

!  DECLARE LOCAL VARIABLES.

      INTEGER  I, J
      REAL  VALUE, ZERO
      DATA  ZERO /0./

!  CHECK FOR LEGAL VALUE OF K.

!***FIRST EXECUTABLE STATEMENT  PCHDF
      IF (K .LT. 3)  GO TO 5001

!  COMPUTE COEFFICIENTS OF INTERPOLATING POLYNOMIAL.

      DO 10  J = 2, K-1
         DO 9  I = 1, K-J
            S(I) = (S(I+1)-S(I))/(X(I+J)-X(I))
    9    CONTINUE
   10 CONTINUE

!  EVALUATE DERIVATIVE AT X(K).

      VALUE = S(1)
      DO 20  I = 2, K-1
         VALUE = S(I) + VALUE*(X(K)-X(I))
   20 CONTINUE

!  NORMAL RETURN.

      IERR = 0
      IS_PCHDF = VALUE
      RETURN

!  ERROR RETURN.

 5001 CONTINUE
!     K.LT.3 RETURN.
      IERR = -1
      CALL IS_XERROR ('PCHDF -- K LESS THAN THREE'
     *           , 26, IERR, 1)
      IS_PCHDF = ZERO
      RETURN
!------------- LAST LINE OF PCHDF FOLLOWS ------------------------------
      END
