      program UVic_ESCM

!=======================================================================

!     UNIVERSITY OF VICTORIA EARTH SYSTEM CLIMATE MODEL

!     A climate model developed by researchers in the Climate Research
!     Group, in the School of Earth and Ocean Sciences, located at the
!     University of Victoria, Victoria, B.C., Canada.

!     Bug reports or questions:

!     Each section of code has a code author. Please direct questions
!     or problems to these authors

!     Many people have contributed to the development of this code
!     and attempts are made to indicate originators of code where
!     appropriate. Authorship here is meant to convey responsibility
!     for the current form of the code and does not necessarily
!     indicate that the author is responsible for all (or even
!     a major portion) of the code content.

!     Requirements:

!     Standard fortran 90 is used

!     Disclaimer:

!     The UVic Earth System Climate Model (UVic_ESCM) is a climate
!     modeling research tool developed at the University of Victoria.
!     Others may use it freely but we assume no responsibility for
!     problems or incorrect use. It is left to the user to ensure that
!     a particular configuration is working correctly.

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

!     This is the main driver. Integration time is divided into a number
!     of equal time segments and SBC are held fixed for each time
!     segment. When coupling, SBC are supplied each time segment (the
!     coupling period) and held fixed for that period.

!     based on code by: r. c. pacanowski and a. rosati
!=======================================================================

#include "param.h"
#if defined uvic_embm
# include "atm.h"
# include "cembm.h"
#endif
#include "coord.h"
#include "csbc.h"
#include "iounit.h"
#include "levind.h"
#include "mapsbc.h"
#if defined uvic_mom
# include "mw.h"
# include "task_on.h"
#endif
#include "scalar.h"
#include "switch.h"
#include "tmngr.h"
#if defined ubc_cidm
# include "ice.h"
# include "subgrid.h"
# include "calendar.h"

      real dtcidm

#endif
      print*, '== UNIVERSITY OF VICTORIA EARTH SYSTEM CLIMATE MODEL =='
      print*, '                                            '

!-----------------------------------------------------------------------
!     initialize i/o units
!-----------------------------------------------------------------------

      call ioinit

!-----------------------------------------------------------------------
!     setup file renaming
!-----------------------------------------------------------------------

      call file_names

!-----------------------------------------------------------------------
!     Initialize S.B.C. names and related items.
!-----------------------------------------------------------------------

      do n=1,numsbc
        sbcocn(:,:,n) = 0.0
        mapsbc(n) = n
      enddo

      m = 0
      m = m + 1
      itaux = m
      m = m + 1
      itauy = m
      m = m + 1
      ihflx = m
      m = m + 1
      ipme = m
#if defined uvic_embm
      m = m + 1
      iws = m
#endif
#if defined uvic_embm_astress && defined uvic_embm
      m = m + 1
      iwa = m
#else
      iwa = 0
#endif
#if defined uvic_embm_adv_q && defined uvic_embm
      m = m + 1
      iwx = m
      m = m + 1
      iwy = m
#else
      iwx = 0
      iwy = 0
#endif
#if defined shortwave
      m = m + 1
      ipsw = m
#else
      ipsw = 0
#endif
      m = m + 1
      isst = m
      m = m + 1
      isss = m
#if defined uvic_ice_evp && defined uvic_embm
      m = m + 1
      isu = m
      m = m + 1
      isv = m
      m = m + 1
      igu = m
      m = m + 1
      igv = m
#else
      isu = 0
      isv = 0
      igu = 0
      igv = 0
#endif

      if ( m .gt. numsbc) then
        print*, '==> Error: increase numsbc in csbc.h to ', m
        stop '=>driver'
      endif

!-----------------------------------------------------------------------
!     do the introductory ocean setup once per run
!-----------------------------------------------------------------------

      call setocn

#if defined ubc_cidm
      print*, ' option ubc_cidm is not ready for use in this version'
      stop
!-----------------------------------------------------------------------
!    do the introductory icesheet setup once per run
!-----------------------------------------------------------------------

      degrad = pi/180.
      do j=2,jmt-1
        phij = -90. - dytdeg(j)/2. + dytdeg(j)*float(j-1)
        area(j) = radius*radius*cos(phij*degrad)
     &   *(dxtdeg(1)*degrad)*(dytdeg(j)*degrad)
        if ((j.ge.jmint).and.(j.le.jmaxt)) then
          do m=1,mj
            jj = (j-jmint)*mj + m
            phijj = phij - dytdeg(j)/2. - dytdeg(j)/float(mj)/2.
     &            + dytdeg(j)/float(mj)*float(m)

            subarea(jj) = radius*radius*cos(phijj*degrad)
     &       *(dxtdeg(1)/float(mi)*degrad)*(dytdeg(j)/float(mj)*degrad)
          enddo
        endif
      enddo
      area(1)   = area(2)
      area(jmt) = area(jmt-1)
      if (jmint.eq.1) then
        do m=1,mj
          jj = (1-jmint)*mj + m
          subarea(jj) = subarea(jj+mj)
        enddo
      endif
      if (jmaxt.eq.jmt) then
        do m=1,mj
          jj = (jmt-jmint)*mj + m
          subarea(jj) = subarea(jj-mj)
        enddo
      endif

      call cidm_init (subhice, subelev, avg_calv, subaice, dtcidm)
#endif

!-----------------------------------------------------------------------
!     do the introductory atmosphere setup once per run
!-----------------------------------------------------------------------

      write (stdout,'(/a36/)') ' ==> Note: the atmos setup follows:'

!     "setatm" must do the following:
!       1) set up the atmospheric S.B.C. grid definition
!       2) define the atmosphere land/sea mask
!       3) set the atmosphere time step "dtatm" {seconds}

#if defined uvic_embm
      call setembm
#else
      call setatm
#endif
#if !defined uvic_embm

!     when the MOM S.B.C. come from a dataset, force the segment time
!     and atmospheric time step to one MOM time step. This will force
!     the number of segments to one and the number of time steps per
!     segment to represent the length of the run in days.

      dtatm = dtts
      segtim = dtts*secday
#endif

!-----------------------------------------------------------------------
!     compute the number of ocean time steps "numots" for this run and
!     the number of ocean time steps per ocean segment "ntspos".
!     compute the number of atmos time steps "numats" for this run and
!     the number of atmos time steps per atmos segment "ntspas".
!     divide the integration time "days" into "numseg" segments.
!     each will be length "segtim" days. Surface boundary conditions
!     are supplied every "segtim" days.
!-----------------------------------------------------------------------

      numots = nint(rundays/(dtts*secday))
      ntspos = nint(segtim/(dtts*secday))
      numats = nint(rundays/(dtatm*secday))
      ntspas = nint(segtim/(dtatm*secday))
      numseg = numots/ntspos

#if defined uvic_embm
      write (stdout,8800) rundays, numseg, segtim, ntspos, ntspas, dtts
     &,                   dtatm
#else
      write (stdout,8800) rundays, numseg, segtim, ntspos, dtts
#endif

#if !defined uvic_embm && !defined uvic_replacst
!     load the tracers (SST & SSS) for each row "j". (zero on land)
!     load from the MW if fully opened otherwise load from disk

      if (wide_open_mw) then
        do jrow=1,jmt
          do i=1,imt
            if (isst .ne. 0) sbcocn(i,jrow,isst) = t(i,1,jrow,1,taup1)
            if (isss .ne. 0) sbcocn(i,jrow,isss) = t(i,1,jrow,2,taup1)
          enddo
        enddo
      else
        do jrow=1,jmt
          if (isst .ne. 0) call getst (jrow, sbcocn(1,1,isst), 1)
          if (isss .ne. 0) call getst (jrow, sbcocn(1,1,isss), 2)
        enddo
      endif

#endif
!-----------------------------------------------------------------------
!     check for consistency in the S.B.C. setup
!-----------------------------------------------------------------------

      call chkcpl (dtatm, dtts)

#if defined ubc_cidm
!-----------------------------------------------------------------------
!     get inital conditions from the icesheet model
!-----------------------------------------------------------------------

      call cidm_sbc_allocate
      call cidm_sbc_atm (subhice, subelev)
      call cidm_sbc_ocn (avg_calv, avg_ltnt)
      call cidm_sbc_init

#endif
# if defined uvic_global_sums
!-----------------------------------------------------------------------
!     get global sums at the start of the run
!-----------------------------------------------------------------------

      dtoih = 0.
      call globalsum (1)

# endif
!-----------------------------------------------------------------------
!     S T A R T    S E G M E N T    L O O P
!-----------------------------------------------------------------------

      do n=1,numseg

!-----------------------------------------------------------------------
!       get the atmospheric S.B.C. from ocean
!-----------------------------------------------------------------------

        call gasbc

!-----------------------------------------------------------------------
!       call the atmospheric model once for each time step until one
!       segment of "segtim" days is complete. hold atmos S.B.C. fixed
!       during each segment and predict average S.B.C. for ocean
!-----------------------------------------------------------------------

        do loop=1,ntspas
#if defined uvic_embm
          call embm
# if defined ubc_cidm
#  if !defined uvic_mom
          call cidmout (snapts, timavgts, tsits, timunit
     &,                 expnam, relyr, stamp)
#  endif
!         integrate land-ice surface boundary conditions
          if (addflux)  call cidm_sbc_sum (subpsno, subprecip)
# endif
#else
          call atmos
#endif
        enddo

#if defined ubc_cidm
        if (mod(relyr,dtcidm) .lt. 1.e-6) then
          call cidm_sbc_avg
          call cidm_sbc_lice
          call cidm(0., dtcidm, subhice, subelev, avg_calv, subaice)
          call cidm_sbc_atm (subhice, subelev)
          call cidm_sbc_ocn (avg_calv, avg_ltnt)
          call cidm_sbc_init
        endif

#endif
#if defined uvic_mom
!-----------------------------------------------------------------------
!       get ocean S.B.C.s from the atmosphere
!-----------------------------------------------------------------------

        call gosbc

!-----------------------------------------------------------------------
!       call the ocean model once for each time step until one
!       segment of "segtim" days is complete. hold ocean S.B.C. fixed
!       during each segment and predict average S.B.C. for atmos
!-----------------------------------------------------------------------

        do loop=1,ntspos
          call mom
# if defined uvic_embm
          call embmout (1, imt, 1, jmt)
# endif
# if defined ubc_cidm
          call cidmout (snapts, timavgts, tsits, timunit
     &,                 expnam, relyr, stamp)
# endif
        enddo

#endif
# if defined uvic_global_sums
!-----------------------------------------------------------------------
!       write change in global sums for heat and fresh water
!-----------------------------------------------------------------------

        if (tsits) call globalsum (2)

# endif
      enddo

# if defined uvic_global_sums
!-----------------------------------------------------------------------
!     get global sums at the end of the run
!-----------------------------------------------------------------------

      call globalsum (3)

# endif
!-----------------------------------------------------------------------
!     E N D    S E G M E N T    L O O P
!-----------------------------------------------------------------------

      write (stdout,9300)

!     show files and close all units

      call showfiles
      call release_all

#if defined uvic_embm
8800  format (/,/,1x,'The model has been configured to run for'
     &,g14.7,' days in ',i10,' segments of ',g14.7,' days each.'
     &,/1x,'There will be ',i6,' ocean time steps per segment'
     &,' and ',i6,' atmosphere time steps per segment.'/
     &,/1x,' The ocean "dtts" =', g14.7, ' seconds'
     &,', and the atmosphere "dtatm" =', g14.7,' seconds'
     &,/,/)
#else
8800  format (/,/,1x,'The model has been configured to run for'
     &,g14.7,' days in ',i10,' segments of ',g14.7,' days each.'
     &,/1x,'There will be ',i6,' ocean time steps per segment.'
     &,/1x,' The ocean "dtts" =', g14.7, ' seconds'
     &,/,/)
#endif
9300  format (/,10x,' ==>  M.O.M. integration is complete.')
      stop
      end

      subroutine chkcpl (dtatm, dtts)
      logical errorc
#include "param.h"
#include "csbc.h"
#include "switch.h"

!-----------------------------------------------------------------------
!     do consistency checks before allowing model to continue
!-----------------------------------------------------------------------

      errorc = .false.
      write (stdout,*) ' '
      write (stdout,*) '    (checking S.B.C. setup)'

      if (dtatm .eq. c0) then
          write (stdout,9000)
     & '==> Error: the atmospheric time step must be set in "setatm"  '
          errorc = .true.
          dtatm = 1.e-6
      endif
!      critv = 1.e-6
      critv = 1.e-4
      if (segtim .ne. c0) then
        r1 = rundays/segtim
      else
        r1 = 0.5
      endif
      r2 = segtim/(dtts*secday)
      r3 = segtim/(dtatm*secday)
      if (segtim .eq. c0) then
          write (stdout,9000)
     & '==> Error: coupling period "segtim" must be specified         '
          errorc = .true.
      elseif (abs(r1-nint(r1)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of segments       '
     &,'    "segtim"  within "rundays" (the length of the run)        '
          errorc = .true.
      elseif (abs(r2-nint(r2)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of density time   '
     &,'    steps "dtts"  within "segtim" (the segment time)          '
          errorc = .true.
      elseif (abs(r3-nint(r3)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of atmos time     '
     &,'    steps "dtatm"  within "segtim" (the segment time)         '
          errorc = .true.
      endif
#if defined uvic_mom && defined uvic_embm && defined restorst
        write (stdout,9000)
     & '==> Warning: restoring to surface tracers (restorst)          '
     &,'    with coupled model                                        '
#endif
#if defined uvic_mom && defined uvic_embm && defined uvic_replacst
        write (stdout,9000)
     & '==> Warning: replacing surface tracers (uvic_replacst)        '
     &,'    with coupled model                                        '
#endif
#if defined restorst && defined uvic_replacst
        write (stdout,9000)
     & '==> Warning: both restoring and replacing surface tracers     '
     &,'    (restorst, uvic_replacst ). Replacing takes precidence    '
#endif
      write (stdout,*) '    (End of S.B.C. checks) '
      write (stdout,*) ' '
      if (errorc) stop '=>chkcpl'

9000  format (/,(1x,a80))
      return
      end

#if !defined uvic_embm && !defined uvic_replacst

      subroutine getst (jrow, ocnout, ntabc)

!-----------------------------------------------------------------------
!     read surface tracers from disk row "jrow"
!-----------------------------------------------------------------------

# include "param.h"
# include "iounit.h"
# include "mw.h"
# include "tmngr.h"

      dimension ocnout(imt,jmt)

      call getrow (latdisk(taup1disk), nslab, jrow
     &,          u(1,1,jmw,1,taup1), t(1,1,jmw,1,taup1))
      do i=1,imt
        ocnout(i,jrow) = t(i,1,jmw,ntabc,taup1)
      enddo
      return
      end
#endif
