      subroutine init_atm (is, ie, js, je)

!=======================================================================
!     initialize variables for the energy-moisture balance model
!     based on code by a.fanning

!     input:
!       is, ie, js, je = starting and ending indicies for i and j

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm

      implicit none

# include "param.h"
# include "atm.h"
# include "solve.h"
# if defined uvic_ice
#  include "ice.h"
#  if defined uvic_ice_evp
#  include "evp.h"
#  endif
# endif
# include "cembm.h"
# include "csbc.h"
# include "levind.h"
# include "mapsbc.h"
# include "switch.h"
# include "tmngr.h"
# if defined ubc_cidm
#  include "subgrid.h"
# endif

      character(10) :: field

      integer i, ie, is, j, je, js, k, l, m, n, nf, nfields, nu

      real tmp, val

!-----------------------------------------------------------------------
!     set initial or default restart conditions
!-----------------------------------------------------------------------

      nats = namix
      irstdy = 0
      msrsdy = 0
      do j=js,je
        do i=is,ie
          do n=1,nat
            val = 0.0
            if (n .eq. 1) val = 10.0
            if (n .eq. 2) val = 0.01
            at(i,j,1,n) = val
            at(i,j,2,n) = val
          enddo
# if defined uvic_embm_adv_q
          avgp(i,j) = 0.0
          accp(i,j) = 0.0
# endif
# if defined uvic_embm_astress
          awx(i,j) = 0.0
          awy(i,j) = 0.0
# endif
# if defined uvic_embm_land
          soilm(i,j,1) = c0
          soilm(i,j,2) = c0
          surf(i,j,1) = c0
# endif
# if defined uvic_ice
          hice(i,j,1) = 0.0
          hice(i,j,2) = 0.0
          aice(i,j,1) = 0.0
          aice(i,j,2) = 0.0
          tice(i,j) = 0.0
          hsno(i,j,1) = 0.0
          hsno(i,j,2) = 0.0
# endif
# if defined uvic_ice_cpts
          do k=1,ncat
             hseff(i,j,1,k) = 0.0
             hseff(i,j,2,k) = 0.0
             A(i,j,1,k) = 0.0
             A(i,j,2,k) = 0.0
             heff(i,j,1,k) = 0.0
             heff(i,j,2,k) = 0.0
             ts(i,j,1,k) = 0.0
             ts(i,j,2,k) = 0.0
          enddo
          do k=1,ntilay
             E(i,j,1,k) = 0.0
             E(i,j,2,k) = 0.0
          enddo
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
          strength(i,j,1) = 0
          strength(i,j,2) = 0
# endif
# if defined uvic_ice_evp
          uice(i,j) = 0.0
          vice(i,j) = 0.0
# endif
           enddo
      enddo

      if (.not. init) then

!-----------------------------------------------------------------------
!       if starting from a restart file
!-----------------------------------------------------------------------

        call getunit (nu, 'restatm.out', 'u s r ieee')

        read (nu) i, j, nats, dayoyr
        read (nu) nfields
!       check that restart matches this setup
        if (j .ne. jmt .or. i .ne. imt ) then
          write (stdout,*) '==> Warning: atmospheric restart'
     &,     ' is the wrong size '
          write (stdout,'(4(a6, i4))')
     &     ' i= ', i, ' imt= ', imt, ' j= ', j, ' jmt= ', jmt
        endif

!-----------------------------------------------------------------------
!       read in the fields defined in the restart
!-----------------------------------------------------------------------

        do nf=1,nfields
          read (nu) field
          if (field .eq. 'rqbar     ') read (nu) tmp
          if (field .eq. 'aqbar     ') read (nu) tmp
          if (field .eq. 'bv        ') read (nu) tmp
          if (field .eq. 'xv        ') read (nu) tmp
          if (field .eq. 'slap raux ') read (nu) tmp
          if (field .eq. 'slap iaux ') read (nu) tmp
          if (field .eq. 'essl aux1 ') read (nu) tmp
          if (field .eq. 'essl aux2 ') read (nu) tmp
          if (field .eq. 'rqbar     ') read (nu) tmp
          if (field .eq. 'aqbar     ') read (nu) tmp
          if (field .eq. 'counters  ') read (nu) tmp
!         the fields above are obsolete and will be deleted sometime
          if (field .eq. 'time      ')
     &      read (nu) itt, irstdy, msrsdy, stamp
          if (field .eq. 'at        ')
     &      read (nu) ((((at(i,j,l,n),i=is,ie),j=js,je),l=1,2),n=1,nat)
          if (field .eq. 'precip    ')
     &      read (nu) ((precip(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sst       ')
     &      read (nu) ((sbcocn(i,j,isst),i=is,ie),j=js,je)
          if (field .eq. 'sss       ')
     &      read (nu) ((sbcocn(i,j,isss),i=is,ie),j=js,je)
          if (field .eq. 'totaltime ') read (nu) totaltime
# if defined uvic_embm_running_average || defined uvic_embm_astress
          if (field .eq. 'rtbar     ')
     &      read (nu) ((rtbar(i,j),i=is,ie),j=js,je)
          if (field .eq. 'atbar     ')
     &      read (nu) ((atbar(i,j),i=is,ie),j=js,je)
# else
          if (field .eq. 'rtbar     ') read (nu) tmp
          if (field .eq. 'atbar     ') read (nu) tmp
# endif
# if defined uvic_embm_adv_q
          if (field .eq. 'avgp      ') read (nu) avgp
          if (field .eq. 'accp      ') read (nu) accp
# else
          if (field .eq. 'avgp      ') read (nu) tmp
          if (field .eq. 'accp      ') read (nu) tmp
# endif
# if defined uvic_embm_astress
          if (field .eq. 'awx       ')
     &      read (nu) ((awx(i,j),i=is,ie),j=js,je)
          if (field .eq. 'awy       ')
     &      read (nu) ((awy(i,j),i=is,ie),j=js,je)
# else
          if (field .eq. 'awx       ') read (nu) tmp
          if (field .eq. 'awy       ') read (nu) tmp
# endif
# if defined uvic_embm_land
          if (field .eq. 'soilm     ')
     &      read (nu) (((soilm(i,j,l),i=is,ie),j=js,je),l=1,2)
          if (field .eq. 'surf      ')
     &      read (nu) ((surf(i,j,1),i=is,ie),j=js,je)
# else
          if (field .eq. 'soilm     ') read (nu) tmp
          if (field .eq. 'surf      ') read (nu) tmp
# endif
# if defined uvic_ice
          if (field .eq. 'hice      ')
     &      read (nu) (((hice(i,j,l),i=is,ie),j=js,je),l=1,2)
          if (field .eq. 'aice      ')
     &      read (nu) (((aice(i,j,l),i=is,ie),j=js,je),l=1,2)
          if (field .eq. 'tice      ')
     &      read (nu) ((tice(i,j),i=is,ie),j=js,je)
          if (field .eq. 'hsno      ')
     &      read (nu) (((hsno(i,j,l),i=is,ie),j=js,je),l=1,2)
# else
          if (field .eq. 'hice      ') read (nu) tmp
          if (field .eq. 'aice      ') read (nu) tmp
          if (field .eq. 'tice      ') read (nu) tmp
          if (field .eq. 'hsno      ') read (nu) tmp
# endif
# if defined uvic_ice_cpts
          if (field .eq. 'heff      ') read (nu)
     &      ((((heff(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
          if (field .eq. 'A         ') read (nu)
     &      ((((A   (i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
          if (field .eq. 'E         ') read (nu)
     &      ((((E(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ntilay)
          if (field .eq. 'ts        ') read (nu)
     &      ((((ts  (i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
          if (field .eq. 'hseff     ') read (nu)
     &      ((((hseff(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
# else
          if (field .eq. 'heff      ') read (nu) tmp
          if (field .eq. 'A         ') read (nu) tmp
          if (field .eq. 'E         ') read (nu) tmp
          if (field .eq. 'ts        ') read (nu) tmp
          if (field .eq. 'hseff     ') read (nu) tmp
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
          if (field .eq. 'strength  ') read (nu)
     &      (((strength(i,j,l),i=is,ie),j=js,je),l=1,2)
# else
          if (field .eq. 'strength  ') read (nu) tmp
# endif
# if defined uvic_ice_evp
          if (field .eq. 'uice      ')
     &      read (nu) ((uice(i,j),i=is,ie),j=js,je)
          if (field .eq. 'vice      ')
     &      read (nu) ((vice(i,j),i=is,ie),j=js,je)
          if (field .eq. 'surf u    ')
     &      read (nu) ((sbcocn(i,j,isu),i=is,ie),j=js,je)
          if (field .eq. 'surf v    ')
     &      read (nu) ((sbcocn(i,j,isv),i=is,ie),j=js,je)
          if (field .eq. 'geo u     ')
     &      read (nu) ((sbcocn(i,j,igu),i=is,ie),j=js,je)
          if (field .eq. 'geo v     ')
     &      read (nu) ((sbcocn(i,j,igv),i=is,ie),j=js,je)
          if (field .eq. 'sig11n    ')
     &      read (nu) ((sig11n(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig11e    ')
     &      read (nu) ((sig11e(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig11s    ')
     &      read (nu) ((sig11s(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig11w    ')
     &      read (nu) ((sig11w(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig22n    ')
     &      read (nu) ((sig22n(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig22e    ')
     &      read (nu) ((sig22e(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig22s    ')
     &      read (nu) ((sig22s(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig22w    ')
     &      read (nu) ((sig22w(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig12n    ')
     &      read (nu) ((sig12n(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig12e    ')
     &      read (nu) ((sig12e(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig12s    ')
     &      read (nu) ((sig12s(i,j),i=is,ie),j=js,je)
          if (field .eq. 'sig12w    ')
     &      read (nu) ((sig12w(i,j),i=is,ie),j=js,je)
# else
          if (field .eq. 'uice      ') read (nu) tmp
          if (field .eq. 'vice      ') read (nu) tmp
          if (field .eq. 'surf u    ') read (nu) tmp
          if (field .eq. 'surf v    ') read (nu) tmp
          if (field .eq. 'geo u     ') read (nu) tmp
          if (field .eq. 'geo v     ') read (nu) tmp
          if (field .eq. 'sig11n    ') read (nu) tmp
          if (field .eq. 'sig11e    ') read (nu) tmp
          if (field .eq. 'sig11s    ') read (nu) tmp
          if (field .eq. 'sig11w    ') read (nu) tmp
          if (field .eq. 'sig22n    ') read (nu) tmp
          if (field .eq. 'sig22e    ') read (nu) tmp
          if (field .eq. 'sig22s    ') read (nu) tmp
          if (field .eq. 'sig22w    ') read (nu) tmp
          if (field .eq. 'sig12n    ') read (nu) tmp
          if (field .eq. 'sig12e    ') read (nu) tmp
          if (field .eq. 'sig12s    ') read (nu) tmp
          if (field .eq. 'sig12w    ') read (nu) tmp
# endif
        enddo

        call release_all

        if (init_time) then
          itt = 0
          iday(imodeltime) = 0
          msday(imodeltime) = 0
          relyr = 0.0
          call mkstmp (stamp, year0, month0, day0, hour0, min0, sec0)
        endif
        write (*,'(a,a,a)')
     &    ' ==>  Reading an atmospheric restart from file'
     &,   ' restatm.out on ', stamp

# if defined ubc_cidm
        subhsno(:,:,1:2) = hsno(:,:,1:2)

# endif
      endif
# if !defined uvic_mom

!-----------------------------------------------------------------------
!     initialize the time manager with specified initial conditions
!     time, user reference time, model time, and how long to integrate.
!-----------------------------------------------------------------------

      call tmngri (year0, month0, day0, hour0, min0, sec0
     &,            ryear, rmonth, rday, rhour, rmin, rsec
     &,            irstdy, msrsdy, runlen, rununits, rundays, dtatm)
# endif

!-----------------------------------------------------------------------
!     zero initial flux accumulators
!-----------------------------------------------------------------------

# if defined uvic_ice_evp || defined uvic_embm_astress
      do n=1,nat+2
# else
      do n=1,nat
# endif
        do j=js,je
          do i=is,ie
            flux(i,j,n) = 0.0
          enddo
        enddo
      enddo

      return
      end

      subroutine save_atm (fname, is, ie, js, je)

!=======================================================================
!     write restart for energy-moisture balance model
!     based on code by a.fanning

!     input:
!       fname = restart file name
!       is, ie, js, je = starting and ending indicies for i and j

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

# include "param.h"
# include "atm.h"
# if defined uvic_ice
#  include "ice.h"
# endif
# if defined uvic_ice_evp && defined uvic_ice
#  include "evp.h"
# endif
# include "cembm.h"
# include "csbc.h"
# include "levind.h"
# include "mapsbc.h"
# include "switch.h"
# include "tmngr.h"

      character(*) :: fname

      integer i, ie, is, j, je, js, k, l, m, n, nfields, nu

      write (*,'(a,a,a,i10,1x,a)')
     &  ' ==>  Writing an atmospheric restart to file '
     &, trim(fname), ' on ts=', itt, stamp
      call getunit (nu, fname, 'u s r ieee')

      nfields = 6
# if defined uvic_embm_running_average || defined uvic_embm_astress
     &        + 2
# endif
# if defined uvic_embm_adv_q
     &        + 2
# endif
# if defined uvic_embm_astress
     &        + 2
# endif
# if defined uvic_embm_land
     &        + 2
# endif
# if defined uvic_ice
     &        + 4
# endif
# if defined uvic_ice_cpts
     &        + 5
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
     &        + 1
# endif
# if defined uvic_ice_evp
     &        + 6
# endif
# if defined uvic_ice_evp
     &        +12
# endif

      write (nu) ie , je , nats, dayoyr
      write (nu) nfields
      write (nu) 'time      '
      write (nu) itt, iday(imodeltime), msday(imodeltime), stamp
      write (nu) 'at        '
      write (nu) ((((at(i,j,l,n),i=is,ie),j=js,je),l=1,2),n=1,nat)
      write (nu) 'precip    '
      write (nu) ((precip(i,j),i=is,ie),j=js,je)
      write (nu) 'sst       '
      write (nu) ((sbcocn(i,j,isst),i=is,ie),j=js,je)
      write (nu) 'sss       '
      write (nu) ((sbcocn(i,j,isss),i=is,ie),j=js,je)
      write (nu) 'totaltime '
      write (nu) totaltime
# if defined uvic_embm_running_average || defined uvic_embm_astress
      write (nu) 'rtbar     '
      write (nu) ((rtbar(i,j),i=is,ie),j=js,je)
      write (nu) 'atbar     '
      write (nu) ((atbar(i,j),i=is,ie),j=js,je)
# endif
# if defined uvic_embm_adv_q
      write (nu) 'avgp      '
      write (nu) ((avgp(i,j),i=is,ie),j=js,je)
      write (nu) 'accp      '
      write (nu) ((accp(i,j),i=is,ie),j=js,je)
# endif
# if defined uvic_embm_astress
      write (nu) 'awx       '
      write (nu) ((awx(i,j),i=is,ie),j=js,je)
      write (nu) 'awy       '
      write (nu) ((awy(i,j),i=is,ie),j=js,je)
# endif
# if defined uvic_embm_land
      write (nu) 'soilm     '
      write (nu) (((soilm(i,j,l),i=is,ie),j=js,je),l=1,2)
      write (nu) 'surf      '
      write (nu) ((surf(i,j,1),i=is,ie),j=js,je)
# endif
# if defined  uvic_ice
      write (nu) 'hsno      '
      write (nu) (((hsno(i,j,l),i=is,ie),j=js,je),l=1,2)
      write (nu) 'hice      '
      write (nu) (((hice(i,j,l),i=is,ie),j=js,je),l=1,2)
      write (nu) 'aice      '
      write (nu) (((aice(i,j,l),i=is,ie),j=js,je),l=1,2)
      write (nu) 'tice      '
      write (nu) ((tice(i,j),i=is,ie),j=js,je)
# endif
# if defined uvic_ice_cpts
      write (nu) 'hseff     '
      write (nu) ((((hseff(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
      write (nu) 'heff      '
      write (nu)  ((((heff(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
      write (nu) 'A         '
      write (nu)  ((((A(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
      write (nu) 'E         '
      write (nu)  ((((E(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ntilay)
      write (nu) 'ts        '
      write (nu)  ((((ts(i,j,l,k),i=is,ie),j=js,je),l=1,2),k=1,ncat)
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
      write (nu) 'strength  '
      write (nu)  (((strength(i,j,l),i=is,ie),j=js,je),l=1,2)
# endif
# if defined uvic_ice_evp
      write (nu) 'uice      '
      write (nu) ((uice(i,j),i=is,ie),j=js,je)
      write (nu) 'vice      '
      write (nu) ((vice(i,j),i=is,ie),j=js,je)
      write (nu) 'surf u    '
      write (nu) ((sbcocn(i,j,isu),i=is,ie),j=js,je)
      write (nu) 'surf v    '
      write (nu) ((sbcocn(i,j,isv),i=is,ie),j=js,je)
      write (nu) 'geo u     '
      write (nu) ((sbcocn(i,j,igu),i=is,ie),j=js,je)
      write (nu) 'geo v     '
      write (nu) ((sbcocn(i,j,igv),i=is,ie),j=js,je)
      write (nu) 'sig11n    '
      write (nu) ((sig11n(i,j),i=is,ie),j=js,je)
      write (nu) 'sig11s    '
      write (nu) ((sig11s(i,j),i=is,ie),j=js,je)
      write (nu) 'sig11e    '
      write (nu) ((sig11e(i,j),i=is,ie),j=js,je)
      write (nu) 'sig11w    '
      write (nu) ((sig11w(i,j),i=is,ie),j=js,je)
      write (nu) 'sig22n    '
      write (nu) ((sig22n(i,j),i=is,ie),j=js,je)
      write (nu) 'sig22s    '
      write (nu) ((sig22s(i,j),i=is,ie),j=js,je)
      write (nu) 'sig22e    '
      write (nu) ((sig22e(i,j),i=is,ie),j=js,je)
      write (nu) 'sig22w    '
      write (nu) ((sig22w(i,j),i=is,ie),j=js,je)
      write (nu) 'sig12n    '
      write (nu) ((sig12n(i,j),i=is,ie),j=js,je)
      write (nu) 'sig12s    '
      write (nu) ((sig12s(i,j),i=is,ie),j=js,je)
      write (nu) 'sig12e    '
      write (nu) ((sig12e(i,j),i=is,ie),j=js,je)
      write (nu) 'sig12w    '
      write (nu) ((sig12w(i,j),i=is,ie),j=js,je)
# endif

      call release_all
#endif

      return
      end

#if defined uvic_old_ws && defined uvic_embm
      subroutine windspd

!=======================================================================
!     convert wind stress tx & ty to wind speed for use in calculating
!     heat flux from atmosphere
!     based on code by a.fanning

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

# include "param.h"
# include "atm.h"
# include "grdvar.h"
# include "cembm.h"
# include "csbc.h"
# include "mapsbc.h"
# if defined uvic_embm_adv_q
#  include "levind.h"
# endif

      integer i, j

      real drag, tmag

!     calculate wind speed

      drag = cdatm*rhoatm
      do j=1,jmt
        do i=1,imt
          tmag = sqrt(sbcocn(i,j,itaux)**2 + sbcocn(i,j,itauy)**2)
          sbcocn(i,j,iws) = sqrt(tmag/drag)
        enddo
      enddo

!     do a four point average to get to t from u,v points

      do j=jmtm1,2,-1
        do i=imtm1,2,-1
          sbcocn(i,j,iws) = 0.25*(sbcocn(i,j,iws) + sbcocn(i-1,j,iws)
     &                    + sbcocn(i-1,j-1,iws) + sbcocn(i,j-1,iws))
        enddo
      enddo

      call embmbc (sbcocn(1,1,iws))

      return
      end
#endif

#if defined uvic_mom
      subroutine read_restart

!=======================================================================

!     read the restart file

!     author:        a. rosati             e-mail=> ar@gfdl.gov
!                    r. c. pacanowski      e-mail=> rcp@gfdl.gov
!=======================================================================

#include "param.h"
#if defined trajectories
# include "ptraj.h"
#endif
#include "coord.h"
#include "emode.h"
#include "grdvar.h"
#include "iounit.h"
#include "levind.h"
#include "mw.h"
#include "task_on.h"
#include "tmngr.h"
#include "switch.h"

      parameter (mxnrvar=nvar+5)
      dimension bufsl(imt,km,mxnrvar)

!-----------------------------------------------------------------------
!     read restart file (unformatted)

!     record 1 on the restart file is the control block (containing
!              the time step and relative year)
!     record`s 2 & 3 are for stream function time levels
!     record`s 4 & 5 are for the d(stream function)/dt fields used
!              to construct a guess for the elliptic solver
!     record 6 is for "kmt" indicating the number of ocean "t" cells
!              from the surface to the ocean floor.
!     the next "jmt" records contain prognostic variables at "tau"
!     the last "jmt" records contain prognostic variables at "tau+1"
!-----------------------------------------------------------------------

      call getunit (iorest, 'restocn.out', 'u s r')

!-----------------------------------------------------------------------
!     read the time step and stamp from the restart file
!     timestep "itt" at this point corresponds to "tau+1" which was the
!     condition at the end of the previous timestep
!-----------------------------------------------------------------------

      read (iorest) stamp, iotext
      read (iorest) itt, irstdy, msrsdy, im, jm, kkm
      if (init_time) then
        itt = 0
        iday(imodeltime) = 0
        msday(imodeltime) = 0
        relyr = 0.0
        call mkstmp (stamp, year0, month0, day0, hour0, min0, sec0)
      endif
      write (stdout,'(a,i10,1x,a)')
     &'===> Reading MOM restart from file restocn.out on ts=',itt, stamp
      if (im .ne. imt .or. jm .ne. jmt .or. kkm .ne. km) then
        write (stdout,*) '=>Error restio. im=',im,' jm=',jm,' km=',kkm
      endif

!-----------------------------------------------------------------------
!     update pointers to tau-1, tau, & tau+1 data on disk based on itt
!     for latitude rows they point to latdisk(1) or latdisk(2)
!     for 2D fields they point to records on kflds
!-----------------------------------------------------------------------

#if defined coarse_grained_parallelism
        taum1disk = mod(itt+2,3) + 1
        taudisk   = mod(itt  ,3) + 1
        taup1disk = mod(itt+1,3) + 1
#else
        taum1disk = mod(itt+1,2) + 1
        taudisk   = mod(itt  ,2) + 1
        taup1disk = taum1disk
#endif

!-----------------------------------------------------------------------
!     update pointers to tau-1, tau, & tau+1 data in the MW based on itt
!-----------------------------------------------------------------------

      if (wide_open_mw) then

!       rotate time levels instead of moving data

        taum1 = mod(itt+0,3) - 1
        tau   = mod(itt+1,3) - 1
        taup1 = mod(itt+2,3) - 1
      endif

#if defined rigid_lid_surface_pressure || defined implicit_free_surface
# if defined restnosp

!-----------------------------------------------------------------------
!     when restarting a prognostic surface pressure run from a
!     stream function restart...
!     initialize ubar and vbar from the stream function case
!     first read psi at "tau" then psi at "tau+1"
!     warning: ubar and vbar will be multiplied by "hr" below when
!     "hr" is available
!-----------------------------------------------------------------------

      do m=2,1,-1
        read (iorest) stamp, iotext
        read (iorest) iimt, ijmt, ptd
        if (iimt .ne. imt .or. ijmt .ne. jmt) then
          write (stdout,*) ' => Error restio #0,imt=',iimt,',jmt=',ijmt
     &,   ' m=',m
        endif
        do jrow=1,jmtm1
          do i=2,imtm1
            diag1   = ptd(i+1,jrow+1) - ptd(i  ,jrow)
            diag0   = ptd(i  ,jrow+1) - ptd(i+1,jrow)
            if (m .eq. 1) then
              ubar(i,jrow,1) = -(diag1+diag0)*dyu2r(jrow)
              ubar(i,jrow,2) =  (diag1-diag0)*dxu2r(i)*csur(jrow)
            else
              ubarm1(i,jrow,1) = -(diag1+diag0)*dyu2r(jrow)
              ubarm1(i,jrow,2) =  (diag1-diag0)*dxu2r(i)*csur(jrow)
            endif
          enddo
        enddo
      enddo
      call setbcx (ubar(1,1,1), imt, jmt)
      call setbcx (ubarm1(1,1,1), imt, jmt)
      call setbcx (ubar(1,1,2), imt, jmt)
      call setbcx (ubarm1(1,1,2), imt, jmt)
      read (iorest) stamp, iotext
      read (iorest)
      read (iorest) stamp, iotext
      read (iorest)

!     set the surface pressure equal to zero

      do it=1,2
        do jrow=1,jmt
          do i=1,imt
            ps(i,jrow,it) = c0
          enddo
        enddo
      enddo
# else

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ps
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
          write (stdout,*) ' => Error restio #A,imt=',iimt,',jmt=',ijmt
      endif

!     first read "tau" then "tau+1"

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ((ubarm1(i,jrow,1),i=1,imt),jrow=1,jmt)
     &,                         ((ubarm1(i,jrow,2),i=1,imt),jrow=1,jmt)
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
          write (stdout,*) ' => Error restio #B,imt=',iimt,',jmt=',ijmt
      endif

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ((ubar(i,jrow,1),i=1,imt),jrow=1,jmt)
     &,                         ((ubar(i,jrow,2),i=1,imt),jrow=1,jmt)
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
          write (stdout,*) ' => Error restio #C,imt=',iimt,',jmt=',ijmt
      endif
# endif

!     initialize the kflds disk which is used for surface pressure.

      do jrow=1,jmt
        do i=1,imt
          pguess(i,jrow) = c0
          ptd(i,jrow)    = ps(i,jrow,1)
        enddo
      enddo
      call oput (kflds, nwds, nkflds-1, ptd)
      call oput (kflds, nwds, nkflds, ptd)

#endif

#if defined stream_function

!     first read "tau" then "tau+1" stream function

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ((psi(i,jrow,2),i=1,imt),jrow=1,jmt)
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
        write (stdout,*) ' => Error restart #1, imt=',iimt,', jmt=',ijmt
      endif

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ((psi(i,jrow,1),i=1,imt),jrow=1,jmt)
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
        write (stdout,*) ' => Error restart #2, imt=',iimt,', jmt=',ijmt
      endif

!     read the two guess fields

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ptd
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
        write (stdout,*) ' => Error restart #3, imt=',iimt,', jmt=',ijmt
      endif
      call oput (kflds, nwds, nkflds-1, ptd)

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, ptd
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
        write (stdout,*) ' => Error restart #4, imt=',iimt,', jmt=',ijmt
      endif
      call oput (kflds, nwds, nkflds, ptd)
#endif

!     read "kmt" which indicates the number of t" cells to ocean floor

      read (iorest) stamp, iotext
      read (iorest) iimt, ijmt, kmt
      if (iimt .ne. imt .or. ijmt .ne. jmt) then
        write (stdout,*) ' => Error restart #5, imt=',iimt,', jmt=',ijmt
      endif

#if defined uvic_mom
!-----------------------------------------------------------------------
!     construct depth arrays associated with "u" cells
!-----------------------------------------------------------------------

      call depth_u (kmt, imt, jmt, zw, km, kmu, h, hr)

#endif
#if defined rigid_lid_surface_pressure || defined implicit_free_surface
# if defined restnosp

!     complete external mode calculation using "hr"

      do m=1,2
        do jrow=1,jmt
          do i=1,imt
            ubar(i,jrow,m)   = ubar(i,jrow,m)*hr(i,jrow)
            ubarm1(i,jrow,m) = ubarm1(i,jrow,m)*hr(i,jrow)
          enddo
        enddo
        call setbcx (ubar(1,1,m), imt, jmt)
        call setbcx (ubarm1(1,1,m), imt, jmt)
      enddo
# endif
#endif

!-----------------------------------------------------------------------
!     read the "tau" latitude rows
!-----------------------------------------------------------------------

      do jrow=1,jmt

        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
        endif

        read (iorest) stamp, iotext
        read (iorest,err=101) ilensl, bufsl
101     continue
        nrvar = ilensl/(imt*km)
        if (nrvar .ne. nvar .and. jrow .eq. 1) then
          write (stdout,*) ' => Warning: ocean restart has a different'
     &,     ' number of slab variables.'
        endif
        if (nrvar .gt. mxnrvar) then
          write (stdout,*) ' => Error: ocean restart has more than '
     &,     'mxnrvar=' ,mxnrvar, ' variables. Increase mxnrvar to '
     &,      nrvar, ' in restio.F'
          stop
        endif
#if defined tcvmix
!       How are the tcvmix variables read in???????
        uu = 1.0/undef
        uu = uu**2
#endif

!       velocity is internal mode component only
!       note that tracers are stored first, then velocities on restart
!       for compatability with previous versions

        do k=1,km
          do i=1,imt
            u(i,k,j,1,tau) = bufsl(i,k,nrvar-1)
            u(i,k,j,2,tau) = bufsl(i,k,nrvar)
            do n=1,nvar-2
              if (n. gt.nrvar-2 ) then
                if (i .eq. 1 .and. jrow .eq. 1 .and. k .eq. 1)
     &            write (stdout,*) ' => Warning: initialising tracer', n
                t(i,k,j,n,tau) = 0.
              else
                t(i,k,j,n,tau) = bufsl(i,k,n)
              endif
            enddo
          enddo
        enddo

!       initialize every latitude

        if (wide_open_mw) then

!         do nothing since "tau" data is in place in the MW

        else
          call putrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
        endif
      enddo

      if (wide_open_mw) then

!       Initialze 1st and last latitude row for tau-1 to prevent
!       use of uninitialized values on boundary row.

        do j=1,jmt,jmt-1
          do k=1,km
            do i=1,imt
              u(i,k,j,1,taum1) = u(i,k,j,1,tau)
              u(i,k,j,2,taum1) = u(i,k,j,2,tau)
              do n=1,nvar-2
                t(i,k,j,n,taum1) = t(i,k,j,n,tau)
              enddo
            enddo
          enddo
        enddo
      endif

!-----------------------------------------------------------------------
!     read the "tau+1" latitude rows
!-----------------------------------------------------------------------

      do jrow=1,jmt

        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
        endif

        read (iorest) stamp, iotext
        read (iorest,err=102) ilensl, bufsl
102     continue
#if defined tcvmix
!       How are the tcvmix variables read in???????
        uu = 1.0/undef
        uu = uu**2
#endif

!       velocity is internal mode component only
!       note that tracers are stored first, then velocities on restart
!       for compatability with previous versions

        do k=1,km
          do i=1,imt
            u(i,k,j,1,taup1) = bufsl(i,k,nrvar-1)
            u(i,k,j,2,taup1) = bufsl(i,k,nrvar)
            do n=1,nvar-2
              if (n. gt.nrvar-2 ) then
                t(i,k,j,n,taup1) = 0.
              else
                t(i,k,j,n,taup1) = bufsl(i,k,n)
              endif
            enddo
          enddo
        enddo

!       initialize every latitude

        if (wide_open_mw) then

!         do nothing since "tau+1" data is in place in the MW

        else
          call putrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif
      enddo

#if defined trajectories
      read (iorest, end=210, err=210) stamp, iotext
      read (iorest, end=210, err=210)
      read (iorest, end=210, err=210) stamp, iotext
# if defined lyapunov
      read (iorest,err=205) npart, pxyz, pijk, em
# else
      read (iorest) npart, pxyz, pijk
# endif
      write (stdout, '(a,i8,a)')
     &'===> also read trajectory restart for ',npart,' particles'
      if (npart .ne. nptraj) then
        write (stdout,*) ' => Particle trajectory restart error:'
        write (stdout,*) '    read ',npart,' particles but expected '
     &,                     nptraj
        stop "=>rdrest"
      endif
      go to 220
# if defined lyapunov
205   continue
      write (stdout,'(/a/a/)')
     & '==>Note: Did not find lyapunov part of trajectory restart...'
     &,'         Setting all particles to Initial Positions'
# endif
210   continue

!     if no restart data is available. initialize the particles

      call ptraji
220   continue
#endif
      rewind iorest

      call relunit (iorest)

      return
      end

      subroutine write_restart (fname)

!=======================================================================

!     write the restart file

!     author:        a. rosati             e-mail=> ar@gfdl.gov
!                    r. c. pacanowski      e-mail=> rcp@gfdl.gov
!=======================================================================

#include "param.h"
#if defined trajectories
# include "ptraj.h"
#endif
#include "coord.h"
#include "emode.h"
#include "grdvar.h"
#include "iounit.h"
#include "levind.h"
#include "mw.h"
#include "switch.h"
#include "task_on.h"
#include "tmngr.h"
      character(*) :: fname
      character(80) :: file_stamp
      character(32) :: ostamp
      dimension bufsl(imt,km,nvar)
#if defined stream_function
      dimension ext(imt,2)
#endif

      call getunit (iorest, 'restocn.out', 'u s r')

      if (init_time) then
        itt = 0
        iday(imodeltime) = 0
        msday(imodeltime) = 0
        call mkstmp (stamp, year0, month0, day0, hour0, min0, sec0)
      endif
      call getunit (iorest, fname, 'u s r')
      write (stdout,'(a,a,a,i8,1x,a)')
     &  '==> Writing MOM restart to file ', trim(fname)
     &, ', ts=', itt, stamp

!-----------------------------------------------------------------------
!     write restart file using unformatted fortran i/o

!     record 1 on the restart file is the control block (containing
!              the time step and relative year)
!     record`s 2 & 3 are for stream function time levels
!     record`s 4 & 5 are for the d(stream function)/dt fields used
!              to construct a guess for the elliptic solver
!     record 6 is for "kmt" indicating the number of ocean "t" cells
!              from the surface to the ocean floor.
!     the next "jmt" records contain prognostic variables at "tau"
!     the last "jmt" records contain prognostic variables at "tau+1"
!-----------------------------------------------------------------------

!     note that timestep "itt" corresponds to "tau+1"

      iotext = ' read (iorest) itt, irstdy, msrsdy, imt, jmt, km'
      write (iorest) stamp, iotext, expnam
      write (iorest) itt, iday(imodeltime), msday(imodeltime)
     &, imt, jmt, km

#if defined rigid_lid_surface_pressure || defined implicit_free_surface

      iotext =
     &' read (iorest) imt,jmt, ((ps(i,j,1),i=1,imt),j=1,jmt), ps(,,2)'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ps

!     (ubarm1,ubar) is at (tau,tau+1) here

      iotext =
     &' read (iorest) imt,jmt,((ubarm1(i,j),i=1,imt),j=1,jmt), vbarm1()'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ((ubarm1(i,jrow,1),i=1,imt),jrow=1,jmt)
     &,                        ((ubarm1(i,jrow,2),i=1,imt),jrow=1,jmt)

      iotext =
     &' read (iorest) imt, jmt, ((ubar(i,j),i=1,imt),j=1,jmt), vbar()'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ((ubar(i,jrow,1),i=1,imt),jrow=1,jmt)
     &,                        ((ubar(i,jrow,2),i=1,imt),jrow=1,jmt)
#endif

#if defined stream_function

!     first do psi at "tau" then at "tau+1"

      iotext = ' read (iorest) imt, jmt, ((psi(i,j,2),i=1,imt),j=1,jmt)'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ((psi(i,jrow,2),i=1,imt),jrow=1,jmt)

      iotext = ' read (iorest) imt, jmt, ((psi(i,j,1),i=1,imt),j=1,jmt)'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ((psi(i,jrow,1),i=1,imt),jrow=1,jmt)

!     guess fields

      call oget (kflds, nwds, nkflds-1, ptd)
      iotext = ' read (iorest) imt, jmt, ((g1(i,j),i=1,imt),j=1,jmt)'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ptd

      call oget (kflds, nwds, nkflds, ptd)
      iotext = ' read (iorest) imt, jmt, ((g2(i,j),i=1,imt),j=1,jmt)'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, ptd
#endif

      iotext = ' read (iorest) imt, jmt, ((kmt(i,j),i=1,imt),j=1,jmt)'
      write (iorest) stamp, iotext, expnam
      write (iorest) imt, jmt, kmt

!-----------------------------------------------------------------------
!     save the "tau" latitude rows
!-----------------------------------------------------------------------

#if defined coarse_grained_parallelism
      taum1disk = mod(itt+2,3) + 1
      taudisk   = mod(itt  ,3) + 1
      taup1disk = mod(itt+1,3) + 1
#endif

      ilensl = imt*km*nvar
      do jrow=1,jmt

        if (wide_open_mw) then
          j = jrow

!         remove external mode from "tau". since psi has been updated
!         psi(,,2) is at "tau"

          if (jrow .lt. jmt) then
#if defined stream_function
            do i=2,imt-1
              diag1    = psi(i+1,jrow+1,2) - psi(i  ,jrow,2)
              diag0    = psi(i  ,jrow+1,2) - psi(i+1,jrow,2)
              ext(i,1)  = -(diag1+diag0)*dyu2r(jrow)*hr(i,jrow)
              ext(i,2)  =  (diag1-diag0)*dxu2r(i)*hr(i,jrow)
     &                       *csur(jrow)
            enddo
            do k=1,km
              do i=2,imt-1
                if (k .le. kmu(i,jrow)) then
                  bufsl(i,k,nvar-1) = (u(i,k,j,1,tau) - ext(i,1))
                  bufsl(i,k,nvar)   = (u(i,k,j,2,tau) - ext(i,2))
                else
                  bufsl(i,k,nvar-1) = c0
                  bufsl(i,k,nvar)   = c0
                endif
              enddo
            enddo
#endif
#if defined rigid_lid_surface_pressure || defined implicit_free_surface

!           since external mode has been updated, ubarm1 is at "tau"

            do k=1,km
              do i=2,imt-1
                if (k .le. kmu(i,jrow)) then
                  bufsl(i,k,nvar-1) = (u(i,k,j,1,tau)-ubarm1(i,jrow,1))
                  bufsl(i,k,nvar)   = (u(i,k,j,2,tau)-ubarm1(i,jrow,2))
                else
                  bufsl(i,k,nvar-1) = c0
                  bufsl(i,k,nvar)   = c0
                endif
              enddo
            enddo
#endif
            call setbcx (bufsl(1,1,nvar-1), imt, km)
            call setbcx (bufsl(1,1,nvar), imt, km)
          else
            do k=1,km
              do i=1,imt
                bufsl(i,k,nvar-1) = c0
                bufsl(i,k,nvar)   = c0
              enddo
            enddo
          endif

        else
          j = jmw
          call getrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
          do k=1,km
            do i=1,imt
              bufsl(i,k,nvar-1) = u(i,k,j,1,tau)
              bufsl(i,k,nvar)   = u(i,k,j,2,tau)
            enddo
          enddo
        endif

        do k=1,km
          do i=1,imt
            do n=1,nvar-2
              bufsl(i,k,n) = t(i,k,j,n,tau)
            enddo
          enddo
        enddo
        write (iotext,'(a15,i4)') ' taudisk jrow =',jrow
        iotext(20:) = ' read(iorest) lensl, (bufsl(i),i=1,lensl)'
        write (iorest) stamp, iotext, expnam
        write (iorest) ilensl, bufsl
      enddo

!-----------------------------------------------------------------------
!     save the "tau+1" latitude rows
!-----------------------------------------------------------------------

      do jrow=1,jmt

        if (wide_open_mw) then
          j = jrow

!         do nothing since data is already in "tau+1" position in MW
!         and velocity is already internal mode only

        else
          j = jmw
          call getrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif

        do k=1,km
          do i=1,imt
            bufsl(i,k,nvar-1) = u(i,k,j,1,taup1)
            bufsl(i,k,nvar) = u(i,k,j,2,taup1)
            do n=1,nvar-2
              bufsl(i,k,n) = t(i,k,j,n,taup1)
            enddo
          enddo
        enddo
        write (iotext,'(a17,i4)') ' taup1disk jrow =',jrow
        iotext(22:) = ' read(iorest) lensl, (bufsl(i),i=1,lensl)'
        write (iorest) stamp, iotext, expnam
        write (iorest) ilensl, bufsl
      enddo

#if defined trajectories
      reltim = relyr
      iotext = 'read (iorest) reltim'
      write (iorest) stamp, iotext, expnam
      write (iorest) reltim
# if defined lyapunov
      iotext ='read (iorest) nptraj, pxyz, pijk, em'
      write (iorest) stamp, iotext, expnam
      write (iorest) nptraj, pxyz, pijk, em
# else
      iotext ='read (iorest) nptraj, pxyz, pijk'
      write (iorest) stamp, iotext, expnam
      write (iorest) nptraj, pxyz, pijk
# endif
      write (stdout, '(a,i8,a)')
     &'===> also writing trajectory restart for ',nptraj, 'particles'
#endif

      call relunit (iorest)

      return
      end
#endif
