      subroutine atm_rest_in (fname, ids, ide, jds, jde)
!=======================================================================
!     input routine for atmospheric restarts

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm

      implicit none

# include "param.h"
# include "atm.h"
# include "cembm.h"
# include "coord.h"
# include "csbc.h"
# if defined uvic_ice_evp
#  include "evp.h"
# endif
# include "grdvar.h"
# if defined uvic_ice
#  include "ice.h"
# endif
# include "mapsbc.h"
# include "tmngr.h"
# include "switch.h"

      character(*) :: fname
      character(80) :: name
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, ids, ide, jds, jde, igs, ige, ig
      integer jgs, jge, jg, ils, ile, jls, jle, is(10), ic(10), it(10)
      integer iu(10), id_time, id_xt, id_xu, id_yt, id_yu, id_xt_e
      integer id_xu_e, id_yt_e, id_yu_e
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)
      real time, tmp(1)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      time = 0.
      ntrec = 1
      call openfile (name, time, ntrec, iou)

!-----------------------------------------------------------------------
!     get global read domain size (may be less than global domain)
!-----------------------------------------------------------------------
      call getaxis('xt', iou, imt, xt, igs, ige, 1., 0.)
      call getaxis('xu', iou, imt, xu, igs, ige, 1., 0.)
      ig  = ige-igs+1
      call getaxis('yt', iou, jmt, yt, jgs, jge, 1., 0.)
      call getaxis('yu', iou, jmt, yu, igs, ige, 1., 0.)
      jg  = jge-jgs+1

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global read domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)

!-----------------------------------------------------------------------
!     read 1d data (t)
!-----------------------------------------------------------------------
      call getvara ('nats', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nats = tmp(1)
      call getvara ('dayoyr', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      dayoyr = tmp(1)
      call getvara ('itt', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      itt = tmp(1)
      call getvara ('irstdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      irstdy = tmp(1)
      call getvara ('msrsdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      msrsdy = tmp(1)
      call getvara ('totaltime', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      totaltime = tmp(1)
      call getvara ('year', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nyear = tmp(1)
      call getvara ('month', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nmonth = tmp(1)
      call getvara ('day', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nday = tmp(1)
      call getvara ('hour', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nhour = tmp(1)
      call getvara ('minute', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nmin = tmp(1)
      call getvara ('second', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      nsec = tmp(1)
      call mkstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      if (init_time) then
        itt = 0
        irstdy = 0
        msrsdy = 0
        iday(imodeltime) = 0
        msday(imodeltime) = 0
        relyr = 0.0
        call mkstmp (stamp, year0, month0, day0, hour0, min0, sec0)
      endif

!-----------------------------------------------------------------------
!     read 3d data (x,y,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      is(3) = 1
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)
      do n=1,nat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call getvara('at1_'//trim(a3), iou, ln, is, ic
     &,   at(ils:ile,jls:jle,1,n), 1., 0.)
        call getvara('at2_'//trim(a3), iou, ln, is, ic
     &,   at(ils:ile,jls:jle,2,n), 1., 0.)
      enddo
      call getvara ('precip', iou, ln, is, ic, precip(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sst', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isst)
     &, 1., 0.)
      call getvara ('sss', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isss)
     &, 1., 0.)
# if defined uvic_embm_running_average || defined uvic_embm_astress
      call getvara ('rtbar', iou, ln, is, ic, rtbar(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('atbar', iou, ln, is, ic, atbar(ils:ile,jls:jle)
     &, 1., 0.)
# endif
# if defined uvic_embm_adv_q
      call getvara ('avgp', iou, ln, is, ic, avgp(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('accp', iou, ln, is, ic, accp(ils:ile,jls:jle)
     &, 1., 0.)
# endif
# if defined uvic_embm_astress
      call getvara ('awx', iou, ln, is, ic, awx(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('awy', iou, ln, is, ic, awy(ils:ile,jls:jle)
     &, 1., 0.)
# endif
# if defined uvic_embm_land
      call getvara ('soilm1', iou, ln, is, ic, soilm(ils:ile,jls:jle,1)
     &, 1., 0.)
      call getvara ('soilm2', iou, ln, is, ic, soilm(ils:ile,jls:jle,2)
     &, 1., 0.)
      call getvara ('surf', iou, ln, is, ic, surf(ils:ile,jls:jle,1)
     &, 1., 0.)
# endif
# if defined uvic_ice
      call getvara ('tice', iou, ln, is, ic, tice(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('hice1', iou, ln, is, ic, hice(ils:ile,jls:jle,1)
     &, 1., 0.)
      call getvara ('hice2', iou, ln, is, ic, hice(ils:ile,jls:jle,2)
     &, 1., 0.)
      call getvara ('aice1', iou, ln, is, ic, aice(ils:ile,jls:jle,1)
     &, 1., 0.)
      call getvara ('aice2', iou, ln, is, ic, aice(ils:ile,jls:jle,2)
     &, 1., 0.)
      call getvara ('hsno1', iou, ln, is, ic, hsno(ils:ile,jls:jle,1)
     &, 1., 0.)
      call getvara ('hsno2', iou, ln, is, ic, hsno(ils:ile,jls:jle,2)
     &, 1., 0.)
# endif
# if defined uvic_ice_cpts
      do n=1,ncat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call getvara ('hseff1'//trim(a3), iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,1,n), 1., 0)
        call getvara ('hseff2//trim(a3)', iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,2,n), 1., 0)
        call getvara ('heff1//trim(a3)', iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,1,n), 1., 0)
        call getvara ('heff2//trim(a3)', iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,2,n), 1., 0)
        call getvara ('A1//trim(a3)', iou, ln, is, ic
     &,   A(ils:ile,jls:jle,1,n), 1., 0)
        call getvara ('A2//trim(a3)', iou, ln, is, ic
     &,   A(ils:ile,jls:jle,2,n), 1., 0)
        call getvara ('ts1//trim(a3)', iou, ln, is, ic
     &,   ts(ils:ile,jls:jle,1,n), 1., 0)
        call getvara ('ts2//trim(a3)', iou, ln, is, ic
     &,   ts(ils:ile,jls:jle,2,n), 1., 0)
      enddo
      do n=1,ntilay
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call getvara ('E1//trim(a3)', iou, ln, is, ic
     &,   E(ils:ile,jls:jle,1,n), 1., 0)
        call getvara ('E2//trim(a3)', iou, ln, is, ic
     &,   E(ils:ile,jls:jle,2,n), 1., 0)
      enddo
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
      call getvara ('strength1', iou, ln, is, ic
     &, strength(ils:ile,jls:jle,1), 1., 0.)
      call getvara ('strength2', iou, ln, is, ic
     &, strength(ils:ile,jls:jle,2), 1., 0.)
# endif
# if defined uvic_ice_evp
      call getvara ('uice', iou, ln, is, ic, uice(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('vice', iou, ln, is, ic, vice(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('su', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isu)
     &, 1., 0.)
      call getvara ('sv', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isv)
     &, 1., 0.)
      call getvara ('gu', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,igu)
     &, 1., 0.)
      call getvara ('gv', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,igv)
     &, 1., 0.)
      call getvara ('sig11n', iou, ln, is, ic, sig11n(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig11s', iou, ln, is, ic, sig11s(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig11e', iou, ln, is, ic, sig11e(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig11w', iou, ln, is, ic, sig11w(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig22n', iou, ln, is, ic, sig22n(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig22s', iou, ln, is, ic, sig22s(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig22e', iou, ln, is, ic, sig22e(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig22w', iou, ln, is, ic, sig22w(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig12n', iou, ln, is, ic, sig12n(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig12s', iou, ln, is, ic, sig12s(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig12e', iou, ln, is, ic, sig12e(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('sig12w', iou, ln, is, ic, sig12w(ils:ile,jls:jle)
     &, 1., 0.)
# endif

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)

      print*, '=> Atm restart read from ',trim(fname),' on ', stamp
#endif

      return
      end

      subroutine atm_rest_out (fname, ids, ide, jds, jde)
!=======================================================================
!     output routine for atmospheric restarts

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm

      implicit none

# include "param.h"
# include "atm.h"
# include "cembm.h"
# include "coord.h"
# include "csbc.h"
# if defined uvic_ice_evp
#  include "evp.h"
# endif
# include "grdvar.h"
# if defined uvic_ice
#  include "ice.h"
# endif
# include "mapsbc.h"
# include "tmngr.h"

      character(*) :: fname
      character(80) :: name
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, ids, ide, jds, jde, igs, ige, ig
      integer jgs, jge, jg, ils, ile, jls, jle, is(10), ic(10), it(10)
      integer iu(10), id_time, id_xt, id_xu, id_yt, id_yu, id_xt_e
      integer id_xu_e, id_yt_e, id_yu_e
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)
      real time, tmp(1)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      time = 0.
      ntrec = -1
      call openfile (name, time, ntrec, iou)

!-----------------------------------------------------------------------
!     set global write domain size
!-----------------------------------------------------------------------
      igs = 1
      ige = imt
      ig  = ige-igs+1
      jgs = 1
      jge = jmt
      jg  = jge-jgs+1

!-----------------------------------------------------------------------
!     start definitions
!-----------------------------------------------------------------------
      call redef (iou)

!-----------------------------------------------------------------------
!     define dimensions
!-----------------------------------------------------------------------
      call defdim ('time', iou, 0, id_time)
      call defdim ('xt', iou, ig, id_xt)
      call defdim ('yt', iou, jg, id_yt)
      call defdim ('xu', iou, ig, id_xu)
      call defdim ('yu', iou, jg, id_yu)
      call defdim ('xt_edges', iou, ig+1, id_xt_e)
      call defdim ('yt_edges', iou, jg+1, id_yt_e)
      call defdim ('xu_edges', iou, ig+1, id_xu_e)
      call defdim ('yu_edges', iou, jg+1, id_yu_e)

!-----------------------------------------------------------------------
!       define 1d data (t)
!-----------------------------------------------------------------------
        call defvar ('nats', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   ' ', ' ',' ')
        call defvar ('dayoyr', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   ' ', ' ',' ')
        call defvar ('itt', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   ' ', ' ',' ')
        call defvar ('irstdy', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   ' ', ' ',' ')
        call defvar ('msrsdy', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   ' ', ' ',' ')
        call defvar ('totaltime', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   ' ', ' ',' ')
        call defvar ('year', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   'year', ' ',' ')
        call defvar ('month', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   'month', ' ',' ')
        call defvar ('day', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   'day', ' ',' ')
        call defvar ('hour', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   'hour', ' ',' ')
        call defvar ('minute', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   'minute', ' ',' ')
        call defvar ('second', iou, 1, (/id_time/), 0., 0., ' ', 'D'
     &,   'second', ' ',' ')

!-----------------------------------------------------------------------
!     define 1d data (x, y or z)
!-----------------------------------------------------------------------
      call defvar ('xt', iou, 1, (/id_xt/), 0., 0., 'X', 'D'
     &, 'longitude of the t grid', 'grid_longitude', 'degrees_east')
      call defvar ('yt', iou, 1, (/id_yt/), 0., 0., 'Y', 'D'
     &, 'latitude of the t grid', 'grid_latitude', 'degrees_north')
      call defvar ('xu', iou, 1, (/id_xu/), 0., 0., 'X', 'D'
     &, 'longitude of the u grid', 'grid_longitude', 'degrees_east')
      call defvar ('yu', iou, 1, (/id_yu/), 0., 0., 'Y', 'D'
     &, 'latitude of the u grid', 'grid_latitude', 'degrees_north')
      call defvar ('xt_edges', iou, 1, (/id_xt_e/), 0., 0., 'X', 'D'
     &, 'longitude of t grid edges','grid_edge_longitude'
     &, 'degrees_east')
      call defvar ('yt_edges', iou, 1, (/id_yt_e/), 0., 0., 'Y', 'D'
     &, 'latitude of t grid edges', 'grid_edge_latitude'
     &, 'degrees_north')
      call defvar ('xu_edges', iou, 1, (/id_xu_e/), 0., 0., 'X', 'D'
     &, 'longitude of u grid edges', 'grid_edge_longitude'
     &, 'degrees_east')
      call defvar ('yu_edges', iou, 1, (/id_yu_e/), 0., 0., 'Y', 'D'
     &, 'latitude of u grid edges', 'grid_edge_latitude'
     &, 'degrees_north')

!-----------------------------------------------------------------------
!     define 3d data (x,y,t)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      it(3) = id_time
      iu(3) = id_time
      do n=1,nat
        if (n .lt. 1000) write(a3,'(i3)') n
        if (n .lt. 100) write(a3,'(i2)') n
        if (n .lt. 10) write(a3,'(i1)') n
        call defvar ('at1_'//trim(a3), iou , 3, it, -1.e6, 1.e6, ' '
     &,   'D', ' ', ' ', ' ')
        call defvar ('at2_'//trim(a3), iou , 3, it, -1.e6, 1.e6, ' '
     &,   'D', ' ', ' ', ' ')
      enddo
      call defvar ('precip', iou, 3, it,  -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sst', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sss', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# if defined uvic_embm_running_average || defined uvic_embm_astress
      call defvar ('rtbar', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('atbar', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
# if defined uvic_embm_adv_q
      call defvar ('avgp', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('accp', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
# if defined uvic_embm_astress
      call defvar ('awx', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('awy', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
# if defined uvic_embm_land
      call defvar ('soilm1', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('soilm2', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('surf', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
# if defined uvic_ice
      call defvar ('tice', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('hice1', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('hice2', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('aice1', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('aice2', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('hsno1', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('hsno2', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
# if defined uvic_ice_cpts
      do n=1,ncat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call defvar ('hseff1//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('hseff2//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('heff1//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('heff2//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('A1//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('A2//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('ts1//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('ts2//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
      enddo
      do n=1,ntilay
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call defvar ('E1//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
        call defvar ('E2//trim(a3)', iou, 3, it, -1.e6, 1.e6
     &,   ' ', 'D', ' ', ' ', ' ')
      enddo
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
      call defvar ('strength1', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('strength2', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
# if defined uvic_ice_evp
      call defvar ('uice', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('vice', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('su', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sv', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('gu', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('gv', iou, 3, iu, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig11n', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig11s', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig11e', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig11w', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig22n', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig22s', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig22e', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig22w', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig12n', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig12s', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig12e', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
      call defvar ('sig12w', iou, 3, it, -1.e6, 1.e6, ' ', 'D'
     &, ' ', ' ', ' ')
# endif
      call enddef (iou)

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global write domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)

!-----------------------------------------------------------------------
!     write 1d data (t)
!-----------------------------------------------------------------------
      tmp(1) = nats
      call putvara ('nats', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = dayoyr
      call putvara ('dayoyr', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = itt
      call putvara ('itt', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = iday(imodeltime)
      call putvara ('irstdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = msday(imodeltime)
      call putvara ('msrsdy', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = totaltime
      call putvara ('totaltime', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      call rdstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      tmp(1) = nyear
      call putvara ('year', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nmonth
      call putvara ('month', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nday
      call putvara ('day', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nhour
      call putvara ('hour', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nmin
      call putvara ('minute', iou, 1, (/1/), (/1/), tmp, 1., 0.)
      tmp(1) = nsec
      call putvara ('second', iou, 1, (/1/), (/1/), tmp, 1., 0.)

!-----------------------------------------------------------------------
!     write 1d data (x or y)
!-----------------------------------------------------------------------
      call putvara ('xt', iou, ig, (/1/), (/ig/), xt(igs:ige), 1., 0.)
      call putvara ('yt', iou, jg, (/1/), (/jg/), yt(jgs:jge), 1., 0.)
      call putvara ('xu', iou, ig, (/1/), (/ig/), xu(igs:ige), 1., 0.)
      call putvara ('yu', iou, jg, (/1/), (/jg/), yu(jgs:jge), 1., 0.)
      call edge_maker (1, xt_e, xt, dxt, xu, dxu, imt)
      call putvara ('xt_edges', iou, ig+1, (/1/), (/ig+1/)
     &, xt_e(igs:ige+1), 1., 0.)
      call edge_maker (1, yt_e, yt, dyt, yu, dyu, jmt)
      call putvara ('yt_edges', iou, jg+1, (/1/), (/jg+1/)
     &, yt_e(jgs:jge+1), 1., 0.)
      call edge_maker (2, xu_e, xt, dxt, xu, dxu, imt)
      call putvara ('xu_edges', iou, ig+1, (/1/), (/ig+1/)
     &, xu_e(igs:ige+1), 1., 0.)
      call edge_maker (2, yu_e, yt, dyt, yu, dyu, jmt)
      call putvara ('yu_edges', iou, jg+1, (/1/), (/jg+1/)
     &, yu_e(jgs:jge+1), 1., 0.)

!-----------------------------------------------------------------------
!     write 3d data (x,y,t)
!-----------------------------------------------------------------------
      is(1) = 1
      ic(1) = ile-ils+1
      is(2) = 1
      ic(2) = jle-jls+1
      is(3) = 1
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)
      do n=1,nat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call putvara('at1_'//trim(a3), iou, ln, is, ic
     &,   at(ils:ile,jls:jle,1,n), 1., 0.)
        call putvara('at2_'//trim(a3), iou, ln, is, ic
     &,   at(ils:ile,jls:jle,2,n), 1., 0.)
      enddo
      call putvara ('precip', iou, ln, is, ic, precip(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sst', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isst)
     &, 1., 0.)
      call putvara ('sss', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isss)
     &, 1., 0.)
# if defined uvic_embm_running_average || defined uvic_embm_astress
      call putvara ('rtbar', iou, ln, is, ic, rtbar(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('atbar', iou, ln, is, ic, atbar(ils:ile,jls:jle)
     &, 1., 0.)
# endif
# if defined uvic_embm_adv_q
      call putvara ('avgp', iou, ln, is, ic, avgp(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('accp', iou, ln, is, ic, accp(ils:ile,jls:jle)
     &, 1., 0.)
# endif
# if defined uvic_embm_astress
      call putvara ('awx', iou, ln, is, ic, awx(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('awy', iou, ln, is, ic, awy(ils:ile,jls:jle)
     &, 1., 0.)
# endif
# if defined uvic_embm_land
      call putvara ('soilm1', iou, ln, is, ic, soilm(ils:ile,jls:jle,1)
     &, 1., 0.)
      call putvara ('soilm2', iou, ln, is, ic, soilm(ils:ile,jls:jle,2)
     &, 1., 0.)
      call putvara ('surf', iou, ln, is, ic, surf(ils:ile,jls:jle,1)
     &, 1., 0.)
# endif
# if defined uvic_ice
      call putvara ('tice', iou, ln, is, ic, tice(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('hice1', iou, ln, is, ic, hice(ils:ile,jls:jle,1)
     &, 1., 0.)
      call putvara ('hice2', iou, ln, is, ic, hice(ils:ile,jls:jle,2)
     &, 1., 0.)
      call putvara ('aice1', iou, ln, is, ic, aice(ils:ile,jls:jle,1)
     &, 1., 0.)
      call putvara ('aice2', iou, ln, is, ic, aice(ils:ile,jls:jle,2)
     &, 1., 0.)
      call putvara ('hsno1', iou, ln, is, ic, hsno(ils:ile,jls:jle,1)
     &, 1., 0.)
      call putvara ('hsno2', iou, ln, is, ic, hsno(ils:ile,jls:jle,2)
     &, 1., 0.)
# endif
# if defined uvic_ice_cpts
      do n=1,ncat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call putvara ('hseff1'//trim(a3), iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,1,n), 1., 0)
        call putvara ('hseff2//trim(a3)', iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,2,n), 1., 0)
        call putvara ('heff1//trim(a3)', iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,1,n), 1., 0)
        call putvara ('heff2//trim(a3)', iou, ln, is, ic
     &,   hseff(ils:ile,jls:jle,2,n), 1., 0)
        call putvara ('A1//trim(a3)', iou, ln, is, ic
     &,   A(ils:ile,jls:jle,1,n), 1., 0)
        call putvara ('A2//trim(a3)', iou, ln, is, ic
     &,   A(ils:ile,jls:jle,2,n), 1., 0)
        call putvara ('ts1//trim(a3)', iou, ln, is, ic
     &,   ts(ils:ile,jls:jle,1,n), 1., 0)
        call putvara ('ts2//trim(a3)', iou, ln, is, ic
     &,   ts(ils:ile,jls:jle,2,n), 1., 0)
      enddo
      do n=1,ntilay
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call putvara ('E1//trim(a3)', iou, ln, is, ic
     &,   E(ils:ile,jls:jle,1,n), 1., 0)
        call putvara ('E2//trim(a3)', iou, ln, is, ic
     &,   E(ils:ile,jls:jle,2,n), 1., 0)
      enddo
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
      call putvara ('strength1', iou, ln, is, ic
     &, strength(ils:ile,jls:jle,1), 1., 0.)
      call putvara ('strength2', iou, ln, is, ic
     &, strength(ils:ile,jls:jle,2), 1., 0.)
# endif
# if defined uvic_ice_evp
      call putvara ('uice', iou, ln, is, ic, uice(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('vice', iou, ln, is, ic, vice(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('su', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isu)
     &, 1., 0.)
      call putvara ('sv', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,isv)
     &, 1., 0.)
      call putvara ('gu', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,igu)
     &, 1., 0.)
      call putvara ('gv', iou, ln, is, ic, sbcocn(ils:ile,jls:jle,igv)
     &, 1., 0.)
      call putvara ('sig11n', iou, ln, is, ic, sig11n(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig11s', iou, ln, is, ic, sig11s(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig11e', iou, ln, is, ic, sig11e(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig11w', iou, ln, is, ic, sig11w(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig22n', iou, ln, is, ic, sig22n(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig22s', iou, ln, is, ic, sig22s(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig22e', iou, ln, is, ic, sig22e(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig22w', iou, ln, is, ic, sig22w(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig12n', iou, ln, is, ic, sig12n(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig12s', iou, ln, is, ic, sig12s(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig12e', iou, ln, is, ic, sig12e(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('sig12w', iou, ln, is, ic, sig12w(ils:ile,jls:jle)
     &, 1., 0.)
# endif

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)

      print*, '=> Atm restart written to ',trim(fname),' on ', stamp
#endif

      return
      end
