      subroutine atm_snap_in (fname, ids, ide, jds, jde, imt, jmt, km
     &,                       nat, ncat, xt, yt, xu, yu, dxt, dyt, dxu
     &,                       dyu, ntrec, timunit, expnam, avgper
     &,                       relyr, stamp, at, esat, precip, evap
     &,                       outlwr, uplwr, upsens, dnswr, upltnt
     &,                       p_alb, a_alb, s_alb, elev
#if defined uvic_embm_adv_q
     &,                       wx, wy
#endif
#if defined uvic_embm_astress
     &,                       awx, awy
#endif
#if defined uvic_embm_land
     &,                       soilm, runoff, surf
#endif
#if defined uvic_ice_cpts
     &,                       ticen, hicen, aicen, hsnon
#elif defined uvic_ice
     &,                       tice, hice, aice, hsno
#endif
#if defined uvic_ice_evp
     &,                       uice, vice, xint, yint
#endif
#if defined rot_grid
     &,                       tlat, tlon, ulat, ulon
#endif
#if defined uvic_save_flxadj
     &,                       flxadj_t, flxadj_s
#endif
     &,                       kmt, mskhr)
!=======================================================================
!     output routine for atmospheric time averages or snapshots

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain
!     imt, jmt ...       = global array dimensions
!     xt, yt ...         = global axes
!     dxt, dyt ...       = grid widths
!     ntrec              = number of time record (zero if not defined)

!   outputs:
!     timunit            = time unit
!     expnam             = experiment name
!     avgper             = length of averaging period
!     relyr              = time in years
!     stamp              = time stamp
!     at, ...            = data to be read

!   local variables
!     igs, ige, jgs, jge = global read domain start and end indicies
!     ig, jg             = global read domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for read on each axis (x,y,t default)
!     ic                 = count for read on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm

      implicit none

      character(*) :: fname, timunit, expnam, stamp
      character(80) :: name, title, file_stamp
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, imt, jmt, km, nat, ncat
      integer ids, ide, jds, jde, igs, ige, ig, jgs, jge, jg, kgs, kge
      integer kg, ils, ile, jls, jle, kls, kle, is(10), ic(10), it(10)
      integer iu(10), id_time, id_xt, id_xu, id_yt, id_yu, id_cat
      integer id_xt_e, id_xu_e, id_yt_e, id_yu_e, id_cat_e
      integer kmt(ids:ide,jds:jde), mskhr(ids:ide,jds:jde)
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt(imt), xu(imt), yt(jmt), yu(jmt)
      real dxt(imt), dxu(imt), dyt(jmt), dyu(jmt)
      real avgper, at(ids:ide,jds:jde,nat)
      real esat(ids:ide,jds:jde), precip(ids:ide,jds:jde)
      real evap(ids:ide,jds:jde), outlwr(ids:ide,jds:jde)
      real uplwr(ids:ide,jds:jde), upsens(ids:ide,jds:jde)
      real dnswr(ids:ide,jds:jde), upltnt(ids:ide,jds:jde)
      real p_alb(ids:ide,jds:jde), a_alb(ids:ide,jds:jde)
      real s_alb(ids:ide,jds:jde), elev(ids:ide,jds:jde)
# if defined uvic_embm_adv_q
      real wx(ids:ide,jds:jde), wy(ids:ide,jds:jde)
# endif
# if defined uvic_embm_astress
      real awx(ids:ide,jds:jde), awy(ids:ide,jds:jde)
# endif
# if defined uvic_embm_land
      real soilm(ids:ide,jds:jde), runoff(ids:ide,jds:jde)
      real surf(ids:ide,jds:jde)
# endif
# if defined uvic_ice_cpts
      real cat(ncat), cat_e(ncat+1)
      real ticen(ids:ide,jds:jde,ncat), hicen(ids:ide,jds:jde,ncat)
      real aicen(ids:ide,jds:jde,ncat), hsnon(ids:ide,jds:jde,ncat)
# elif defined uvic_ice
      real tice(ids:ide,jds:jde), hice(ids:ide,jds:jde)
      real aice(ids:ide,jds:jde), hsno(ids:ide,jds:jde)
# endif
# if defined uvic_ice_evp
      real uice(ids:ide,jds:jde), vice(ids:ide,jds:jde)
      real xint(ids:ide,jds:jde), yint(ids:ide,jds:jde)
# endif
# if defined rot_grid
      real tlat(ids:ide,jds:jde), tlon(ids:ide,jds:jde)
      real ulat(ids:ide,jds:jde), ulon(ids:ide,jds:jde)
# endif
# if defined uvic_save_flxadj
      real flxadj_t(ids:ide,jds:jde), flxadj_s(ids:ide,jds:jde)
# endif
      real rkmt(ids:ide,jds:jde), rmskhr(ids:ide,jds:jde)
      real relyr, tmp(1), xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1)
      real yu_e(jmt+1)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      i = 0
      call openfile (name, relyr, i, iou)

!-----------------------------------------------------------------------
!     get global attributes
!-----------------------------------------------------------------------
      call getglobal (iou, fname, title, expnam, timunit)

!-----------------------------------------------------------------------
!     get global read domain size (may be less than global domain)
!-----------------------------------------------------------------------
      call getaxis('xt', iou, imt, xt, igs, ige, 1., 0.)
      call getaxis('xu', iou, imt, xu, igs, ige, 1., 0.)
      ig  = ige-igs+1
      call getaxis('yt', iou, jmt, yt, jgs, jge, 1., 0.)
      call getaxis('yu', iou, jmt, yu, igs, ige, 1., 0.)
      jg  = jge-jgs+1
      kgs = 1
      kge = ncat
      kg  = kge-kgs+1

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global read domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)
      kls = max(1,kgs)
      kle = min(ncat,kge)

!-----------------------------------------------------------------------
!     read 1d data (t)
!-----------------------------------------------------------------------
      call getvara ('time', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      relyr = tmp(1)
      call getvara ('year', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nyear = tmp(1)
      call getvara ('month', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nmonth = tmp(1)
      call getvara ('day', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nday = tmp(1)
      call getvara ('hour', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nhour = tmp(1)
      call getvara ('minute', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nmin = tmp(1)
      call getvara ('second', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nsec = tmp(1)
      call mkstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      call getvara ('period', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      avgper = tmp(1)

!-----------------------------------------------------------------------
!     read 2d data (x,y)
!-----------------------------------------------------------------------
      is(1) = ils-igs+1
      ic(1) = ile-ils+1
      is(2) = jls-jgs+1
      ic(2) = jle-jls+1
      ln = ic(1)*ic(2)
      call getvara ('elev', iou, ln, is, ic, elev(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('kmt', iou, ln, is, ic, rkmt(ils:ile,jls:jle)
     &, 1., 0.)
      kmt(ils:ile,jls:jle) = rkmt(ils:ile,jls:jle)
      call getvara ('mskhr', iou, ln, is, ic, rmskhr(ils:ile,jls:jle)
     &, 1., 0.)
      mskhr(ils:ile,jls:jle) = rmskhr(ils:ile,jls:jle)
# if defined rot_grid
      call getvara ('tlat', iou, ln, is, ic, tlat(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('tlon', iou, ln, is, ic, tlon(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('ulat', iou, ln, is, ic, ulat(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('ulon', iou, ln, is, ic, ulon(ils:ile,jls:jle)
     &, 1., 0.)
# endif

!-----------------------------------------------------------------------
!     read 3d data (x,y,t)
!-----------------------------------------------------------------------
      is(1) = ils-igs+1
      ic(1) = ile-ils+1
      is(2) = jls-jgs+1
      ic(2) = jle-jls+1
      is(3) = ntrec
      ic(3) = 1
      call getvara ('sat', iou, ln, is, ic, at(ils:ile,jls:jle,1)
     &, 1., 273.15)
      call getvara ('shum', iou, ln, is, ic, at(ils:ile,jls:jle,2)
     &, 1., 0.)
      do n=3,nat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call getvara ('at_'//trim(a3), iou, ln, is, ic
     &,   at(ils:ile,jls:jle,n), 1., 0.)
      enddo
      call getvara ('esat', iou, ln, is, ic, esat(ils:ile,jls:jle)
     &, 1., 273.15)
      call getvara ('precip', iou, ln, is, ic, precip(ils:ile,jls:jle)
     &, 100., 0.)
      call getvara ('evap', iou, ln, is, ic, evap(ils:ile,jls:jle)
     &, 100., 0.)
      call getvara ('outlwr', iou, ln, is, ic, outlwr(ils:ile,jls:jle)
     &, 0.001, 0.)
      call getvara ('uplwr', iou, ln, is, ic, uplwr(ils:ile,jls:jle)
     &, 0.001, 0.)
      call getvara ('upsens', iou, ln, is, ic, upsens(ils:ile,jls:jle)
     &, 0.001, 0.)
      call getvara ('dnswr', iou, ln, is, ic, dnswr(ils:ile,jls:jle)
     &, 0.001, 0.)
      call getvara ('upltnt', iou, ln, is, ic, upltnt(ils:ile,jls:jle)
     &, 0.001, 0.)
      call getvara ('p_alb', iou, ln, is, ic, p_alb(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('a_alb', iou, ln, is, ic, a_alb(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('s_alb', iou, ln, is, ic, s_alb(ils:ile,jls:jle)
     &, 1., 0.)
# if defined uvic_embm_adv_q
      call getvara ('wx', iou, ln, is, ic, wx(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('wy', iou, ln, is, ic, wy(ils:ile,jls:jle)
     &, 0.01, 0.)
# endif
# if defined uvic_embm_astress
      call getvara ('awx', iou, ln, is, ic, awx(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('awy', iou, ln, is, ic, awy(ils:ile,jls:jle)
     &, 0.01, 0.)
# endif
# if defined uvic_embm_land
      call getvara ('soilm', iou, ln, is, ic, soilm(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('runoff', iou, ln, is, ic, runoff(ils:ile,jls:jle)
     &, 0.1, 0.)
      call getvara ('surf', iou, ln, is, ic, surf(ils:ile,jls:jle)
     &, 1., 273.15)
# endif
# if defined uvic_ice && !defined uvic_ice_cpts
      call getvara ('tice', iou, ln, is, ic, tice(ils:ile,jls:jle)
     &, 1., 273.15)
      call getvara ('hice', iou, ln, is, ic, hice(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('aice', iou, ln, is, ic, aice(ils:ile,jls:jle)
     &, 1., 0.)
      call getvara ('hsno', iou, ln, is, ic, hsno(ils:ile,jls:jle)
     &, 0.01, 0.)
# endif
# if defined uvic_ice_evp
      call getvara ('uice', iou, ln, is, ic, uice(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('vice', iou, ln, is, ic, vice(ils:ile,jls:jle)
     &, 0.01, 0.)
      call getvara ('xint', iou, ln, is, ic, xint(ils:ile,jls:jle)
     &, 0.1, 0.)
      call getvara ('yint', iou, ln, is, ic, yint(ils:ile,jls:jle)
     &, 0.1, 0.)
# endif
# if defined uvic_save_flxadj
      call getvara ('flxadj_t', iou, ln, is, ic
     &, flxadj_t(ils:ile,jls:jle), 41852., 0.)

      call getvara ('flxadj_s', iou, ln, is, ic
     &, flxadj_s(ils:ile,jls:jle), 10., 0.)

# endif
# if defined uvic_ice_cpts

!-----------------------------------------------------------------------
!     read 4d data (x,y,c,t)
!-----------------------------------------------------------------------
      is(1) = ils-igs+1
      ic(1) = ile-ils+1
      is(2) = jls-jgs+1
      ic(2) = jle-jls+1
      is(3) = kls-kgs+1
      ic(3) = kle-kls+1
      is(4) = ntrec
      ic(4) = 1
      ln = ic(1)*ic(2)*ic(3)*ic(4)
      call getvara ('ticen', iou, ln, is, ic
     &, ticen(ils:ile,jls:jle,kls:kle), 1., 273.15)
      call getvara ('hicen', iou, ln, is, ic
     &, hicen(ils:ile,jls:jle,kls:kle), 0.01, 0.)
      call getvara ('aicen', iou, ln, is, ic
     &, aicen(ils:ile,jls:jle,kls:kle), 1., 0.)
      call getvara ('hsnon', iou, ln, is, ic
     &, hsnon(ils:ile,jls:jle,kls:kle), 0.01, 0.)
# endif

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)
#endif

      return
      end

      subroutine atm_snap_out (fname, ids, ide, jds, jde, imt, jmt, km
     &,                        nat, ncat, xt, yt, xu, yu, dxt, dyt, dxu
     &,                        dyu, ntrec, timunit, expnam, avgper
     &,                        relyr, stamp, at, esat, precip
     &,                        evap, outlwr, uplwr, upsens, dnswr
     &,                        upltnt, p_alb, a_alb,  s_alb, elev
#if defined uvic_embm_adv_q
     &,                        wx, wy
#endif
#if defined uvic_embm_astress
     &,                        awx, awy
#endif
#if defined uvic_embm_land
     &,                        soilm, runoff, surf
#endif
#if defined uvic_ice_cpts
     &,                        ticen, hicen, aicen, hsnon
#elif defined uvic_ice
     &,                        tice, hice, aice, hsno
#endif
#if defined uvic_ice_evp
     &,                        uice, vice, xint, yint
#endif
#if defined rot_grid
     &,                        tlat, tlon, ulat, ulon
#endif
#if defined uvic_save_flxadj
     &,                        flxadj_t, flxadj_s
#endif
     &,                        kmt, mskhr)
!=======================================================================
!     output routine for atmospheric time averages or snapshots

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, is, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain
!     imt, jmt ...       = global array dimensions
!     xt, yt ...         = global axes
!     dxt, dyt ...       = grid widths
!     ntrec              = number of time record (zero if not defined)
!     timunit            = time unit
!     expnam             = experiment name
!     avgper             = length of averaging period
!     relyr              = time in years
!     stamp              = time stamp
!     at, ...            = data to be written

!   outputs:
!     ntrec              = number of time record in file

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm

      implicit none

      character(*) :: fname, timunit, expnam, stamp
      character(80) :: name, title, file_stamp
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, imt, jmt, km, nat, ncat
      integer ids, ide, jds, jde, igs, ige, ig, jgs, jge, jg, kgs, kge
      integer kg, ils, ile, jls, jle, kls, kle, is(10), ic(10), it(10)
      integer iu(10), id_time, id_xt, id_xu, id_yt, id_yu
      integer id_cat, id_xt_e, id_xu_e, id_yt_e, id_yu_e, id_cat_e
      integer kmt(ids:ide,jds:jde), mskhr(ids:ide,jds:jde)
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt(imt), xu(imt), yt(jmt), yu(jmt)
      real dxt(imt), dxu(imt), dyt(jmt), dyu(jmt)
      real avgper, at(ids:ide,jds:jde,nat)
      real esat(ids:ide,jds:jde), precip(ids:ide,jds:jde)
      real evap(ids:ide,jds:jde), outlwr(ids:ide,jds:jde)
      real uplwr(ids:ide,jds:jde), upsens(ids:ide,jds:jde)
      real dnswr(ids:ide,jds:jde), upltnt(ids:ide,jds:jde)
      real p_alb(ids:ide,jds:jde), a_alb(ids:ide,jds:jde)
      real s_alb(ids:ide,jds:jde), elev(ids:ide,jds:jde)
# if defined uvic_embm_adv_q
      real wx(ids:ide,jds:jde), wy(ids:ide,jds:jde)
# endif
# if defined uvic_embm_astress
      real awx(ids:ide,jds:jde), awy(ids:ide,jds:jde)
# endif
# if defined uvic_embm_land
      real soilm(ids:ide,jds:jde), runoff(ids:ide,jds:jde)
      real surf(ids:ide,jds:jde)
# endif
# if defined uvic_ice_cpts
      real cat(ncat), cat_e(ncat+1)
      real ticen(ids:ide,jds:jde,ncat), hicen(ids:ide,jds:jde,ncat)
      real aicen(ids:ide,jds:jde,ncat), hsnon(ids:ide,jds:jde,ncat)
# elif defined uvic_ice
      real tice(ids:ide,jds:jde), hice(ids:ide,jds:jde)
      real aice(ids:ide,jds:jde), hsno(ids:ide,jds:jde)
# endif
# if defined uvic_ice_evp
      real uice(ids:ide,jds:jde), vice(ids:ide,jds:jde)
      real xint(ids:ide,jds:jde), yint(ids:ide,jds:jde)
# endif
# if defined rot_grid
      real tlat(ids:ide,jds:jde), tlon(ids:ide,jds:jde)
      real ulat(ids:ide,jds:jde), ulon(ids:ide,jds:jde)
# endif
# if defined uvic_save_flxadj
      real flxadj_t(ids:ide,jds:jde), flxadj_s(ids:ide,jds:jde)
# endif
      real rkmt(ids:ide,jds:jde), rmskhr(ids:ide,jds:jde)
      real relyr, tmp(1), xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1)
      real yu_e(jmt+1)

      logical defined

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      defined = .true.
      name = fname
      if (ntrec .eq. 0) defined = .false.
      call openfile (name, relyr, ntrec, iou)

!-----------------------------------------------------------------------
!     set global write domain size (may be less than global domain)
!-----------------------------------------------------------------------
# if defined cyclic
      igs = 2
      ige = imt-1
# else
      igs = 1
      ige = imt
# endif
      ig  = ige-igs+1
      jgs = 1
      jge = jmt
      do j=2,jmt
        if (yt(j-1) .lt. -90. .and. yt(j) .gt. -90.) jgs = j
        if (yt(j-1) .lt.  90. .and. yt(j) .gt. 90.) jge = j-1
      enddo
      jg  = jge-jgs+1
      kgs = 1
      kge = ncat
      kg  = kge-kgs+1

      if (.not. defined .or. ntrec .eq. 0) then

!-----------------------------------------------------------------------
!       start definitions
!-----------------------------------------------------------------------
        call redef (iou)

!-----------------------------------------------------------------------
!       set global attributes
!-----------------------------------------------------------------------
        if (avgper .gt. 1.e-6) then
          title = 'Time Average'
        else
          title = 'Snapshot'
        endif
        call putglobal (iou, name, title, expnam, timunit)

!-----------------------------------------------------------------------
!       define dimensions
!-----------------------------------------------------------------------
        call defdim ('time', iou, 0, id_time)
        call defdim ('xt', iou, ig, id_xt)
        call defdim ('yt', iou, jg, id_yt)
        call defdim ('xu', iou, ig, id_xu)
        call defdim ('yu', iou, jg, id_yu)
        call defdim ('xt_edges', iou, ig+1, id_xt_e)
        call defdim ('yt_edges', iou, jg+1, id_yt_e)
        call defdim ('xu_edges', iou, ig+1, id_xu_e)
        call defdim ('yu_edges', iou, jg+1, id_yu_e)
#  if defined uvic_ice_cpts
        call defdim ('cat', iou, kg, id_cat)
        call defdim ('cat_edges', iou, kg+1, id_cat_e)
#  endif

!-----------------------------------------------------------------------
!       define 1d data (t)
!-----------------------------------------------------------------------
        call defvar ('time', iou, 1, (/id_time/), 0., 0., 'T', 'D'
     &,   'time since initial condition', 'time', trim(timunit))
        call defvar ('year', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'year', ' ',' ')
        call defvar ('month', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'month', ' ',' ')
        call defvar ('day', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'day', ' ',' ')
        call defvar ('hour', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'hour', ' ',' ')
        call defvar ('minute', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'minute', ' ',' ')
        call defvar ('second', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'second', ' ',' ')
        if (avgper .gt. 1.e-6) then
          call defvar ('period', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,     'averaging period', ' ',' ')
        endif

!-----------------------------------------------------------------------
!       define 1d data (x, y or z)
!-----------------------------------------------------------------------
        call defvar ('xt', iou, 1, (/id_xt/), 0., 0., 'X', 'F'
     &,   'longitude of the t grid', 'grid_longitude', 'degrees_east')
        call defvar ('yt', iou, 1, (/id_yt/), 0., 0., 'Y', 'F'
     &,   'latitude of the t grid', 'grid_latitude', 'degrees_north')
        call defvar ('xu', iou, 1, (/id_xu/), 0., 0., 'X', 'F'
     &,   'longitude of the u grid', 'grid_longitude', 'degrees_east')
        call defvar ('yu', iou, 1, (/id_yu/), 0., 0., 'Y', 'F'
     &,   'latitude of the u grid', 'grid_latitude', 'degrees_north')
        call defvar ('xt_edges', iou, 1, (/id_xt_e/), 0., 0., 'X', 'F'
     &,   'longitude of t grid edges','grid_edge_longitude'
     &,   'degrees_east')
        call defvar ('yt_edges', iou, 1, (/id_yt_e/), 0., 0., 'Y', 'F'
     &,   'latitude of t grid edges', 'grid_edge_latitude'
     &,   'degrees_north')
        call defvar ('xu_edges', iou, 1, (/id_xu_e/), 0., 0., 'X', 'F'
     &,   'longitude of u grid edges', 'grid_edge_longitude'
     &,   'degrees_east')
        call defvar ('yu_edges', iou, 1, (/id_yu_e/), 0., 0., 'Y', 'F'
     &,   'latitude of u grid edges', 'grid_edge_latitude'
     &,   'degrees_north')
#  if defined uvic_ice_cpts
        call defvar ('cat', iou, 1, id_cat, 0., 0., 'C', 'F'
     &,   'ice category', ' ',' ')
        call defvar ('cat_edges', iou, 1, id_cat_e, 0., 0., 'C', 'F'
     &,   'ice category edges', ' ',' ')
#  endif

!-----------------------------------------------------------------------
!       define 2d data (x,y)
!-----------------------------------------------------------------------
        it(1) = id_xt
        iu(1) = id_xu
        it(2) = id_yt
        iu(2) = id_yu
        call defvar ('elev', iou, 2, it, -1.e4, 1.e4, ' ', 'F'
     &,   'land elevation and ocean depth', 'surface_altitude', 'm')
        call defvar ('kmt', iou, 2, it, 0., 1000., ' ', 'I'
     &,   'ocean depth grid level', ' ' ,'level')
        call defvar ('mskhr', iou, 2, it, 0., 1000., ' ', 'I'
     &,   'horizontal region mask', ' ' ,'number')
# if defined rot_grid
        call defvar ('tlat', iou, 2, it, -1.e4, 1.e4, ' ', 'F'
     &,   'tracer grid latitude', 'latitude', 'deg')
        call defvar ('tlon', iou, 2, it, -1.e4, 1.e4, ' ', 'F'
     &,   'tracer grid longitude', 'longitude', 'deg')
        call defvar ('ulat', iou, 2, iu, -1.e4, 1.e4, ' ', 'F'
     &,   'velocity grid latitude', 'latitude', 'deg')
        call defvar ('ulon', iou, 2, iu, -1.e4, 1.e4, ' ', 'F'
     &,   'velocity grid longitude', 'longitude', 'deg')
# endif

!-----------------------------------------------------------------------
!       define 3d data (x,y,t)
!-----------------------------------------------------------------------
        it(1) = id_xt
        iu(1) = id_xu
        it(2) = id_yt
        iu(2) = id_yu
        it(3) = id_time
        iu(3) = id_time
        call defvar ('sat', iou, 3, it, 100., 400., ' ', 'F'
     &,   'sea surface atmospheric temperature', 'air_temperature', 'K')
        call defvar ('shum', iou,3, it, -100., 100., ' ', 'F'
     &,   'atmospheric surface specific humidity', 'specific_humidity'
     &,   '1')
        do n=3,nat
          if (n .lt. 1000) write(a3,'(i3)') n
          if (n .lt. 100) write(a3,'(i2)') n
          if (n .lt. 10) write(a3,'(i1)') n
          call defvar ('at_'//trim(a3), iou ,3, it, -100., 100., ' '
     &,     'F', 'unknown tracer '//trim(a3)
     &,     'unknown_tracer_'//trim(a3), 'unknown')
        enddo
        call defvar ('esat', iou, 3, it, 100., 400., ' ', 'F'
     &,   'atmospheric surface temperature', 'air_temperature', 'K')
        call defvar ('precip', iou, 3, it, 0., 1., ' ', 'F'
     &,   'precipitation (includes snow in water equivalent)'
     &,   'precipitation_mass_flux_density', 'kg m-2 s-1')
        call defvar ('evap', iou, 3, it, 0., 1., ' ', 'F'
     &,   'upward evaporation plus sublimation'
     &,   'surface_upward_water_vapour_mass_flux_density', 'kg m-2 s-1')
        call defvar ('outlwr', iou, 3, it, -1000., 1000., ' ', 'F'
     &,   'outgoing longwave from top of atmosphere'
     &,   'TOA_upward_longwave_heat_flux_density', 'W m-2')
        call defvar ('uplwr', iou, 3, it, -1000., 1000., ' ', 'F'
     &,   'surface upward longwave'
     &,   'surface_emitted_longwave_heat_flux_density', 'W m-2')
        call defvar ('upsens', iou, 3, it, -1000., 1000., ' ', 'F'
     &,   'surface upward sensible heat'
     &,   'surface_upward_sensible_heat_flux_density', 'W m-2')
        call defvar ('dnswr', iou, 3, it, -1000., 1000., ' ', 'F'
     &,   'net surface downward short wave (absorbed)'
     &,   'surface_downward_shortwave_heat_flux_density', 'W/m^2')
        call defvar ('upltnt', iou, 3, it, -1000., 1000., ' ', 'F'
     &,   'surface upward latent heat'
     &,   'surface_upward_latent_heat_flux_density', 'W m-2')
        call defvar ('p_alb', iou, 3, it, 0., 1., ' ', 'F'
     &,   'planetary albedo', 'planetary_albedo', '1')
        call defvar ('a_alb', iou, 3, it, 0., 1., ' ', 'F'
     &,   'atmospheric albedo', 'atmospheric_albedo', '1')
        call defvar ('s_alb', iou, 3, it, 0., 1., ' ', 'F'
     &,   'surface albedo', 'surface_albedo', '1')
# if defined uvic_embm_adv_q
        call defvar ('wx', iou, 3, iu, -1000., 1000., ' ', 'F'
     &,   'eastward wind for advection of humidity'
     &,   'eastward_wind', 'm s-1')
        call defvar ('wy', iou, 3, iu, -1000., 1000., ' ', 'F'
     &,   'northward wind for advection of humidity'
     &,   'northward_wind', 'm s-1')
# endif
# if defined uvic_embm_astress
        call defvar ('awx', iou, 3, iu, -1000., 1000., ' ', 'F'
     &,   'eastward wind  anomoly', 'eastward_wind_anomaly', 'm s-1')
        call defvar ('awy', iou, 3, iu, -1000., 1000., ' ', 'F'
     &,   'northward wind anomoly', 'northward_wind_anomaly', 'm s-1')
# endif
# if defined uvic_embm_land
        call defvar ('soilm', iou, 3, it, 0., 100., ' ', 'F'
     &,   'soil moisture', 'soil_water_content', 'kg m-2')
        call defvar ('runoff', iou, 3, it, 0., 1., ' ', 'F'
     &,   'run off', 'surface_runoff_mass_flux_density', 'kg m-2 s-1')
        call defvar ('surf', iou, 3, it, 100., 400., ' ', 'F'
     &,   'surface temperature below atmosphere (water, soil, ice)'
     &,   'surface_temperature', 'K')
# endif
# if defined uvic_save_flxadj
        call defvar ('flxadj_t', iou, 3, it, -1.e4, 1.e4, ' ', 'F'
     &,   'surface downward heat flux adjustment'
     &,   ' ', 'W m-2')
        call defvar ('flxadj_s', iou,3, it, -100., 100., ' ', 'F'
     &,   'surface downward salt flux adjustment'
     &,   ' ', 'kg m-2 s-1')
# endif
# if defined uvic_ice && !defined uvic_ice_cpts
        call defvar ('tice', iou, 3, it, 100., 400., ' ', 'F'
     &,   'surface ice temperature', 'surface_sea_ice_temperature'
     &,   'K')
        call defvar ('hice', iou, 3, it, 0., 1.e6, ' ', 'F'
     &,   'ice thickness', 'sea_ice_thickness', 'm')
        call defvar ('aice', iou, 3, it, 0., 100., ' ', 'F'
     &,   'ice area fraction (includes land ice area fraction)'
     &,   'sea_ice_area_fraction', '1')
        call defvar ('hsno', iou, 3, it, -0.01, 1.e6, ' ', 'F'
     &,   'surface snow thickness', 'surface_snow_thickness', 'm')
# endif
# if defined uvic_ice_evp
        call defvar ('uice', iou, 3, iu, -1000., 1000., ' ', 'F'
     &,   'eastward ice velocity', 'sea_ice_eastward_velocity'
     &,   'm s-1')
        call defvar ('vice', iou, 3, iu, -1000., 1000., ' ', 'F'
     &,   'northward ice velocity', 'sea_ice_northward_velocity'
     &,   'm s-1')
        call defvar ('xint', iou, 3, iu, -1.e8, 1.e9, ' ', 'F'
     &,   'eastward ice interaction', 'eastward_stress_of_sea_ice'
     &,   'Pa')
        call defvar ('yint', iou, 3, iu, -1.e8, 1.e9, ' ', 'F'
     &,   'northward ice interaction', 'northward_stress_of_sea_ice'
     &,   'Pa')
# endif
# if defined uvic_ice_cpts

!-----------------------------------------------------------------------
!       define time dependent 4d data (x,y,c,t)
!-----------------------------------------------------------------------
        it(1) = id_xt
        iu(1) = id_xu
        it(2) = id_yt
        iu(2) = id_yu
        it(3) = id_cat
        iu(3) = id_cat
        it(4) = id_time
        iu(4) = id_time
        call defvar ('ticen', iou, 4, it, 100., 400., ' ', 'F'
     &,   'multi-catagory ice temperature'
     &,   'surface_sea_ice_temperature', 'K')
        call defvar ('hicen', iou, 4, it, 0., 1.e6, ' ', 'F'
     &,   'multi-catagory ice thickness', 'sea_ice_thickness', 'm')
        call defvar ('aicen', iou, 4, it, 0., 100., ' ', 'F'
     &,   'multi-catagory ice area', 'sea_ice_area_fraction', '1')
        call defvar ('hsnon', iou, 4, it, 0., 1.e6, ' ', 'F'
     &,   'multi-catagory snow thickness', 'surface_snow_thickness'
     &,   'm')
# endif

!-----------------------------------------------------------------------
!       end definitions
!-----------------------------------------------------------------------
        call enddef (iou)
        if (ntrec .eq. 0 ) ntrec = 1

      endif

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global write domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)
      kls = max(1,kgs)
      kle = min(ncat,kge)

!-----------------------------------------------------------------------
!     write 1d data (t)
!-----------------------------------------------------------------------
      tmp(1) = relyr
      call putvara ('time', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      call rdstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      tmp(1) = nyear
      call putvara ('year', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nmonth
      call putvara ('month', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nday
      call putvara ('day', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nhour
      call putvara ('hour', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nmin
      call putvara ('minute', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nsec
      call putvara ('second', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      if (avgper .gt. 1.e-6) then
        tmp(1) = avgper
        call putvara ('period', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      endif

      if (ntrec .eq. 1 .or. .not. defined) then

!-----------------------------------------------------------------------
!       write 1d data (x, y or z)
!-----------------------------------------------------------------------
        call putvara ('xt', iou, ig, (/1/), (/ig/), xt(igs:ige), 1., 0.)
        call putvara ('yt', iou, jg, (/1/), (/jg/), yt(jgs:jge), 1., 0.)
        call putvara ('xu', iou, ig, (/1/), (/ig/), xu(igs:ige), 1., 0.)
        call putvara ('yu', iou, jg, (/1/), (/jg/), yu(jgs:jge), 1., 0.)
        call edge_maker (1, xt_e, xt, dxt, xu, dxu, imt)
        call putvara ('xt_edges', iou, ig+1, (/1/), (/ig+1/)
     &,   xt_e(igs:ige+1), 1., 0.)
        call edge_maker (1, yt_e, yt, dyt, yu, dyu, jmt)
        call putvara ('yt_edges', iou, jg+1, (/1/), (/jg+1/)
     &,   yt_e(jgs:jge+1), 1., 0.)
        call edge_maker (2, xu_e, xt, dxt, xu, dxu, imt)
        call putvara ('xu_edges', iou, ig+1, (/1/), (/ig+1/)
     &,   xu_e(igs:ige+1), 1., 0.)
        call edge_maker (2, yu_e, yt, dyt, yu, dyu, jmt)
        call putvara ('yu_edges', iou, jg+1, (/1/), (/jg+1/)
     &,   yu_e(jgs:jge+1), 1., 0.)
#  if defined uvic_ice_cpts
        do n=1,ncat
          cat(n) = float(n)
          cat_e(n) = cat(n) - 0.5
        enddo
        cat_e(ncat+1) = cat(ncat) + 0.5
        call putvara ('cat', iou, kg, (/1/), (/kg/), cat(kgs:kge), 1.
     &,   0.)
        call putvara ('cat_edges', iou, kg+1, (/1/), (/kg+1/)
     &,   cat_e(kgs:kge+1), 1., 0.)
#  endif

!-----------------------------------------------------------------------
!       write 2d data (x,y)
!-----------------------------------------------------------------------
        is(1) = ils-igs+1
        ic(1) = ile-ils+1
        is(2) = jls-jgs+1
        ic(2) = jle-jls+1
        ln = ic(1)*ic(2)
        call putvara ('elev', iou, ln, is, ic, elev(ils:ile,jls:jle)
     &,   0.01, 0.)
        rkmt(ils:ile,jls:jle) = kmt(ils:ile,jls:jle)
        call putvara ('kmt', iou, ln, is, ic, rkmt(ils:ile,jls:jle)
     &,   1., 0.)
        rmskhr(ils:ile,jls:jle) = mskhr(ils:ile,jls:jle)
        call putvara ('mskhr', iou, ln, is, ic, rmskhr(ils:ile,jls:jle)
     &,   1., 0.)
#  if defined rot_grid
        call putvara ('tlat', iou, ln, is, ic, tlat(ils:ile,jls:jle)
     &,   1., 0.)
        call putvara ('tlon', iou, ln, is, ic, tlon(ils:ile,jls:jle)
     &,   1., 0.)
        call putvara ('ulat', iou, ln, is, ic, ulat(ils:ile,jls:jle)
     &,   1., 0.)
        call putvara ('ulon', iou, ln, is, ic, ulon(ils:ile,jls:jle)
     &,   1., 0.)
#  endif

      endif

!-----------------------------------------------------------------------
!     write 3d data (x,y,t)
!-----------------------------------------------------------------------
      is(1) = ils-igs+1
      ic(1) = ile-ils+1
      is(2) = jls-jgs+1
      ic(2) = jle-jls+1
      is(3) = ntrec
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)
      call putvara('sat', iou, ln, is, ic, at(ils:ile,jls:jle,1)
     &, 1., 273.15)
      call putvara('shum', iou, ln, is, ic, at(ils:ile,jls:jle,2)
     &, 1., 0.)
      do n=3,nat
        if (n .lt. 1000) write(a3, '(i3)') n
        if (n .lt. 100) write(a3, '(i2)') n
        if (n .lt. 10) write(a3, '(i1)') n
        call putvara('at_'//trim(a3), iou, ln, is, ic
     &,   at(ils:ile,jls:jle,n), 1., 0.)
      enddo
      call putvara('esat', iou, ln, is, ic, esat(ils:ile,jls:jle)
     &, 1., 273.15)
      call putvara ('precip', iou, ln, is, ic, precip(ils:ile,jls:jle)
     &, 10., 0.)
      call putvara ('evap', iou, ln, is, ic, evap(ils:ile,jls:jle)
     &, 10., 0.)
      call putvara ('outlwr', iou, ln, is, ic, outlwr(ils:ile,jls:jle)
     &, 0.001, 0.)
      call putvara ('uplwr', iou, ln, is, ic, uplwr(ils:ile,jls:jle)
     &, 0.001, 0.)
      call putvara ('upsens', iou, ln, is, ic, upsens(ils:ile,jls:jle)
     &, 0.001, 0.)
      call putvara ('dnswr', iou, ln, is, ic, dnswr(ils:ile,jls:jle)
     &, 0.001, 0.)
      call putvara ('upltnt', iou, ln, is, ic, upltnt(ils:ile,jls:jle)
     &, 0.001, 0.)
      call putvara ('p_alb', iou, ln, is, ic, p_alb(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('a_alb', iou, ln, is, ic, a_alb(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('s_alb', iou, ln, is, ic, s_alb(ils:ile,jls:jle)
     &, 1., 0.)
# if defined uvic_embm_adv_q
      call putvara ('wx', iou, ln, is, ic, wx(ils:ile,jls:jle)
     &, 0.01, 0.)
      call putvara ('wy', iou, ln, is, ic, wy(ils:ile,jls:jle)
     &, 0.01, 0.)
# endif
# if defined uvic_embm_astress
      call putvara ('awx', iou, ln, is, ic, awx(ils:ile,jls:jle)
     &, 0.01, 0.)
      call putvara ('awy', iou, ln, is, ic, awy(ils:ile,jls:jle)
     &, 0.01, 0.)
# endif
# if defined uvic_embm_land
      call putvara ('soilm', iou, ln, is, ic, soilm(ils:ile,jls:jle)
     &, 0.01, 0.)
      call putvara ('runoff', iou, ln, is, ic, runoff(ils:ile,jls:jle)
     &, 10., 0.)
      call putvara ('surf', iou, ln, is, ic, surf(ils:ile,jls:jle)
     &, 1., 273.15)
# endif
# if defined uvic_ice && !defined uvic_ice_cpts
      call putvara ('tice', iou, ln, is, ic, tice(ils:ile,jls:jle)
     &, 1., 273.15)
      call putvara ('hice', iou, ln, is, ic, hice(ils:ile,jls:jle)
     &, 0.01, 0.)
      call putvara ('aice', iou, ln, is, ic, aice(ils:ile,jls:jle)
     &, 1., 0.)
      call putvara ('hsno', iou, ln, is, ic, hsno(ils:ile,jls:jle)
     &, 0.01, 0.)
# endif
# if defined uvic_ice_evp
      call putvara ('uice', iou, ln, is, ic, uice(ils:ile,jls:jle)
     &, 0.01, 0.)
      call putvara ('vice', iou, ln, is, ic, vice(ils:ile,jls:jle)
     &, 0.01, 0.)
      call putvara ('xint', iou, ln, is, ic, xint(ils:ile,jls:jle)
     &, 0.1, 0.)
      call putvara ('yint', iou, ln, is, ic, yint(ils:ile,jls:jle)
     &, 0.1, 0.)
# endif
# if defined uvic_save_flxadj
      call putvara('flxadj_t', iou, ln, is, ic
     &, flxadj_t(ils:ile,jls:jle), 41852., 0.)

      call putvara('flxadj_s', iou, ln, is, ic
     &, flxadj_s(ils:ile,jls:jle), 10., 0.)
# endif
# if defined uvic_ice_cpts

!-----------------------------------------------------------------------
!     write 4d data (x,y,c,t)
!-----------------------------------------------------------------------
      is(1) = ils-igs+1
      ic(1) = ile-ils+1
      is(2) = jls-jgs+1
      ic(2) = jle-jls+1
      is(3) = kls-kgs+1
      ic(3) = kle-kls+1
      is(4) = ntrec
      ic(4) = 1
      ln = ic(1)*ic(2)*ic(3)*ic(4)
      call putvara ('ticen', iou, ln, is, ic
     &, ticen(ils:ile,jls:jle,kls:kle), 1., 273.15)
      call putvara ('hicen', iou, ln, is, ic
     &, hicen(ils:ile,jls:jle,kls:kle), 0.01, 0.)
      call putvara ('aicen', iou, ln, is, ic
     &, aicen(ils:ile,jls:jle,kls:kle), 1., 0.)
      call putvara ('hsnon', iou, ln, is, ic
     &, hsnon(ils:ile,jls:jle,kls:kle), 0.01, 0.)
# endif

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)
#endif

      return
      end
