      subroutine setembm

!=======================================================================
!     initialize the energy-moisture balance model
!     based on code by a.fanning

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================
#if defined uvic_embm

      implicit none

# include "param.h"
# include "solve.h"
# include "switch.h"
# include "coord.h"
# include "grdvar.h"
# include "atm.h"
# include "cembm.h"
# if defined uvic_ice
#  include "ice.h"
#  if defined uvic_ice_cpts
#   include "thermo.h"
#  endif
#  if defined uvic_ice_evp
#   include "evp.h"
#  endif
# endif
# include "riv.h"
# include "tmngr.h"
# include "levind.h"
# include "mapsbc.h"
# include "csbc.h"
# include "scalar.h"

      integer i, ie, ii, iou, is, j, je, jj, js, jz, k, m, n, nsolve, nu
      integer nsum, ntrec

      real astl, asul, ctl, dn_s(nat), de_s(nat), dlam, dphi, dtatms
      real dte, dyz, eccice, esatm_s, grarea, saltmax, si, solins_s
      real stl, sul, t1, time, tlat_s, ulat_s, yz_max, yz_min, wz, zrel

# if defined uvic_ice_cpts
      logical rowflg(ncat) ! flag for computing ridg. matrix row
      real Hi(ncat)        ! ice thickness (m)
      real Hmean(ncat)     ! a dummy variable at setup (m)
# endif

      namelist /tsteps/ dtts, dtuv, dtsf, dtatm, dtatms, namix, segtim
      namelist /co2/    co2ccn, co2yri, co2yrf, co2rate, co2for
      namelist /paleo/  pyear
      namelist /ice/    niats, nivts, dampice, tsno, hsno_max
      namelist /adv_q/  diffactor

      is = 1
      ie = imt
      js = 1
      je = jmt

!-----------------------------------------------------------------------
!     define default parameters
!-----------------------------------------------------------------------

!     set defaults namelist co2
      co2ccn     = 350
      co2yri     = 1.e20
      co2yrf     = 1.e20
      co2rate    = 0.
      co2for     = 5.77e03

!     set defaults namelist paleo
      pyear      = 0.

!     set defaults namelist ice
      niats      = 1
      nivts      = 1
      dampice    = 5.
      tsno       = -5.
      hsno_max   = 1000.
# if defined uvic_embm_snow_transient
!     no limit
      hsno_max   = 1.e30
# endif

!     set defaults namelist adv_q
      diffactor  = 50.

!     set other stuff
      solarconst = 1.368e6
      scatter    = 0.30
      pass       = 1. - scatter

      ssp        = 1.e6
      cdatm      = 1.e-3
      cpatm      = 1.004e7
      ht         = 8.4e5
      hq         = 1.8e5
      rhoatm     = 1.250e-3
      rlapse     = 6.5e-5

      rhoocn     = 1.035
      esocn      = 5.347e-5
      vlocn      = 2.501e10
      socn       = 34.9e-3

      cdice      = 5.5e-3
      rhoice     = 0.913
      rhosno     = 0.330
      esice      = 5.347e-5
      slice      = 2.835e10
      flice      = 3.34e9
      condice    = 2.1656e5

      soilmax    = 15.
      eslnd      = 5.347e-5
      cs_alb     = 0.08
      ice_calb   = 0.45
      sno_calb   = 0.35

      nivc       = 1
      dtatms     = 1800.
      ns         = 30

      call getunit (iou, 'control.in', 'f s r')
      read  (iou, tsteps, end=100)
100   continue
      write (stdout, tsteps)
      call relunit (iou)

      call getunit (iou, 'control.in', 'f s r')
      read  (iou, co2, end=102)
102   continue
      write (stdout,co2)
      call relunit (iou)
      co2ccni = co2ccn

      call getunit (iou, 'control.in', 'f s r')
      read  (iou, paleo, end=103)
103   continue
      write (stdout,paleo)
      call relunit (iou)

      call getunit (iou, 'control.in', 'f s r')
      read  (iou, ice, end=104)
104   continue
      write (stdout,ice)
      call relunit (iou)
# if defined uvic_embm_adv_q

      call getunit (iou, 'control.in', 'f s r')
      read  (iou, adv_q, end=105)
105   continue
      write (stdout,adv_q)
      call relunit (iou)
# endif

# if !defined uvic_mom
!     if not coupled set ocean and coupling time to atmosphere
      dtts = dtatm
      segtim = dtts*secday

# endif
#  if defined uvic_embm_explicit || defined uvic_embm_explicit_q
      if (dtatms .ne. 0.) ns = nint(dtatm/dtatms)

#  endif
# if defined time_averages
      if (mod(timavgint, segtim) .gt. 1.e-6 .and. timavgint .gt. 0.)
     &  then
        t1 = nint(timavgint/segtim)*segtim
        if (t1 .lt. segtim) t1 = segtim
        write (stdout,'(/,(1x,a))')
     &    '==> Warning: "timavgint" does not contain an integral number'
     &,   '              of coupling time steps "segtim".              '
        write (stdout,*) '              (changed "timavgint" from '
     &, timavgint,' days to ', t1,' days to insure this condition)'
        timavgint = t1
      endif
      if (timavgint .eq. 0.) then
        write (stdout,'(/,(1x,a))')
     &    '==> Warning: averaging interval "timavgint" = 0. implies no '
     &,   '             averaging when "time_averages" is enabled      '
      endif
      if (timavgint .gt. timavgper) then
        write (stdout,'(/,(1x,a))')
     &    '==> Warning: the interval "timavgint" exceeds the averaging '
     &,   '             period "timavgper" for option "time_averages"  '
      endif
      if (timavgint .lt. timavgper) then
        write (stdout,'(/,(1x,a))')
     &    '==> Warning: averaging period "timavgper" exceeds interval  '
     &,   '             "timavgint". Setting timavgper = timavgint     '
        timavgper = timavgint
      endif
      if (timavgper .eq. 0.) then
        write (stdout,'(/,(1x,a))')
     &    '==> Warning: the averaging period "timavgper" is zero. The  '
     &,   '             average will be over only one time step!       '
      endif
      write (stdout,'(/,1x,a,f10.2,a,/,1x,a,f10.2,a)')
     &  '==> Time averages will be written every ', timavgint, ' days, '
     &, '    with an averaging period of         ', timavgper, ' days. '

# endif
!-----------------------------------------------------------------------
!     calculate the relative CO2 forcing term
!-----------------------------------------------------------------------

      call co2forc

# if !defined uvic_embm_co2_lin && !defined uvic_embm_co2_exp
      write(stdout,*)
      write(stdout,*) 'CO2 concentration (versus today) =',co2ccn/350.
      write(stdout,*) 'Yields radiative forcing (W/m2) = ',anthro*1.e-3

# endif
!-----------------------------------------------------------------------
!     calculate the expansion coefficients for Berger's solution for
!     the year of the initial conditions
!-----------------------------------------------------------------------

# if defined uvic_embm_berger_transient
      pyear = year0 + relyr
# endif
      call berger (pyear)

!-----------------------------------------------------------------------
!     calculate latitude dependent variables
!-----------------------------------------------------------------------

      radian = 360./(2.*pi)
# if defined rot_grid
!     calculate coriolis parameter
      do j=1,jmt
        do i=1,imt
          fcor(i,j) = 2.*omega*sin(ulat(i,j)/radian)
        enddo
      enddo

      do j=1,jmt
        do i=1,imt
          tlat_s = tlat(i,j)
          stl = sin(tlat(i,j)/radian)
          ctl = cos(tlat(i,j)/radian)
          ulat_s = ulat(i,j)
          sul = sin(ulat(i,j)/radian)
# else
      do j=1,jmt
!-----------------------------------------------------------------------
!       calculate coriolis parameter
!-----------------------------------------------------------------------

        fcor(j) = 2.*omega*sine(j)
        tlat_s = yt(j)
        stl = sin(yt(j)/radian)
        ctl = cos(yt(j)/radian)
        ulat_s = yu(j)
        sul = sin(yu(j)/radian)
# endif

          astl = abs(stl)
          asul = abs(sul)

!-----------------------------------------------------------------------
!         calculate annual average insolation
!-----------------------------------------------------------------------

          solins_s = 0.
          do n=1, 365
            call insolation (float(n), tlat_s, si)
            solins_s = solins_s + si
          enddo
          solins_s = (solins_s + 0.25*si) /365.25

!-----------------------------------------------------------------------
!         planetary and atmospheric emissivity
!-----------------------------------------------------------------------

          esatm_s = 5.57e-5*(0.8666 + 0.0408*stl - 0.2553*stl**2
     &               - 0.4660*stl**3 + 0.9877*stl**4 + 2.0257*stl**5
     &               - 2.3374*stl**6 - 3.1990*stl**7 + 2.8581*stl**8
     &               + 1.6070*stl**9 -1.2685*stl**10)

!-----------------------------------------------------------------------
!         diffusivities
!-----------------------------------------------------------------------

          do n=1,nat
# if defined uvic_embm_adv_q
            if (n .eq. 1) then
              dn_s(1) = 3.e10*(1.3 - 1.1*asul**2 + .15*sul)
              de_s(1) = 3.e10*(1.3 - 1.1*astl**2 + .15*stl)
            endif
            if (n .eq. 2) then
              dn_s(2) = 1.e10
              de_s(2) = 1.e10
            endif
# else
            if (n .eq. 1) then
              dn_s(1) = 2.4e10*(1.1 - 0.8*asul**3 + .2*sul)
              de_s(1) = 2.4e10*(1.1 - 0.8*astl**3 + .2*stl)
            endif
            if (n .eq. 2) then
              dn_s(2) = 1.7e10*(2. - 17.4*asul + 117.2*asul**2
     &                - 274.*asul**3 + 258.2*asul**4 - 85.8*asul**5)
              de_s(2) = 1.7e10*(2. - 17.4*astl + 117.2*astl**2
     &                - 274.*astl**3 + 258.2*astl**4 - 85.8*astl**5)
            endif
# endif
            if (n .ge. 3) then
              dn_s(n) = 1.e10
              de_s(n) = 1.e10
            endif
          enddo
# if defined rot_grid

!-----------------------------------------------------------------------
!         load grid latitude and longitude dependent arrays
!-----------------------------------------------------------------------

          solins(i,j) = solins_s
          esatm(i,j) = esatm_s
          do n=1,nat
            dn(i,j,n) = dn_s(n)
            de(i,j,n) = de_s(n)
          enddo
        enddo
      enddo
# else

!-----------------------------------------------------------------------
!         load grid latitude dependent arrays
!-----------------------------------------------------------------------

        solins(j) = solins_s
        esatm(j) = esatm_s
#  if defined uvic_embm_2d_diff || defined uvic_embm_land
        do i=1,imt
          do n=1,nat
            dn(i,j,n) = dn_s(n)
            de(i,j,n) = de_s(n)
          enddo
        enddo
#  else
        do n=1,nat
          dn(j,n) = dn_s(n)
          de(j,n) = de_s(n)
        enddo
#  endif
#  if defined uvic_embm_explicit || defined uvic_embm_explicit_q
        filter(j) = 1. - sin(yt(j)/radian)**ns
#  endif
      enddo
# endif

!     set solver parameters

      nsolve = 0
# if !defined uvic_embm_explicit || defined uvic_embm_explicit_q
#  if defined uvic_global_sums
      itin(1:nat)  = 500        ! max solver iterations
      epsin(1:nat) = 1.e-14    ! solver tolerance
#  else
      itin(1:nat)  = 100        ! max solver iterations
      epsin(1) = 1.e-4         ! solver tolerance
      epsin(2:nat) = 1.0e-11    ! solver tolerance
#  endif
# endif
# if defined uvic_embm_essl
      nsolve = nsolve + 1
      iparm(2)= 2               ! solver method
      iparm(3)= 7               ! if iparm(2)=3 then iparm(3)=k,5<k<10
      iparm(4)= 4               ! preconditioning
      iparm(5)= 2               ! stopping criterion
      call einfo(0)
      call errset(2110,256,-1,0,1,2110)
# endif
# if defined uvic_embm_slap
      nsolve = nsolve + 1
# endif
# if defined uvic_embm_mgrid
      nsolve = nsolve + 1
      levelin = 20              ! max coarse grid level
# endif
# if defined uvic_embm_adi
      nsolve = nsolve + 1
      itin(1:nat) = 1
      itout(1:nat) = 1
# endif
# if defined uvic_embm_explicit && !defined uvic_embm_explicit_q
      nsolve = nsolve + 1
# endif
      if (nsolve .ne. 1) then
        write(*,*) '==> Error: more or less than one solver defined.'
        write(*,*) '           Use only one of uvic_embm_adi,'
     &,   ' uvic_embm_mgrid, uvic_embm_slap, uvic_embm_essl or'
     &,   ' uvic_embm_explicit'
        stop '=>setembm'
      endif

# if defined uvic_ice
!-----------------------------------------------------------------------
!     check latent heats will sum to zero
!-----------------------------------------------------------------------

      if (slice .ne. vlocn + flice) write (stdout,'(/,a)')
     &   '==> Warning: changing latent heat of fusion to conserve heat'
        flice = slice - vlocn

# else

          write (stdout,*) '==> Warning: ice model is not defined.'
     &,     ' heat flux may be limited to prevent freezing sst.'

# endif
!-----------------------------------------------------------------------
!     initialize the river model
!-----------------------------------------------------------------------

      call rivinit

# if defined uvic_embm_solve2x || defined uvic_embm_solve2y
!-----------------------------------------------------------------------
!     calculate grid ratio for the coarse grid atmospheric solver
!-----------------------------------------------------------------------

#  if defined uvic_embm_solve2y
      do jj=1,jjmtm2
        j = jj*2
#  else
      do j=2,jmtm1
#  endif
#  if defined uvic_embm_solve2x
        do ii=1,iimtm2
          i = ii*2
#  else
        do i=2,imtm1
#  endif
          grarea = dxt(i)*dyt(j)*cst(j)
#  if defined uvic_embm_solve2x
          grarea = grarea + dxt(i+1)*dyt(j)*cst(j)
#  endif
#  if defined uvic_embm_solve2y
          grarea = grarea + dxt(i)*dyt(j+1)*cst(j+1)
#  endif
#  if defined uvic_embm_solve2x && defined uvic_embm_solve2y
          grarea = grarea + dxt(i+1)*dyt(j+1)*cst(j+1)
#  endif
          gr(i,j) = dxt(i)*dyt(j)*cst(j)/grarea
#  if defined uvic_embm_solve2x
          gr(i+1,j) = dxt(i+1)*dyt(j)*cst(j)/grarea
#  endif
#  if defined uvic_embm_solve2y
          gr(i,j+1) = dxt(i)*dyt(j+1)*cst(j+1)/grarea
#  endif
#  if defined uvic_embm_solve2x && defined uvic_embm_solve2y
          gr(i+1,j+1) = dxt(i+1)*dyt(j+1)*cst(j+1)/grarea
#  endif
        enddo
      enddo

# endif
!-----------------------------------------------------------------------
!     calculate grid terms for the atmospheric solver
!-----------------------------------------------------------------------

# if defined uvic_embm_solve2y
      do jj=2,jjmtm2
        j = jj*2
        wtj(j) = 0.5*dyt(j)/(dyt(j-1)+dyt(j)+dyt(j+1)+dyt(j+2))
        wtj(j-1) = 0.5*dyt(j+1)/(dyt(j-1)+dyt(j)+dyt(j+1)+dyt(j+2))
        ygrd(j) = dyt(j)*cst(j)/(cst(j)*dyt(j)+cst(j+1)*dyt(j+1))
        ygrd(j+1) = dyt(j+1)*cst(j+1)/(cst(j+1)*dyt(j+1)+cst(j)*dyt(j))
      enddo

      do j=2,jmtm2
        dsgrd(j) = csu(j-1)/((dyt(j)+dyt(j-1))*csu(j)*
     &             (dyu(j)+dyu(j+1)))
        dngrd(j) = csu(j+1)/((dyt(j+2)+dyt(j+1))*csu(j)*
     &             (dyu(j)+dyu(j+1)))
        asgrd(j) = csu(j-1)/(2.*csu(j)*(dyu(j)+dyu(j+1)))
        angrd(j) = csu(j+1)/(2.*csu(j)*(dyu(j)+dyu(j+1)))
# else
      do j=2,jmtm1
        dsgrd(j) = csu(j-1)/(dyu(j-1)*cst(j)*dyt(j))
        dngrd(j) = csu(j)/(dyu(j)*cst(j)*dyt(j))
        asgrd(j) = csu(j-1)/(2.*cst(j)*dyt(j))
        angrd(j) = csu(j)/(2.*cst(j)*dyt(j))
# endif
      enddo
# if defined uvic_embm_solve2x

      do ii=1,iimtm2
        i = ii*2
        wti(i) = 0.5*dxt(i)/(dxt(i-1)+dxt(i)+dxt(i+1)+dxt(i+2))
        wti(i+1) = 0.5*dxt(i+1)/(dxt(i-1)+dxt(i)+dxt(i+1)+dxt(i+2))
        xgrd(i) = dxt(i)/(dxt(i) + dxt(i+1))
        xgrd(i+1) = dxt(i+1)/(dxt(i) + dxt(i+1))
      enddo

      do i=2,imtm2
        dwgrd(i) = 1./((dxt(i) + dxt(i-1))*(dxu(i) + dxu(i+1)))
        degrd(i) = 1./((dxt(i+2) + dxt(i+1))*(dxu(i) + dxu(i+1)))
        azgrd(i) = 1./(2.*(dxu(i) + dxu(i+1)))
# else
      do i=2,imtm1
        dwgrd(i) = 1./(dxu(i-1)*dxt(i))
        degrd(i) = 1./(dxu(i)*dxt(i))
        azgrd(i) = 1./(2.*dxt(i))
# endif
      enddo

# if defined uvic_ice_cpts
#  if !defined uvic_ice_cpts5 && !defined uvic_ice_cpts10 && defined roth_press
      write(*,*) 'you are strongly discouraged from using roth_press'
      stop ' with fewer than 5 ice categories'
#  endif
!-----------------------------------------------------------------------
!     setup the vectors identifying first and last layer in each bin
!-----------------------------------------------------------------------

#  if defined uvic_ice_cpts1
      nilay(1) = 4
#   if defined uvic_ice_cpts3 || defined uvic_ice_cpts5 || defined uvic_ice_cpts10
!     stop if uvic_ice_cpts1 and uvic_ice_cpts3, 5 or 10 is defined
      stop 'single and multiple ice categories options defined'
#   endif
#  elif defined uvic_ice_cpts3
      nilay(1) = 2
      nilay(2) = 4
      nilay(3) = 8

      hstar(1) = 50.
      hstar(2) = 250.
#  elif defined uvic_ice_cpts5
      nilay(1) = 2
      nilay(2) = 4
      nilay(3) = 4
      nilay(4) = 8
      nilay(5) = 8

      hstar(1) = 40.
      hstar(2) = 90.
      hstar(3) = 200.
      hstar(4) = 350.
#  elif defined uvic_ice_cpts10
      nilay(1) = 2
      nilay(2) = 2
      do n=3,5
        nilay(n) = 4
      enddo
      do n=6,ncat
        nilay(n) = 8
      enddo

      hstar(1) = 25.
      hstar(2) = 50.
      hstar(3) = 75.
      hstar(4) = 100.
      hstar(5) = 140.
      hstar(6) = 190.
      hstar(7) = 330.
      hstar(8) = 500.
      hstar(9) = 700.
#  else
      stop 'setup needed in setembm.F for ',ncat,' ice categories'
#  endif

      hstar(0) = 10.
      hstar(ncat) = 200000. !should not be used, make it real big anyway

      nsum = 0.
      do n=1,ncat
        nsum = nsum + nilay(n)
      enddo
      if (nsum .ne. ntilay) stop 'the sum of nilay must be ntilay'

      print*, 'cpts ice model set up with ncat=',ncat,
     &  ' ice categories and 1 open water category'
      print*, '   category intervals are: ',(hstar(n), n=0, ncat)
      print*, '   layers per category are:',(nilay(n), n=1, ncat)

!     minimum allowable fract
      asmall(0) = a0small
      do n=1,ncat
        asmall(n) = aismall
      enddo

!     matrix used to assist in heat transf. from cat i to j
      do n=1,ncat                     ! if nilay = { 2,4,8 }
        do m=1,n
          ncrel(n,m) = 1              !    ncrel = | 1 2 4 |
        enddo                         !            | 1 1 2 |
        do m=n,ncat                   !            | 1 1 1 |
          ncrel(n,m) = nilay(m)/nilay(n)
        enddo
      enddo
!     vectors identifying first and last layer in each bin
      layer1(1) = 1                   ! if nilay = { 2,4,8 }
      layern(1) = nilay(1)            !   layer1 = { 1,3,7 }
      do n=2,ncat                     !   layern = { 2,6,16}
        layer1(n) = layern(n-1) + 1
        layern(n) = layern(n-1) + nilay(n)
      enddo

!     default ridg matrices,  comp. all rows (rowflg=true)
!     assume ice thickness is mean of range for n < ncat
!     and 1m thicker than lower limit for ncat
      do n=1,ncat
        do k=1,ncat
          M_def(n,k) = 0.
          N_def(n,k) = 0.
          HN_def(n,k) = 0.
        enddo
        rowflg(n) = .true.
        if (n .lt. ncat) then
          Hi(n) = 0.5*(hstar(n-1) + hstar(n))
        else
           Hi(ncat) = hstar(ncat-1) + 1.*centi
        endif
      enddo
      call comp_matrices (rowflg, Hi, Hmean, M_def, N_def, HN_def)

!     setup the salinity profile and the melting temperature
!     for each layer
      salnew = 5.
!      saltmax = 3.2
      saltmax = 5.

      do n=1,ncat
        do k=1,nilay(n)
          zrel = (k-0.5)/nilay(n)
          saltz(k,n) = saltmax/2.*(1.+sin(3.14159*(
     &                     zrel**(0.40706205/(zrel+0.57265966))-0.5)))
        enddo
        saltz(nilay(n)+1,n) = saltmax
        do k=1,nilay(n)
          tmelz(k,n) = -saltz(k,n)*alpha
        enddo
        print*, 'Category ',n
        write(*,'(A17,10(1x,f8.3))')
     &    '   salt profile:',(saltz(k,n),k=1,nilay(n)+1)
        write(*,'(A17,10(1x,f8.3))')
     &    '   melt temp:   ',(tmelz(k,n),k=1,nilay(n))
      enddo

! it would be nice to do this in a parameter statement
! because these are not variable
      rflice = flice*rhoice    ! specific latent heat of fushion ice
      rflsno = flice*rhosno    ! specific latent heat of fushion snow
      rslice = slice*rhoice    ! specific latent heat of sublim ice
      rslsno = slice*rhosno    ! specific latent heat of sublim snow

      rvlice = vlocn*rhoice    ! specific latent heat of vapor*rhoice
      rvlsno = vlocn*rhosno    ! specific latent heat of vapor*rhosno
      rcpice = cpice*rhoice    ! specific heat capacity of fresh ice
      rcpsno = cpsno*rhosno    ! specific heat capacity of snow

      rcpatm = rhoatm*cpatm
      rvlatm = rhoatm*vlocn
      rslatm = rhoatm*slice

      gamma  = rflice*ALPHA    ! heat capacity C=Cpi+gamma*salinity/T**2

# endif
# if defined uvic_old_albedo
!-----------------------------------------------------------------------
!     set surface coalbedo to zero
!-----------------------------------------------------------------------

      do j=1,jmt
        do i=1,imt
          s_calb(i,j) = 0.
        enddo
      enddo
      pcfactor = 0.
# else
!-----------------------------------------------------------------------
!     read constant annual average snow-free surface albedos
!-----------------------------------------------------------------------

!     read surface albedo
      time = 0.
      ntrec = 1
      call openfile ('data/sf_alb.nc', time, ntrec, iou)
      call getvara ('sf_alb', iou, imt*jmt, (/1,1/)
     &,            (/imt,jmt/), s_calb, 1., 0.)
      call closefile (iou)
# endif

# if defined rot_grid

!     calculate new grid yz for zonal averages
      yz_max = -90.
      yz_min = 90.
      do j=1,jmt
        do i=1,imt
          yz_max = max(yz_max, tlat(i,j))
          yz_min = min(yz_min, tlat(i,j))
        enddo
      enddo
      dyz = (yz_max-yz_min)/float(jmz-3)
      yz_max = yz_max + dyz
      yz_min = yz_min - dyz
      dyz = (yz_max-yz_min)/float(jmzm1)
      do jz=1,jmz
        yz(jz) = yz_min + float(jz-1)*dyz
# else
      do jz=1,jmt
# endif
        area_z(jz) = epsln
        s_alb_z(jz) = 0.
      enddo

!     convert to coalbedo and find weights for zonal averaging
      do j=1,jmt
        do i=1,imt
          s_calb(i,j) = c1 - s_calb(i,j)
# if defined rot_grid
          do jz=1,jmzm1
            if (tlat(i,j).ge.yz(jz) .and. tlat(i,j).le.yz(jz+1)) then
              wt_zonal(i,j) = jz + (yz(jz)-tlat(i,j))/(yz(jz)-yz(jz+1))
            endif
          enddo
# endif
       enddo
      enddo

!     calculate zonal average snow free albedo. s_calb may then be
!     changed but the constant zonal average, s_alb_z should not.
      do j=2,jmtm1
        do i=2,imtm1
          grarea = dxt(i)*dyt(j)*cst(j)
# if defined rot_grid
          jz = int(wt_zonal(i,j))
          wz = wt_zonal(i,j) - jz
          area_z(jz) = area_z(jz) + (c1-wz)*grarea
          area_z(jz+1) = area_z(jz+1) + wz*grarea
          s_alb_z(jz) = s_alb_z(jz) + (c1-wz)*s_calb(i,j)*grarea
          s_alb_z(jz+1) = s_alb_z(jz+1) + wz*s_calb(i,j)*grarea
        enddo
      enddo
      do jz=1,jmz
!       ensure zonal surface albedo is between zero and one.
        s_alb_z(jz) = c1 - min(max(s_alb_z(jz)/area_z(jz), 0.), c1)
      enddo
# else
          area_z(j) = area_z(j) + grarea
          s_alb_z(j) = s_alb_z(j) + s_calb(i,j)*grarea
        enddo
!       ensure zonal surface albedo is between zero and one.
        s_alb_z(j) = c1 - min(max(s_alb_z(j)/area_z(j), 0.), c1)
      enddo
# endif
!     ensure pass is between zero and one.
      pass = min(max(pass, epsln), c1-epsln)

!-----------------------------------------------------------------------
!     set initial conditions or read a restart
!-----------------------------------------------------------------------

      newcoef(:,:) = .true.

      if (init) then
        nats = namix
        irstdy = 0
        msrsdy = 0
        at(:,:,:,:) = 0.
        at(:,:,:,1) = 10.
        at(:,:,:,2) = 0.01
# if defined uvic_embm_adv_q
        avgp(:,:) = 0.
# endif
# if defined uvic_embm_astress
        awx(:,:) = 0.
        awy(:,:) = 0.
# endif
# if defined uvic_embm_land
        soilm(:,:,1) = 0.
        soilm(:,:,2) = 0.
        surf(:,:,1) = 0.
# endif
# if defined uvic_ice
        hice(:,:,1) = 0.
        hice(:,:,2) = 0.
        aice(:,:,1) = 0.
        aice(:,:,2) = 0.
        tice(:,:) = 0.
        hsno(:,:,1) = 0.
        hsno(:,:,2) = 0.
# endif
# if defined uvic_ice_cpts
        hseff(:,:,:,:) = 0.
        A(:,:,:,:) = 0.
        heff(:,:,:,:) = 0.
        ts(:,:,:,:) = 0.
        E(:,:,:,:) = 0.
# endif
# if defined uvic_ice_cpts_roth_press && defined uvic_ice_cpts
        strength(:,:,:) = 0
# endif
# if defined uvic_ice_evp
        uice(:,:) = 0.
        vice(:,:) = 0.
        sig11n(:,:) = 0.
        sig11e(:,:) = 0.
        sig11s(:,:) = 0.
        sig11w(:,:) = 0.
        sig22n(:,:) = 0.
        sig22e(:,:) = 0.
        sig22s(:,:) = 0.
        sig22w(:,:) = 0.
        sig12n(:,:) = 0.
        sig12e(:,:) = 0.
        sig12s(:,:) = 0.
        sig12w(:,:) = 0.
# endif
#   if defined ubc_cidm
        subssh(:,:)    = 0.
        subprecip(:,:) = 0.
        subpsno(:,:)   = 0.
        subhsno(:,:,:) = 0.
#   endif
# if !defined uvic_embm_explicit || defined uvic_embm_explicit_q
        bv(:) = 0.
        xv(:) = 0.
# endif
# if defined uvic_embm_slap
        raux(:) = 0.
        iaux(:) = 0
# endif
# if defined uvic_embm_essl
        aux1(:,:,:) = 0.
        aux2(:) = 0.
        rparm(1) = 0.
        iparm(1) = 0
# endif
      else

# if defined uvic_old_rest
        call init_atm (is, ie, js, je)
# else
        call atm_rest_in ('restatm.nc', is, ie, js, je)
# endif

      endif

# if !defined uvic_mom

!-----------------------------------------------------------------------
!     initialize the time manager with specified initial conditions
!     time, user reference time, model time, and how long to integrate.
!-----------------------------------------------------------------------

      call tmngri (year0, month0, day0, hour0, min0, sec0
     &,            ryear, rmonth, rday, rhour, rmin, rsec
     &,            irstdy, msrsdy, runlen, rununits, rundays, dtatm)
# endif
      flux(:,:,:) = 0.0

!-----------------------------------------------------------------------
!     read any atmospheric forcing data
!-----------------------------------------------------------------------

      call setatm

#if defined uvic_embm_astress

!-----------------------------------------------------------------------
!     read average air temperature
!-----------------------------------------------------------------------

      time = 0.
      ntrec = 1
      call openfile ('data/tbar.nc', time, ntrec, iou)
      call getvara ('tbar', iou, imt*jmt, (/1,1/)
     &,            (/imt,jmt/), tbar, 1., 273.15)
      call closefile (iou)
#endif
#if defined uvic_embm_no_mountains

!-----------------------------------------------------------------------
!     zero land elevations
!-----------------------------------------------------------------------

      elev(:,:) = 0.
#else

!-----------------------------------------------------------------------
!     read land elevations
!-----------------------------------------------------------------------

      time = 0.
      ntrec = 1
      call openfile ('data/elev.nc', time, ntrec, iou)
      call getvara ('elev', iou, imt*jmt, (/1,1/)
     &,            (/imt,jmt/), elev, 0.01, 0.)
      call closefile (iou)
!     check for negative elevations
      do j=1,jmt
        do i=1,imt
          if (elev(i,j) .lt. 0.) elev(i,j) = 0.
        enddo
      enddo
#endif

!-----------------------------------------------------------------------
!     calculate albedo
!-----------------------------------------------------------------------

      call albedo

# if defined uvic_embm_running_average || defined uvic_embm_astress
      if (init) then
!-----------------------------------------------------------------------
!       initialize running annual averages
!-----------------------------------------------------------------------

        totaltime = 0.
        do j=1,jmt
          do i=1,imt
            atbar(i,j) = 0.
#  if defined uvic_embm_astress
            rtbar(i,j) = tbar(i,j)
#  else
            rtbar(i,j) = 0.
#  endif
          enddo
        enddo
      endif

# endif
# if defined uvic_ice_evp
!-----------------------------------------------------------------------
!     calculate B grid ice masks
!-----------------------------------------------------------------------

      do j=1,jmtm1
        do i=1,imtm1
          tm(i,j) = 0.
          um(i,j) = 0.
          if (kmt(i,j) .gt. 0) tm(i,j) = 1.
          if (kmu(i,j) .gt. 0) um(i,j) = 1.
        enddo
      enddo
      call embmbc (tm)
      call embmbc (um)

      dlam = dxu(int(imt/2))/100.
      dphi = dyu(int(jmt/2))/100.
      diff1 = 0.004
      diff1 = diff1*dlam
      diff2 = diff1*dlam**2

# endif
# if defined uvic_ice_evp
!----------------------------------------------------------------------
!     initialize elastic viscous plastic variables
!-----------------------------------------------------------------------

      eccice = 10.
      ecc2 = 1./(eccice**2)
      ecc2m = 2.*(1.-ecc2)
      ecc2p = (1.+ecc2)
      zetamin = 4.e11
      eyc = 0.25
      dte = dtatm/float(ndte)
      dtei = 1./dte
      floor = 1.e-11
      do j=2,jmtm1
        do i=2,imtm1
           xyminevp=(min(cst(j)*dxt(i),dyt(j)))**2
        enddo
      enddo

# endif
# if defined uvic_embm_even_fluxes && defined uvic_mom
!-----------------------------------------------------------------------
!     check for even flux averaging
!-----------------------------------------------------------------------

      if (mod(namix,2) .ne. mod(nint(segtim*86400./dtatm),2)) then
        write(*,*) '==> Error: time steps between mixing and coupling'
     &, ' must both be even to use uvic_embm_even_fluxes.'
        stop '=>setembm'
      endif

      if (mod(namix,2) .ne. mod(nats,2)) then
        write(*,*) '==> Warning: restart was not saved with even flux'
     &, ' averaging. Starting with a mixing time step.'
        nats = namix
      endif

# endif
!-----------------------------------------------------------------------
!     check ice velocity calculation
!-----------------------------------------------------------------------

      if (nivts .gt. nint(segtim*86400./dtatm)) then
        write(*,*) '==> Warning: ice velocities will be calculated '
     &, 'every coupling time.'
        nivts =  nint(segtim*86400./dtatm)
      endif
# if defined uvic_embm_solve2y
!-----------------------------------------------------------------------
!     check atmosphere is even size in y
!-----------------------------------------------------------------------

      if (mod(jmtm2,2) .ne. 0) then
        write(*,*) '==> Error: atmosphere must be even sized in '
     &, 'latitude to use uvic_embm_solve2y.'
        stop '=>setembm'
      endif
# endif
# if defined uvic_embm_solve2x
!-----------------------------------------------------------------------
!     check atmosphere is even size in x
!-----------------------------------------------------------------------

      if (mod(imtm2,2) .ne. 0) then
        write(*,*) '==> Error: atmosphere must be even sized in '
     &, 'latitude to use uvic_embm_solve2x.'
        stop '=>setembm'
      endif
# endif
# if defined time_averages
!-----------------------------------------------------------------------
!     zero time average accumulators
!-----------------------------------------------------------------------

      call ta_atm_snap (is, ie, js, je, 0)

# endif
# if defined time_averages
!-----------------------------------------------------------------------
!     zero integrated time average accumulators
!-----------------------------------------------------------------------

      call ta_atm_tsi (0)

# endif
# if defined uvic_embm_landice
!-----------------------------------------------------------------------
!     read land ice mask
!-----------------------------------------------------------------------

      call getunit (iou, 'ice_masks', 'f s r')
      do j=jmt,1,-1
        read(iou,'(1x,i3,1x,200(i1))')(aicel(i,j,2),i=1,imt)
      enddo
      call relunit (iou)

# endif
#endif
      return
      end
