      subroutine ocn_tsi_in (fname, ntrec, timunit, expnam, avgper
     &,                      relyr, stamp, ektot, tbar1, tbar2
     &,                      travar1, travar2, dtabs1, dtabs2, scan
     &,                      otmax, otmin)
!=======================================================================
!     input routine for ocean time step integrals

!   inputs:
!     fname      = file name
!     ntrec      = number of time record

!   outputs:
!     timunit    = time unit
!     expnam     = experiment name
!     avgper     = length of averaging period
!     relyr      = time in years
!     stamp      = time stamp
!     ektot, ... = data to be read

!   local variables
!     iou        = io unit (ncid)

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      character(*) :: fname, timunit, expnam, stamp
      character(80) :: name, title, file_stamp

      integer i, id_time, iou, ntrec
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real ektot, tbar1, tbar2, travar1, travar2, dtabs1, dtabs2, scan
      real otmax, otmin, avgper, relyr, tmp(1)

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      name = fname
      i = 0
      call openfile (name, relyr, i, iou)

!-----------------------------------------------------------------------
!     get global attributes
!-----------------------------------------------------------------------
      call getglobal (iou, name, title, expnam, timunit)

!-----------------------------------------------------------------------
!     read 1d data (t)
!-----------------------------------------------------------------------
      call getvara ('time', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      relyr = tmp(1)
      call getvara ('year', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nyear = tmp(1)
      call getvara ('month', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nmonth = tmp(1)
      call getvara ('day', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nday = tmp(1)
      call getvara ('hour', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nhour = tmp(1)
      call getvara ('minute', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nmin = tmp(1)
      call getvara ('second', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      nsec = tmp(1)
      call mkstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      call getvara ('period', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      avgper = tmp(1)
      call getvara ('ke', iou, 1, (/ntrec/), (/1/), tmp, 1.e-2, 0.)
      ektot = tmp(1)
      call getvara ('tbar', iou, 1, (/ntrec/), (/1/), tmp, 1., 273.15)
      tbar1 = tmp(1)
      call getvara ('sbar', iou, 1, (/ntrec/), (/1/), tmp, 1000., 35.)
      tbar2 = tmp(1)
      call getvara ('tvar', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      travar1 = 0.0
      if (tmp(1) .gt. 0.0) travar1 = ((sqrt(tmp(1))-273.15))**2
      call getvara ('svar', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      travar2 = 0.0
      if (tmp(1) .gt. 0.0) travar2 = ((sqrt(tmp(1))-35.)/1000.)**2
      call getvara ('abs_chg_t', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      dtabs1 = tmp(1)
      call getvara ('abs_chg_s', iou, 1, (/ntrec/), (/1/), tmp
     &, 1000., 0.)
      dtabs2 = tmp(1)
      call getvara ('scans', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      scan = tmp(1)
      call getvara ('motsf_max', iou, 1, (/ntrec/), (/1/), tmp
     &, 1.e-6, 0.)
      otmax = tmp(1)
      call getvara ('motsf_min', iou, 1, (/ntrec/), (/1/), tmp
     &, 1.e-6, 0.)
      otmin = tmp(1)

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)

      return
      end

      subroutine ocn_tsi_out (fname, ntrec, timunit, expnam, avgper
     &,                       relyr, stamp, ektot, tbar1, tbar2
     &,                       travar1, travar2, dtabs1, dtabs2, scan
     &,                       otmax, otmin)
!=======================================================================
!     output routine for ocean time step integrals

!   inputs:
!     fname      = file name
!     ntrec      = number of time record (zero if not defined)
!     timunit    = time unit
!     expnam     = experiment name
!     avgper     = length of averaging period
!     relyr      = time in years
!     stamp      = time stamp
!     ektot, ... = data to be written

!   outputs:
!     ntrec      = number of time record in file

!   local variables
!     iou        = io unit (ncid)

!     author:   m.eby   e-mail: eby@uvic.ca
!=======================================================================

      implicit none

      character(*) :: fname, timunit, expnam, stamp
      character(80) :: name, title, file_stamp

      integer id_time, iou, ntrec
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real ektot, tbar1, tbar2, travar1, travar2, dtabs1, dtabs2, scan
      real otmax, otmin, avgper, relyr, tmp(1)

      logical defined

!-----------------------------------------------------------------------
!     open file and get latest record number
!-----------------------------------------------------------------------
      defined = .true.
      name = fname
      if (ntrec .eq. 0) defined = .false.
      call openfile (name, relyr, ntrec, iou)

      if (.not. defined .or. ntrec .eq. 0) then

!-----------------------------------------------------------------------
!       start definitions
!-----------------------------------------------------------------------
        call redef (iou)

!-----------------------------------------------------------------------
!       set global attributes
!-----------------------------------------------------------------------
        if (avgper .gt. 1.e-6) then
          title = 'Time Average Time Step Integrals'
        else
          title = 'Time Step Integrals'
        endif
        call putglobal (iou, name, title, expnam, timunit)

!-----------------------------------------------------------------------
!       define dimensions
!-----------------------------------------------------------------------
        call defdim ('time', iou, 0, id_time)

!-----------------------------------------------------------------------
!       define 1d data (t)
!-----------------------------------------------------------------------
        call defvar ('time', iou, 1, (/id_time/), 0., 0., 'T', 'D'
     &,   'time since initial condition', 'time', trim(timunit))
        call defvar ('year', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'year', ' ',' ')
        call defvar ('month', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'month', ' ',' ')
        call defvar ('day', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'day', ' ',' ')
        call defvar ('hour', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'hour', ' ',' ')
        call defvar ('minute', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'minute', ' ',' ')
        call defvar ('second', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,   'second', ' ',' ')
        if (avgper .gt. 1.e-6) then
          call defvar ('period', iou, 1, (/id_time/), 0., 0., ' ', 'F'
     &,     'averaging period', ' ',' ')
        endif
        call defvar ('ke', iou, 1, (/id_time/), 0., 1.e6, ' ', 'F'
     &,   'Kinetic energy per unit volume', ' ', 'J m-3')
        call defvar ('tbar', iou, 1, (/id_time/), 100., 400., ' ', 'F'
     &,   'Global average T', ' ', 'K')
        call defvar ('sbar', iou, 1, (/id_time/), 0., 50., ' ', 'F'
     &,   'Global average S', ' ', 'psu')
        call defvar ('tvar', iou, 1, (/id_time/), 0., 1.e10, ' ', 'F'
     &,   'Variance of T', ' ', 'K2')
        call defvar ('svar', iou, 1, (/id_time/), 0., 1.e10, ' ', 'F'
     &,   'Variance of S', ' ', 'psu2')
        call defvar ('abs_chg_t', iou, 1, (/id_time/), 0., 1.e10, ' '
     &,   'F', '|dT/dt|', ' ', 'K s-1')
        call defvar ('abs_chg_s', iou, 1, (/id_time/), 0., 1.e10, ' '
     &,   'F', '|dS/dt|', ' ', 'psu s-1')
        call defvar ('scans', iou, 1, (/id_time/), 0., 1.e4, ' ', 'F'
     &,   'elliptic solver iterations', ' ', 'scans')
        call defvar ('motsf_max', iou, 1, (/id_time/), 0., 1.e20, ' '
     &,   'F', 'maximum meridional overturning streamfuncion', ' '
     &,   'm3 s-1')
        call defvar ('motsf_min', iou, 1, (/id_time/), 0., 1.e20, ' '
     &,   'F', 'minimum meridional overturning streamfuncion', ' '
     &,   'm3 s-1')

!-----------------------------------------------------------------------
!       end definitions
!-----------------------------------------------------------------------
        call enddef (iou)
        if (ntrec .eq. 0) ntrec = 1

      endif

!-----------------------------------------------------------------------
!     write 1d data (t)
!-----------------------------------------------------------------------
      tmp(1) = relyr
      call putvara ('time', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      call rdstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      tmp(1) = nyear
      call putvara ('year', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nmonth
      call putvara ('month', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nday
      call putvara ('day', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nhour
      call putvara ('hour', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nmin
      call putvara ('minute', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = nsec
      call putvara ('second', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      if (avgper .gt. 1.e-6) then
        tmp(1) = avgper
        call putvara ('period', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      endif
      tmp(1) = ektot
      call putvara ('ke', iou, 1, (/ntrec/), (/1/), tmp, 1.e-2, 0.)
      tmp(1) = tbar1
      call putvara ('tbar', iou, 1, (/ntrec/), (/1/), tmp, 1., 273.15)
      tmp(1) = tbar2
      call putvara ('sbar', iou, 1, (/ntrec/), (/1/), tmp, 1000., 35.)
      tmp(1) = 0.0
      if (travar1 .gt. 0.0) tmp(1) = (sqrt(travar1)+273.15)**2
      call putvara ('tvar', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = 0.0
      if (travar2 .gt. 0.0) tmp(1) = (sqrt(travar2)*1000.+35.)**2
      call putvara ('svar', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = dtabs1
      call putvara ('abs_chg_t', iou, 1, (/ntrec/), (/1/), tmp, 1., 0.)
      tmp(1) = dtabs2
      call putvara ('abs_chg_s', iou, 1, (/ntrec/), (/1/), tmp
     &, 1000., 0.)
      call putvara ('scans', iou, 1, (/ntrec/), (/1/), (/scan/), 1., 0.)
      tmp(1) = otmax
      call putvara ('motsf_max', iou, 1, (/ntrec/), (/1/), tmp
     &, 1.e-6, 0.)
      tmp(1) = otmin
      call putvara ('motsf_min', iou, 1, (/ntrec/), (/1/), tmp
     &, 1.e-6, 0.)

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      call closefile (iou)

      return
      end
