#if defined ramdrive && !defined distributed_memory
      subroutine ostart (lu, fname, ntot, nwrec, nbuf)

!=======================================================================

!     ramdrive section uses memory as a ramdisk

!     check for validity of unitnumber and show memory requirement

!     lu    = i/o unit number associated with "fname"
!     fname = filename
!     ntot  = total length of unit, in words
!     nwrec = number of words in the record on the unit
!     nbuf  = number of buffers for the unit
!=======================================================================

      character(*) :: fname
# include "param.h"
# include "iounit.h"
# if defined coarse_grained_parallelism
      if (lu .ne. kflds .and. lu .ne. latdisk(1) .and.
     &                        lu .ne. latdisk(2) .and.
     &                        lu .ne. latdisk(3)) then
        write (stdout,'(/1x,a,/,11x,a,/11x,a,i10)')
     &  '=>Error: ostart... when using the "ramdrive" option, only'
     &,' units "kflds","latdisk(1)","latdisk(2)","latdisk(3)" are '
     &,' allowed and you were trying to open unit #',lu
        stop '=>ostart'
      endif
# else
      if (lu .ne. kflds .and. lu .ne. latdisk(1) .and.
     &                        lu .ne. latdisk(2)) then
        write (stdout,'(/1x,a,/,11x,a,/11x,a,i10)')
     &  '=>Error: ostart... when using the "ramdrive" option, only'
     &,' units "kflds","latdisk(1)" or "latdisk(2)" are allowed'
     &,' you were trying to open unit #',lu
        stop '=>ostart'
      endif
# endif
      write (stdout,'(/a,a,a,1pg10.3,a)')
     & ' Memory resource for file ',fname,' = ',ntot*1.e-6,' (MW)'

!     attach "lu" to "fname" to remove "lu" from the list of possible
!     unit numbers. This is just a dummy file.

      call getunit (lu, fname, 'sequential, unformatted, rewind')
      return
      end

      subroutine ofind (lu, nwrs, nrec)
      return
      end

      subroutine oclose (lu)
      call relunit (lu)
      return
      end

      subroutine getrow (lu, nwrs, nrec, u, t)

!=======================================================================
!     get velocity and tracer data from ramdisk latitude row "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = latitude row
!         u     = velocities
!         t     = tracers
!         ntrac = number of tracers
!=======================================================================

# include "param.h"
# include "iounit.h"

# if defined coarse_grained_parallelism
#  include "taskrows.h"
      parameter (num_timelevels = 3)
      parameter (jtaskmax = (jmt-2)/nprocessors+1 + 2*(1+jextra))
!DIR$ TASKCOMMON distrib
      common /distrib/ sdisk(nslab,jtaskmax,3)
# else
      parameter (num_timelevels = 2)
      common /distrib/ sdisk(nslab,jmt,2)
# endif
      parameter (nsl = nslab*jmt, ntb=num_timelevels*nsl*(1-jmw/jmt)+1)
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)

# if defined coarse_grained_parallelism
      jlocal = nrec - jstask(pn) + 1
# else
      jlocal = nrec
# endif

# if defined coarse_grained_parallelism
      if (lu .ne. latdisk(1) .and. lu .ne. latdisk(2)
     &    .and. lu .ne. latdisk(3)) then
        write (stdout,*) '=>Error: unit ',lu,' is not allowed. nrec='
     &, nrec
        stop '=>getrow'
      endif
# else
      if (lu .ne. latdisk(1) .and. lu .ne. latdisk(2)) then
        write (stdout,*) '=>Error: unit ',lu,' is not allowed. nrec='
     &, nrec
        stop '=>getrow'
      endif
# endif

      if (lu .eq. latdisk(1)) then
        ntl = 1
      elseif (lu .eq. latdisk(2)) then
        ntl = 2
# if defined coarse_grained_parallelism
      elseif (lu .eq. latdisk(3)) then
        ntl = 3
# endif
      endif

      call getlat (sdisk(1,jlocal,ntl), u, t)

      return

      entry putrow (lu, nwrs, nrec, u, t)

!=======================================================================
!     put velocity and tracer data to ramdisk latitude row "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = latitude row
!         u     = velocities
!         t     = tracers
!         ntrac = number of tracers
!=======================================================================

# if defined coarse_grained_parallelism
      jlocal = nrec - jstask(pn) + 1
# else
      jlocal = nrec
# endif

# if defined coarse_grained_parallelism
      if (lu .ne. latdisk(1) .and. lu .ne. latdisk(2)
     &    .and. lu .ne. latdisk(3)) then
        write (stdout,*) '=>Error: unit ',lu,' is not allowed. nrec='
     &, nrec
        stop '=>putrow'
      endif
# else
      if (lu .ne. latdisk(1) .and. lu .ne. latdisk(2)) then
        write (stdout,*) '=>Error: unit ',lu,' is not allowed. nrec='
     &, nrec
        stop '=>putrow'
      endif
# endif

      if (lu .eq. latdisk(1)) then
        ntl = 1
      elseif (lu .eq. latdisk(2)) then
        ntl = 2
# if defined coarse_grained_parallelism
      elseif (lu .eq. latdisk(3)) then
        ntl = 3
# endif
      endif

      call putlat (sdisk(1,jlocal,ntl), u, t)
      return
      end

      subroutine getlat (disk, u, t)

!-----------------------------------------------------------------------
!     copy contiguous portions of virtual disk to non-contiguous
!     portions of memory.
!-----------------------------------------------------------------------

#include "param.h"
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac), disk(imt,km,nvar)
      do n=1,2
        do k=1,km
          do i=1,imt
            u(i,k,1,n) = disk(i,k,n)
          enddo
        enddo
      enddo
      do n=1,ntrac
        do k=1,km
          do i=1,imt
            t(i,k,1,n) = disk(i,k,n+2)
          enddo
        enddo
      enddo
      return
      end

      subroutine putlat (disk, u, t)

!-----------------------------------------------------------------------
!     copy non-contiguous portions of memory to contiguous portions
!     of virtual disk.
!-----------------------------------------------------------------------

#include "param.h"
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac), disk(imt,km,nvar)
      do n=1,2
        do k=1,km
          do i=1,imt
            disk(i,k,n) = u(i,k,1,n)
          enddo
        enddo
      enddo
      do n=1,ntrac
        do k=1,km
          do i=1,imt
            disk(i,k,n+2) = t(i,k,1,n)
          enddo
        enddo
      enddo
      return
      end

      subroutine oget (lu, nwrs, nrec, a)

!=======================================================================
!     get 2D field data from ramdisk record "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = record
!         a     = two dimensional field
!=======================================================================

# include "param.h"
# include "iounit.h"

      parameter (n12=nwds*nkflds)
      dimension sdisk2d(n12), a(nwds)
      save sdisk2d

      if (lu .ne. kflds) then
        write (stdout,*) '=>Error: unit # ',lu,' not allowed. rec=',nrec
        stop '=>oget'
      else
        ns = (nrec-1)*nwrs + 1
        ne = ns + nwrs - 1
        do n=ns,ne
          a(n-ns+1) = sdisk2d(n)
        enddo
      endif
      return

      entry oput (lu, nwrs, nrec, a)

!=======================================================================
!     put 2D field data to ramdisk record "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = record
!         a     = two dimensional field
!=======================================================================

      if (lu .ne. kflds) then
        write (stdout,*) '=>Error: unit # ',lu,' not allowed. rec=',nrec
        stop '=>oget'
      else
        ns = (nrec-1)*nwrs + 1
        ne = ns + nwrs - 1
        do n=ns,ne
          sdisk2d(n) = a(n-ns+1)
        enddo
      endif
      return
      end

#endif

#if defined ramdrive && defined distributed_memory
      subroutine ostart (lu, fname, ntot, nwrec, nbuf)

!=======================================================================

!     ramdrive section uses memory as a ramdisk on distributed systems

!     check for validity of unitnumber and show memory requirement

!     lu    = i/o unit number associated with "fname"
!     fname = filename
!     ntot  = total length of unit, in words
!     nwrec = number of words in the record on the unit
!     nbuf  = number of buffers for the unit
!=======================================================================

      character(*) :: fname
# include "param.h"
# include "iounit.h"
      if (lu .ne. kflds .and. lu .ne. latdisk(1) .and.
     &                        lu .ne. latdisk(2) .and.
     &                        lu .ne. latdisk(3)) then
        write (stdout,'(/1x,a,/,11x,a,/11x,a,i10)')
     &  '=>Error: ostart... when using the "ramdrive" option, only'
     &,' units "kflds","latdisk(1)","latdisk(2)","latdisk(3)" are '
     &,' allowed and you were trying to open unit #',lu
        stop '=>ostart'
      endif
      write (stdout,'(/a,a,a,1pg10.3,a)')
     & ' Memory resource for file ',fname,' = ',ntot*1.e-6,' (MW)'
      return
      end

!     attach "lu" to "fname" to remove "lu" from the list of possible
!     unit numbers. This is just a dummy file.

      call getunit (lu, fname, 'sequential, unformatted, rewind')
      return
      end

      subroutine ofind (lu, nwrs, nrec)
      return
      end

      subroutine oclose (lu)
      return
      end

      subroutine getrow (lu, nwrs, nrec, u, t)

!=======================================================================
!     get velocity and tracer data from ramdisk latitude row "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = latitude row
!         u     = velocities
!         t     = tracers
!         ntrac = number of tracers
!=======================================================================

# include "param.h"
# include "iounit.h"
# include "taskrows.h"

      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)

!     for distributed systems

!DIR$ TASKCOMMON distrib
      parameter (jtaskmax = (jmt-2)/nprocessors+1 + 2*(1+jextra))
      common /distrib/ sdisk(nslab,jtaskmax,3)

      jlocal = nrec - jstask(pn) + 1
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2) .or.
     &    lu .eq. latdisk(3)) then
        if (lu .eq. latdisk(1)) then
          ntl = 1
        elseif (lu .eq. latdisk(2)) then
          ntl = 2
        elseif (lu .eq. latdisk(3)) then
          ntl = 3
        endif
        call getlat (sdisk(1,jlocal,ntl), u, t)
      else
        write (stdout,*) '=>Error: unit ',lu,' is not allowed. jlocal='
     &, jlocal,' jrow=',nrec
        stop '=>getrow'
      endif
      return

      entry putrow (lu, nwrs, nrec, u, t)

!=======================================================================
!     put velocity and tracer data to ramdisk latitude row "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = latitude row
!         u     = velocities
!         t     = tracers
!         ntrac = number of tracers
!=======================================================================

      jlocal = nrec - jstask(pn) + 1
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2) .or.
     &    lu .eq. latdisk(3)) then
        if (lu .eq. latdisk(1)) then
          ntl = 1
        elseif (lu .eq. latdisk(2)) then
          ntl = 2
        elseif (lu .eq. latdisk(3)) then
          ntl = 3
        endif
        call putlat (sdisk(1,jlocal,ntl), u, t)
      else
        write (stdout,*) '=>Error: unit ',lu,' is not allowed. jlocal='
     &, jlocal, ' jrow=',nrec
        stop '=>putrow'
      endif
      return
      end

      subroutine getlat (disk, u, t)

!-----------------------------------------------------------------------
!     copy contiguous portions of virtual disk to non-contiguous
!     portions of memory.
!-----------------------------------------------------------------------

#include "param.h"
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)
      dimension disk(imt,km,nvar)
      do n=1,2
        do k=1,km
          do i=1,imt
            u(i,k,1,n) = disk(i,k,n)
          enddo
        enddo
      enddo
      do n=1,ntrac
        do k=1,km
          do i=1,imt
            t(i,k,1,n) = disk(i,k,n+2)
          enddo
        enddo
      enddo
      return
      end

      subroutine putlat (disk, u, t)

!-----------------------------------------------------------------------
!     copy non-contiguous portions of memory to contiguous portions
!     of virtual disk.
!-----------------------------------------------------------------------

#include "param.h"
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac), disk(imt,km,nvar)
      do n=1,2
        do k=1,km
          do i=1,imt
            disk(i,k,n) = u(i,k,1,n)
          enddo
        enddo
      enddo
      do n=1,ntrac
        do k=1,km
          do i=1,imt
            disk(i,k,n+2) = t(i,k,1,n)
          enddo
        enddo
      enddo
      return
      end

      subroutine oget (lu, nwrs, nrec, a)

!=======================================================================
!     get 2D field data from ramdisk record "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = record
!         a     = two dimensional field
!=======================================================================

# include "param.h"
# include "iounit.h"

      parameter (n12=nwds*nkflds)
      dimension sdisk2d(n12), a(nwds)
      save sdisk2d

      if (lu .ne. kflds) then
        write (stdout,*) '=>Error: unit # ',lu,' not allowed. rec=',nrec
        stop '=>oget'
      else
        ns = (nrec-1)*nwrs + 1
        ne = ns + nwrs - 1
        do n=ns,ne
          a(n-ns+1) = sdisk2d(n)
        enddo
      endif
      return

      entry oput (lu, nwrs, nrec, a)

!=======================================================================
!     put 2D field data to ramdisk record "nrec"

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = record
!         a     = two dimensional field
!=======================================================================

      if (lu .ne. kflds) then
        write (stdout,*) '=>Error: unit # ',lu,' not allowed. rec=',nrec
        stop '=>oget'
      else
        ns = (nrec-1)*nwrs + 1
        ne = ns + nwrs - 1
        do n=ns,ne
          sdisk2d(n) = a(n-ns+1)
        enddo
      endif
      return
      end

      subroutine ipc (n_source, n_target, jfrom, jto, ntl, nwords)

!=======================================================================
!     inter-processor communication: copy data from latitude "jfrom"
!     on source processor to "jto" on target processor.

!     n_source  = source processor
!     n_target  = target processor
!     jfrom     = latitude row on source processor
!     jto       = latitude row on target processor
!=======================================================================

# include "param.h"

!     for distributed systems

!DIR$ TASKCOMMON distrib
      parameter (jtaskmax = (jmt-2)/nprocessors+1 + 2*(1+jextra))
      common /distrib/ sdisk(nslab,jtaskmax,3)

      call shmem_get (sdisk(1,jto,ntl),   n_target
     &,               sdisk(1,jfrom,ntl), n_source, nwords)
      return
      end

#endif

      subroutine copy_all_rows (ifrom, ito)

!-----------------------------------------------------------------------
!     copy all latitude rows from time level "ifrom" to time level "ito"
!     for prognostic variables in the MW

!     author:  r.c.pacanowski   e-mail  rcp@gfdl.gov
!-----------------------------------------------------------------------

#include "param.h"
#include "mw.h"
      do n=1,2
        do j=1,jmw
          do k=1,km
            do i=1,imt
              u(i,k,j,n,ito) = u(i,k,j,n,ifrom)
            enddo
          enddo
        enddo
      enddo
      do n=1,nvar-2
        do j=1,jmw
          do k=1,km
            do i=1,imt
              t(i,k,j,n,ito) = t(i,k,j,n,ifrom)
            enddo
          enddo
        enddo
      enddo
      return
      end

      subroutine euler_shuffle

!-----------------------------------------------------------------------
!     after the second pass of an euler backward step, exchange "tau"
!     and "tau+1" data, after shuffling, data will be in proper position
!     for the next time step.

!     author:  r.c.pacanowski   e-mail  rcp@gfdl.gov
!-----------------------------------------------------------------------

#include "param.h"
#include "mw.h"
      do n=1,2
        do j=1,jmw
          do k=1,km
            do i=1,imt
              temp             = u(i,k,j,n,tau)
              u(i,k,j,n,tau)   = u(i,k,j,n,taup1)
              u(i,k,j,n,taup1) = temp
            enddo
          enddo
        enddo
      enddo
      do n=1,nt
        do j=1,jmw
          do k=1,km
            do i=1,imt
              temp             = t(i,k,j,n,tau)
              t(i,k,j,n,tau)   = t(i,k,j,n,taup1)
              t(i,k,j,n,taup1) = temp
            enddo
          enddo
        enddo
      enddo
      return
      end

#if defined fio
      subroutine  ostart (lu, fname, ntot, nwrec, nbuf)

!=======================================================================

!     fortran i/o section

!     assign a unit number ot "fname" and show resource requriements

!     fname = filename
!     lu    = i/o unit number associated with "fname"
!     ntot  = total length of unit, in words
!     nwrec = number of words in the record on the unit
!     nbuf  = number of buffers supplied to the unit
!=======================================================================

      character(*) :: fname
      character(80) :: optlst
# include "param.h"
# include "iounit.h"
      if (fname .eq. 'latdisk1' .or. fname .eq. 'latdisk2') then
        lenblk = nwrec / nvar
      else
        lenblk = nwrec
      endif
      write (optlst,'(a,i10)') 'direct recl:words = ',lenblk
      call getunit (lu, fname, optlst)
      write (stdout,'(/a,a,a,1pg10.3,a)')
     & ' Direct access disk resource for file ',fname,' = '
     &, ntot*1.e-6,' (MW)'
      return

      entry oclose (lu)
      return

      entry ofind (lu, nwrs, nrec)
      return
      end

      subroutine getrow (lu, nwrs, nrec, u, t)

!=======================================================================
!     get velocity and tracer data from latitude row "nrec" on disk

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = latitude row
!         u     = velocities
!         t     = tracers
!         ntrac = number of tracers
!=======================================================================

# include "param.h"
      parameter (ntrac=nvar-2)
# include "iounit.h"

      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)

      nfrst = (nrec-1)*nwrs + 1
# if defined coarse_grained_parallelism
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2) .or.
     &    lu .eq. latdisk(3)) then
# else
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2)) then
# endif
        call getlat (lu, u, t, nfrst, nwrs)
      else
        write (stdout,*) '=>Error: unit #',lu,' not allowed. rec=',nrec
        stop '=>getrow'
      endif
      return
      end

      subroutine getlat (lu, u, t, nfrst, nwrs)

!-----------------------------------------------------------------------
!     transfer contiguous portions of disk to non-contiguous
!     portions of memory.
!-----------------------------------------------------------------------

#include "param.h"
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)
      dimension buf(imt,km)
      nwords = nwrs / nvar
      nfirst = (nfrst-1)/nwords+1
      do n=1,2
        nrec = nfirst + n - 1
# if defined coarse_grained_parallelism
CMIC$ GUARD
        read (lu, rec=nrec) buf
CMIC$ END GUARD
# else
        read (lu, rec=nrec) buf
# endif
        do k=1,km
          do i=1,imt
            u(i,k,1,n) = buf(i,k)
          enddo
        enddo
      enddo
      nfirst = nfirst + 2  - 1
      do n=1,ntrac
        nrec = nfirst + n
# if defined coarse_grained_parallelism
CMIC$ GUARD
        read (lu, rec=nrec) buf
CMIC$ END GUARD
# else
        read (lu, rec=nrec) buf
# endif
        do k=1,km
          do i=1,imt
            t(i,k,1,n) = buf(i,k)
          enddo
        enddo
      enddo
      return
      end

      subroutine oget (lu, nwrs, nrec, a)

!=======================================================================
!     get data from record "nrec" on disk

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = record
!         a     = array where data will go
!=======================================================================

# include "param.h"
# include "iounit.h"

      dimension a(nwrs)

# if defined coarse_grained_parallelism
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2) .or.
     &    lu .eq. latdisk(3)) then
# else
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2)) then
# endif
        write (stdout,*) '=>Error: unit #',lu, ' not allowed. rec=',nrec
        stop '=>oget'
      else
# if defined coarse_grained_parallelism
CMIC$ GUARD
        read (lu, rec=nrec) a
CMIC$ END GUARD
# else
        read (lu, rec=nrec) a
# endif
      endif
      return
      end

      subroutine putrow (lu, nwrs, nrec, u, t)

!=======================================================================
!     put velocity and tracer data to latitude row "nrec" on disk

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = latitude row
!         u     = velocities
!         t     = tracers
!         ntrac = number of tracers
!=======================================================================

# include "param.h"
      parameter (ntrac=nvar-2)
# include "iounit.h"

      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)

      nfrst = (nrec-1)*nwrs + 1
# if defined coarse_grained_parallelism
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2) .or.
     &    lu .eq. latdisk(3)) then
# else
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2)) then
# endif
        call putlat (lu, u, t, nfrst, nwrs)
      else
        write (stdout,*) '=>Error: unit #',lu,' not allowed. rec=',nrec
        stop '=>getrow'
      endif
      return
      end

      subroutine putlat (lu, u, t, nfrst, nwrs)

!-----------------------------------------------------------------------
!     transfer non-contiguous portions of memory to
!     contiguous portions of disk.
!-----------------------------------------------------------------------

#include "param.h"
      parameter (ntrac=nvar-2)
      dimension u(imt,km,jmw,2), t(imt,km,jmw,ntrac)
      dimension buf(imt,km)
      nwords = nwrs / nvar
      nfirst = (nfrst-1)/nwords+1
      do n=1,2
        nrec = nfirst + n - 1
        do k=1,km
          do i=1,imt
            buf(i,k) = u(i,k,1,n)
          enddo
        enddo
# if defined coarse_grained_parallelism
CMIC$ GUARD
        write (lu, rec=nrec) buf
CMIC$ END GUARD
# else
        write (lu, rec=nrec) buf
# endif
      enddo
      nfirst = nfirst + 2  - 1
      do n=1,ntrac
        nrec = nfirst + n
        do k=1,km
          do i=1,imt
            buf(i,k) = t(i,k,1,n)
          enddo
        enddo
# if defined coarse_grained_parallelism
CMIC$ GUARD
        write (lu, rec=nrec) buf
CMIC$ END GUARD
# else
        write (lu, rec=nrec) buf
# endif
      enddo
      return
      end

      subroutine oput (lu, nwrs, nrec, a)

!=======================================================================
!     put data to record "nrec" on disk

!         lu    = i/o unit number associated with "fname"
!         nwrs  = number of words to read
!         nrec  = record
!         a     = array where data will go
!=======================================================================

# include "param.h"
# include "iounit.h"

      dimension a(nwrs)

# if defined coarse_grained_parallelism
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2) .or.
     &    lu .eq. latdisk(3)) then
# else
      if (lu .eq. latdisk(1) .or. lu .eq. latdisk(2)) then
# endif
        write (stdout,*) '=>Error: unit #',lu, ' not allowed. rec=',nrec
        stop '=>oget'
      else
# if defined coarse_grained_parallelism
CMIC$ GUARD
        write (lu, rec=nrec) a
CMIC$ END GUARD
# else
        write (lu, rec=nrec) a
# endif
      endif
      return
      end

#endif
