! source file: /usr/local/models/UVic_ESCM/2.7/source/mom/mw.h
!======================= include file "mw.h" ===========================

!                     M E M O R Y    W I N D O W

!                         Data layout on disk

!     On disk, a latitude consists of a row of prognostic quantities.
!     Latitudes are ordered from south to north on two logical disk
!     units: one for time level "tau" and one for time level "tau-1".
!     Newly computed "tau+1" quantities are written to the "tau-1" disk
!     which serves double duty as a "tau+1" disk.

!                         Data layout in memory

!     A memory window "MW" is constructed to hold a group of "jmw"
!     adjacent latitude rows of prognostic quantities from disks "tau-1"
!     and "tau".  Parameter "jmw" controls the size of the MW and can be
!     set anywhere from a miminum of three latitude rows to all "jmt"
!     latitudes.  In addition, the MW holds diagnostic quantities
!     (density, hydrostatic pressure gradients, and advective
!     velocities) along with work arrays for constructing intermediate
!     fluxes used in solving the tracer and momentum equations. A
!     latitude row in the MW is referred to by index "j" and corresponds
!     to the physical latitude row "jrow" on disk.

!                   Data flow between disk and memory

!     The MW is loaded with prognostic data from the first "jww"
!     latitude rows on disks "tau" and "tau-1". As the tracer and
!     momentum equations are solved for rows j=2 through jmw-1 in the
!     MW, the solutions are written to disk "tau+1". When finished, the
!     MW is moved northward by moving quantities from the northernmost
!     two rows into the southernmost two rows within the MW. The
!     remaining MW rows are then loaded with more latitude rows from
!     disk. The process continues until all latitude rows 2 through
!     jmt-1 on disk "tau+1" have been updated.

!                   Parallel processing considerations

!     When the MW is opened all the way ("jmw" = "jmt"), there is no
!     need for disk units since all three time levels for all prognostic
!     quantities at all latitudes reside in the MW. Therefore, there is
!     no need to read data, write data or move the MW northward (or
!     re-calculate the three dimensional land/sea mask arrays). After a
!     time step has finished, time level pointers on the next time step
!     are simply updated to reflect where the "tau-1", "tau" and "tau+1"
!     data is. In principle, MOM 2 can be viewed as a sucession of
!     triply nested do loops with parallelism exposed at the do loop
!     level (fine grained parallelism) as opposed to the latitude row
!     level (coarse grained parallelism) of MOM 1. This structure
!     significantly reduces the memory requirements compared to MOM 1
!     when executing on multiple processors. On one processor, the
!     memory requirement is comparable to MOM 1.

!     based on code by: R. C. Pacanowski
!=======================================================================

!     taum1 = tau-1 time level for variables in MW
!     tau   = tau   time level for variables in MW
!     taup1 = tau+1 time level for variables in MW

      integer taum1, tau, taup1
      common /mwi/ taum1, tau, taup1

!-----------------------------------------------------------------------
!     MW arrays for prognostic equations:
!-----------------------------------------------------------------------

!     u(i,k,j,n,tau) = total velocity where:
!      i   = index for longitude
!      k   = index for depth
!      j   = index for latitude row within MW
!      n   = component (1 = zonal, 2 = meridional)
!      tau = time level (tau-1, tau, tau+1)

!      (only internal modes are on disk and at tau+1 in the MW)

!     t(i,k,j,n,tau) = tracer where:
!      i   = index for longitude
!      k   = index for depth
!      j   = index for latitude row within MW
!      n   = component (1 = temperature, 2 = salinity)
!            if nt > 2 then other tracers are allowed.
!      tau = time level (tau-1, tau, tau+1)

!     note: temperature is potential temperature in degrees Celsius and
!           salinity is in "model units", the deviation from 0.035 grams
!           of salt/cm**3 of water, or, assuming a water density of
!           1 gram/cm**3, the deviation from 0.035 g of salt/g of water.
!           one can convert model units to the more common units of parts
!           per thousand (ppt) by adding 0.035 grams/cm**3 to the model
!           units and then multiplying by 1000.

      common /mw/ u(imt,km,jmw,2,-1:1), t(imt,km,jmw,nt,-1:1)

!     indicies for ocean tracer array
!       itemp = index for temperature
!       isalt = index for  salinity
!       idic  = index for  disolved inorganic carbon
!       inutr = index for  nutrients
!       iphyt = index for  phytoplankton
!       izoop = index for  zooplankton
!       idetr = index for  detritus
!       mapt  = map for ocean tracer names

      character(10) :: mapt(nt)
      common /mw_c/ mapt

      integer itemp, isalt, idic, inutr, iphyt, izoop, idetr
      common /mw_i/ itemp, isalt, idic, inutr, iphyt, izoop, idetr

!-----------------------------------------------------------------------
!     MW arrays for diagnostic equations and workspace:
!-----------------------------------------------------------------------

!     diagnostic advective velocities are in units of cm/sec

!     adv_vet  = advective velocity on the eastern face of a "T" cell
!     adv_vnt  = advective velocity on the northern face of a "T" cell
!     adv_veu  = advective velocity on the eastern face of a "u" cell
!     adv_vnu  = advective velocity on the northern face of a "u" cell
!     adv_vbt  = advective velocity on the bottom face of a "T" cell
!     adv_vbu  = advective velocity on the bottom face of a "u" cell

!     rho      = density at center of a "T" cell in units of gm/cm**3
!                note: there is an arbitrary constant which is only a
!                function of depth in "rho". It is related to
!                subtracting a reference level density for purposes of
!                accuracy.

!     grad_p   = hydrostatic pressure gradient for "u" cell. There are
!                two components: (1,2) is for (dp/dx, dp/dy)

      common /mw/ adv_vet(imt,km,jsmw:jmw), adv_vnt(imt,km,1:jmw)
      common /mw/ adv_veu(imt,km,jsmw:jemw)
      common /mw/ adv_vnu(imt,km,1:jemw)
      common /mw/ adv_vbt(imt,0:km,jsmw:jmw)

      common /mw/ adv_vbu(imt,0:km,jsmw:jemw)

      common /mw/ rho(imt,km,jsmw:jmw)

      common /mw/ grad_p(imt,km,jsmw:jemw,2)

!     tmask = tracer cell land/sea mask   = (0.0, 1.0) on (land, sea)
!     umask = velocity cell land/sea mask = (0.0, 1.0) on (land, sea)

      common /mw/ tmask(imt,km,1:jmw), umask(imt,km,1:jmw)

!     these workspace arrays are recalculated for each component of the
!     equations so do not have to be moved as the MW moves northward.

!     adv_fe   = advective flux across the eastern face of a cell
!     adv_fn   = advective flux across the northern face of a cell
!                (removed in most cases and put directly into the
!                 statement functions for speed optimization.)
!     adv_fb   = advective flux across the bottom face of a cell

!     diff_fe  = diffusive flux across the eastern face of a cell
!     diff_fn  = diffusive flux across the northern face of a cell
!     diff_fb  = diffusive flux across the bottom face of a cell
!     source   = source term

      common /mw/ adv_fe(imt,km,jsmw:jemw)

      common /mw/ adv_fb(imt,0:km,jsmw:jemw)

      common /mw/ diff_fe(imt,km,jsmw:jemw)

      common /mw/ diff_fb(imt,0:km,jsmw:jemw)

!     these grid factors are for optimizations and are recalculated as
!     the MW moves northward so they do not have to be moved.

      common /mw/ cstdxtr(imt,jsmw:jmw), cstdxur(imt,jsmw:jmw)
      common /mw/ cstdxt2r(imt,jsmw:jmw),ah_cstdxur(imt,jsmw:jmw)
      common /mw/ csudxur(imt,jsmw:jmw)
      common /mw/ csudxtr(imt,jsmw:jmw)
      common /mw/ csudxu2r(imt,jsmw:jmw),am_csudxtr(imt,jsmw:jmw)

!     these variables are either constant or globally dimensiond by
!     "jmt", so they do not need to be moved as the MW moves northward

!     advmet = coeff for metric advection.
!     cori   = coriolis parameter for velocity component "n"

      common /advec/ advmet(jmt,2)

      common /coriol/ cori(imt,jmt,2)

      common /extwrk/ cf(imt,jmt,-1:1,-1:1)

!     smf = surface momentum flux
!          1 => zonal wind stress (dynes/cm**2)
!          2 => meridional wind stress (dynes/cm**2)
!     bmf = bottom momentum flux
!          1 => zonal bottom drag (dynes/cm**2)
!          2 => meridional bottom drag (dynes/cm**2)
!     stf = surface tracer flux
!          1 => surface heat flux (cal/cm**2/sec = cm*degC/sec = ly/sec)
!          2 => surface salt flux (grams of salt/cm**2/sec)
!     btf = bottom tracer flux (for consistency but normally zero!)
!          1 => bottom heat flux (cal/cm**2/sec = cm*degC/sec = ly/sec)
!          2 => bottom salt flux (grams of salt/cm**2/sec)

      common /mw/ smf(imt,1:jmw,2),  bmf(imt,1:jmw,2)
      common /mw/ stf(imt,1:jmw,nt), btf(imt,1:jmw,nt)

