! source file: /usr/local/models/UVic_ESCM/2.7/source/embm/fluxes.F
      subroutine fluxes (is, ie, js, je)

!=======================================================================
!     calculate energy and moisture fluxes

!     Note: evaporation and precipitation are in g cm-2 s-1
!           and humidities are in g g-1

!     for Thompson and Warren outgoing radiation (see: Thompson S.J.,
!     and S.G. Warren 'parameterization of outgoing ...'J. Atmos. Sci.,
!     39, 2667-2680, 1982.

!     based on code by: A. Fanning, D. Matthews and M. Eby
!=======================================================================

      implicit none

      include "param.h"
      include "cembm.h"
      include "atm.h"
      include "csbc.h"

      include "ice.h"

      include "veg.h"

      integer i, ie, iem1, imax, is, isp1, iter
      integer j, je, jem1, jmax, js, jsp1, maxit

      real b00, b10, b20, b01, b11, b21, b02, b12, b22, b03, b13, b23
      real dalt, Dalton, delta, df, dt, dultnt, dulwr, dusens, emax
      real f, fb, ff, fg, fh, fl, fm, qair, qlnd, rhrh, sr, scrit
      real ssh, tair, tlnd, tlold, tol, ultnt, ulwr, usens, wspd

      isp1 = is + 1
      iem1 = ie - 1
      jsp1 = js + 1
      jem1 = je - 1

!-----------------------------------------------------------------------
!     set appropriate constants
!-----------------------------------------------------------------------
      fb = 0.94*rhoatm*cpatm

      maxit = 10
      tol = 0.01
      emax = 0.0
      imax = 0
      jmax = 0
      scrit = 0.75*soilmax
      dalt = 3.3e-3
      ff = rhoatm*vlocn

!     Thomson and Warren constants

      b00 = 2.43414e2
      b10 = -3.47968e1
      b20 = 1.02790e1
      b01 = 2.60065
      b11 = -1.62064
      b21 = 6.34856e-1
      b02 = 4.40272e-3
      b12 = -2.26092e-2
      b22 = 1.12265e-2
      b03 = -2.05237e-5
      b13 = -9.67e-5
      b23 = 5.62925e-5

      do j=jsp1,jem1
        do i=isp1,iem1

!-----------------------------------------------------------------------
!         set the incoming short wave
!-----------------------------------------------------------------------
          dnswr(i,j) = solins(i,j)*a_calb(i,j)*pass*sbc(i,j,icalb)

!-----------------------------------------------------------------------
!         set wind speed and elevated air temperature
!-----------------------------------------------------------------------
          wspd = sbc(i,j,iws)
          tair = at(i,j,2,isat) - elev(i,j)*rlapse

     &         - hicel(i,j,2)*rlapse

!-----------------------------------------------------------------------
!         calculate outgoing longwave radiation
!-----------------------------------------------------------------------
          rhrh = rh(i,j)*rh(i,j)
          outlwr(i,j) = 1.0e3*(b00 + b10*rh(i,j) + b20*rhrh
     &                + (b01 + b11*rh(i,j) + b21*rhrh)*tair
     &                + (b02 + b12*rh(i,j) + b22*rhrh)*tair**2
     &                + (b03 + b13*rh(i,j) + b23*rhrh)*tair**3)

     &                - anthro

          if (tmsk(i,j) .ge. 0.5) then

!-----------------------------------------------------------------------
!           calculations only for ocean points
!-----------------------------------------------------------------------
            dt = sbc(i,j,isst) - at(i,j,2,isat)
            fg = Dalton(dt, wspd)*wspd

!-----------------------------------------------------------------------
!           calculate evaporation or sublimation (ensure it is positive)
!-----------------------------------------------------------------------
            ssh = cssh*exp(17.67*sbc(i,j,isst)/(sbc(i,j,isst) + 243.5))
            evap(i,j) = max(c0, rhoatm*fg*(ssh - at(i,j,2,ishum)))
            upltnt(i,j) = vlocn*evap(i,j)

!-----------------------------------------------------------------------
!           calculate upward sensible heat flux
!-----------------------------------------------------------------------
            upsens(i,j) = fb*fg*(dt)

!-----------------------------------------------------------------------
!           calculate upward longwave re-radiation
!-----------------------------------------------------------------------
            uplwr(i,j) = esocn*(sbc(i,j,isst) + 273.15)**4
     &                 - esatm*(at(i,j,2,isat) + 273.15)**4

          elseif (land_map(i,j) .ne. 0) then

!----------------------------------------------------------------------
!           set fluxes over land from the land model
!---------------------------------------------------------------------
            upltnt(i,j) = 0.0
            evap(i,j) = sbc(i,j,ievap)
            upsens(i,j) = sbc(i,j,isens)
            uplwr(i,j) = sbc(i,j,ilwr)

          else

!-----------------------------------------------------------------------
!            calculations only for land points

!           find land temperature by balancing the surface heat budget
!             dwsr = ultnt + usens + ulwr
!           using Newton's method:
!             t(i+1) = t(i) - f(t(i))/df(t(i))
!           where:
!             f(t(i)) = dwsr - ultnt - usens - ulwr
!             -df(t(i)) = dultnt - dusens - dulwr
!-----------------------------------------------------------------------
            tlnd = surf(i,j)
            tlold = tlnd
            fm = esatm*(tair + 273.15)**4
            fg = rhoatm

!           calculate stomatal resistance

            sr = (1.-crops(i,j,2))*veg_rs(iveg(i,j))
     &         + crops(i,j,2)*veg_rs(icrops)
            dalt = veg_dalt(i,j)

!           add in aerodynamic resistance
            sr = sr + 1.0/(dalt*wspd + epsln)
!           set beta parameter for calculating actual evaporation
            fh = min(max(c0+epsln, (soilm(i,j,lf)/soilmax)**(0.25)),c1)
!           set coefficients for latent heat (fl) and evaporation (fg)
            fl = fh*ff/(sr)
            fg = fh*fg/(sr)

            dusens = fb*dalt*wspd

!-----------------------------------------------------------------------
!           start loop for all land grid points
!-----------------------------------------------------------------------
            qair = rh(i,j)*cssh*exp(17.67*tair/(tair + 243.5))
            do iter=1,maxit
              qlnd = cssh*exp(17.67*tlnd/(tlnd + 243.5))
              if (qlnd .gt. qair) then
                ultnt = fl*(qlnd - qair)
                dultnt = fl*qlnd*17.67*243.5/(tlnd + 243.5)**2
              else
                ultnt = 0.0
                dultnt = 0.0
              endif
              usens = dusens*(tlnd - tair)
              ulwr = eslnd*(tlnd + 273.15)**4 - fm
              dulwr = 4.0*eslnd*(tlnd + 273.15)**3
              f = dnswr(i,j) - ultnt - usens - ulwr
              df = dultnt + dusens + dulwr
              delta = f/df
              tlnd = tlnd + delta
              if (abs(delta) .le. tol) goto 100
            enddo
!           if not converged, set to last converged temperature
            if (abs(delta) .gt. emax) then
              emax = abs(delta)
              imax = i
              jmax = j
                 tlnd = tlold
            endif

!-----------------------------------------------------------------------
!           calculate fluxes on land
!-----------------------------------------------------------------------
100         continue

            surf(i,j) = tlnd
            qlnd = cssh*exp(17.67*tlnd/(tlnd + 243.5))
            evap(i,j) = max(c0, fg*(qlnd - qair))
            evap(i,j) = max(c0, min(soilm(i,j,lf)/dts, evap(i,j)))

            flux(i,j,ishum) = evap(i,j)*(1.0 - aice(i,j,2))

            upltnt(i,j) = vlocn*evap(i,j)
            upsens(i,j) = dusens*(tlnd - tair)
            uplwr(i,j) = eslnd*(tlnd + 273.15)**4 - fm

!           ensure fluxes are balanced since land can't absorb error
            upsens(i,j) = dnswr(i,j) - upltnt(i,j) - uplwr(i,j)

          endif

        enddo
      enddo

      if (emax .gt. 0.0) write (stdout,*)
     &  '==> Warning: land surface temperature not converging: '
     &, 'emax, i, j, soilm:', emax, imax, jmax, soilm(imax,jmax,2)

      return
      end

      subroutine precipitate (is, ie, js, je)

!=======================================================================
!     calculate precipitation explicitly and update humidity

!     based on code by: A. Fanning, D. Matthews and M. Eby
!=======================================================================

      implicit none

      include "param.h"
      include "scalar.h"
      include "cembm.h"
      include "atm.h"
      include "switch.h"

      include "ice.h"

      include "csbc.h"
      include "mtlm.h"

      integer i, ie, iem1, is, isp1, j, je, jem1, js, jsp1, k, n, negq

      real fb, fc, qmax, rate, tair, soiltemp, pson, psot, ssh

      real hs(imt,jmt)

      data negq /0/
      save negq

      if (eoyear) negq = 0

      isp1 = is + 1
      iem1 = ie - 1
      jsp1 = js + 1
      jem1 = je - 1

!-----------------------------------------------------------------------
!     set appropriate constants
!-----------------------------------------------------------------------
      fb = rhoatm*shq/dts
      fc = dts/rhosno

      call unloadland (POINTS, LYING_SNOW, imt, jmt, land_map, hs)
!     convert from kg/m2 to cm
      hs(:,:) = hs(:,:)*0.1/rhosno

      k = 0
      do j=jsp1,jem1
        do i=isp1,iem1

!-----------------------------------------------------------------------
!         check if specific humidity is greater than 90% of saturation
!-----------------------------------------------------------------------
          tair = at(i,j,2,isat) - elev(i,j)*rlapse*rfactor

     &         - hicel(i,j,2)*rlapse*rfactor

          ssh = cssh*exp(17.67*tair/(tair + 243.5))
          qmax = 0.9*ssh
          if (at(i,j,2,ishum) .gt. qmax) then
            precip(i,j) = fb*(at(i,j,2,ishum) - qmax)
            at(i,j,2,ishum) = qmax
          else
            precip(i,j) = 0.0
            if (at(i,j,2,ishum) .lt. 0.0) then
              at(i,j,2,ishum) = 0.0
              write(stdout,*) '=>Warning: negative q(i,j) at ', i, j
              negq = negq + 1
              if (negq .gt. 50) stop 'negq > 50 in fluxes.f'
              stop
            endif
          endif
          rh(i,j) = max(c0, min(c1, at(i,j,2,ishum)/(ssh + epsln)))

!-----------------------------------------------------------------------
!         calculate snowfall (hsno at tau was set in the ice model)
!-----------------------------------------------------------------------
!         tair may be adjusted by a snowfall offset temperature tsno
          tair = at(i,j,2,isat) - tsno - elev(i,j)*rlapse

     &         - hicel(i,j,2)*rlapse

          psno(i,j) = 0.0

          hs(i,j) = hs(i,j) + hsno(i,j,2)
          if (tair .le. c0 .and. hs(i,j) .lt. hsno_max)
     &      psno(i,j) = min(precip(i,j), hsno_max - hs(i,j))

          if (tmsk(i,j) .ge. 0.5) then
!           only allow snow where there is sea ice
            psno(i,j) = psno(i,j)*aice(i,j,2)
            hsno(i,j,2) = hsno(i,j,2) + fc*psno(i,j)
            if (addflux) flux(i,j,ishum) = flux(i,j,ishum)
     &                                   - dts*psno(i,j)

          elseif (land_map(i,j) .eq. 0) then

            hsno(i,j,2) = hsno(i,j,2) + fc*psno(i,j)
          endif

!-----------------------------------------------------------------------
!         update soilm and allocate surplus soil moisture to runoff
!-----------------------------------------------------------------------

          if (tmsk(i,j) .lt. 0.5 .and. land_map(i,j) .eq. 0) then

            flux(i,j,ishum) = flux(i,j,ishum) - precip(i,j) + psno(i,j)
            soiltemp = soilm(i,j,2)
            soilm(i,j,2) = soilm(i,j,lf) - dts*flux(i,j,ishum)
            soilm(i,j,2) = max(c0, soilm(i,j,2))
            soilm(i,j,1) = soiltemp
            if (soilm(i,j,2) .gt. soilmax) then
              runoff(i,j) = (soilm(i,j,2) - soilmax)/dts
              soilm(i,j,2) = soilmax
            else
              runoff(i,j) = c0
            endif
          endif

        enddo
      enddo

      call embmbc (psno)

      call embmbc (hsno(1,1,2))

      return
      end

      function Dalton (dt, ws)
!=======================================================================
!     calculate the Dalton number based on stability criterion

!     based on code by: A. Fanning
!=======================================================================

      implicit none

      real Dalton, dt, ws

      Dalton = 1.0022e-3 + 8.22e-5*(dt) + 2.66e-6*ws
      if (Dalton .lt. 6.0e-05) Dalton = 6.0e-05
      if (Dalton .gt. 2.2e-03) Dalton = 2.2e-03

      return
      end

      subroutine co2forc
!=======================================================================
!     calculate global average CO2 forcing

!     based on code by: A. Fanning and M. Eby
!=======================================================================

      implicit none

      include "cembm.h"

!-----------------------------------------------------------------------
!     current CO2 concentration is assumed to be 350 ppm
!     relative forcing (anthro) is added to current forcing
!-----------------------------------------------------------------------
      anthro = co2for*alog(co2ccn/350.0)

      return
      end
