!=======================================================================

!            Initialization routines, basic ice sheet model

!   Program bism models ice flow over a 2D vertically-integrated
!   ice sheet, using the Mahaffy (1976) ADI scheme on a spherical grid.
!   Thermomechanical coupling is optional.  When invoked, the vertical
!   dimension is added in the dynamic solution and 3D velocity fields
!   are calculated.  Thermodynamics are solved after Jenssen (1977) in
!   the style of Huybrechts (1988) and followers.  Dynamics and
!   thermodynamics are solved on vertically-transformed co-ordinates
!   with linear interpolation between grids.

!   This is a stream-lined, pure ice-sheet version of the general mixture
!   model which includes ice-stream dynamics.  User inputs are required
!   for: numerical/physical option flags, set in script file "bism_opt",
!        model integration limits: timein, timeout, delt, deltt
!        model dimensions and spatial discretization: phimin, phimax,
!                themin, themax, delphid, delthed (in "bism_par")
!         bed topography: load in "loadtopo.f90"
!         intital ice thickness: load in "loadice.f90"
!         temperature and precipitation: used each timestep in
!                                       "degday.f90" and "surfbal.f90"

!   Script file bism_opt sets on/off flags and bism_par sets global
!   parameters and dimensions at run-time.
!   MODULE global_vars declares global arrays and variables.
!   MODULE global_param defines global model and physical constants.
!   All code is implicit none; though I usually use (a-h,o-z) explicitly
!   for double precision variables and (i-n) explicitly for integers.
!   External to MAIN bism, all variables are locally defined.
!   Global variables are passed as arguments in the subroutine calls,
!   including explicit array dimensions.  Global constants are passed
!   through MODULE global_param.

!   based on code by: S. Marshall
!=======================================================================

#if defined uvic_embm
      subroutine cidm_init(avg_Hice,avg_hi,avg_calv,avg_aice)
#else
      subroutine cidm_init
#endif

#if defined ubc_cidm
!-----------------------------------------------------------------------
!     declare constants and variables
!-----------------------------------------------------------------------

! ... global
      USE global_vars
      USE subgrid_arrays
# if !defined uvic_embm
      USE deg_day_vars
# endif
      USE hydrol_arrays
      USE geol_arrays
      USE icetd_arrays
      USE bedtd_arrays
      USE shelf_arrays
# if defined uvic_embm
      USE cidm_mod_embm
# endif
      implicit none

# if defined uvic_embm
! ... subroutine arguments
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:iimt,1:jjmt) :: avg_Hice,&
                                            avg_hi,avg_calv,avg_aice
# endif

! ... local
# if defined uvic_embm
      integer :: i,j,k
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:,:) :: sum_aice
# else
      integer :: iyear,isumriv
# endif
      LOGICAL :: ltdcur
      REAL(KIND=dp),EXTERNAL :: B

      WRITE(*,*) '******************* SETCIDM START *******************'

!-----------------------------------------------------------------------
!     load basic model parameters
!-----------------------------------------------------------------------

      WRITE(*,*) 'loading basic model parameters...'
      call cidm_input

!-----------------------------------------------------------------------
!     number of tracer grids
!-----------------------------------------------------------------------

      nlon = NINT((phimax-phimin)/delphid)
      nlat = NINT((themax-themin)/delthed)

!-----------------------------------------------------------------------
!     allocate array dimensions and initialize arrays
!-----------------------------------------------------------------------

      WRITE(*,*) 'allocating and initializing arrays...'
      call is_arrays

!-----------------------------------------------------------------------
!     calculate grid-cell geometric attributes
!-----------------------------------------------------------------------

      WRITE(*,*) 'calculating geometric constatnts...'
      call cidm_geom
# if defined uvic_embm
      call cidm_geom_embm
# endif

!-----------------------------------------------------------------------
!     load in vertical grid transforms for ice dynamics
!     (zeta), thermodynamics (xi) and the bed model (xibf, xibc)
!-----------------------------------------------------------------------

      WRITE(*,*) 'setting grid parameters...'
      call is_grids

!-----------------------------------------------------------------------
!     sea level, sealevel (m); bed topography, hg (m), and
!     land mask, iland [1/0/-1]
!-----------------------------------------------------------------------

      sealevel = c0

      WRITE(*,*) 'loading bed topography...'
      call cidm_get_bed(hg0,hg,iland)
      hgk  = hg
      hgkt = hg

!-----------------------------------------------------------------------
!     ice thickness, Hice (m); pressure, pice (Nm^-2)
!-----------------------------------------------------------------------

      WRITE(*,*) 'loading ice thickness...'
      call cidm_get_ice(iland,Hice)
      Hicek  = Hice
      Hicekt = Hice

      pice = rhoi*grav*Hice

!-----------------------------------------------------------------------
!     get surface boundary conditions for atmosphere component
!-----------------------------------------------------------------------

      if (init == 2) then   ! read restart file

!-----------------------------------------------------------------------
!       read restart
!-----------------------------------------------------------------------

        call cidm_rest_read(avg_Hice,avg_hi,avg_calv,avg_aice)

!       this was sofar in cidm_get_ice A.S. 04/25/01 cAAA
        if (ishelf > 0) then    ! shelf-ice option
           WHERE (iland == -1) Hice = c0
        else
           WHERE (iland ==  0) Hice = c0
        endif

        WHERE (Hice < c0) Hice = c0
        WHERE (idomain == -1) Hice = c0

        call cidm_lbc_2d_real(hg)
        call cidm_lbc_2d_real(Hice)

      else

!-----------------------------------------------------------------------
!       convert grids
!-----------------------------------------------------------------------

        call cidm_sbc_hl(Hice,avg_Hice)
        call cidm_sbc_hl(hi,avg_hi)
        avg_calv = c0
        ALLOCATE(sum_aice(1:nlon,1:nlat))
        sum_aice = Hice*rhoi/rhos
        sum_aice = MIN(sum_aice,c1)
        sum_aice = MAX(sum_aice,c0)
        call cidm_sbc_hl(sum_aice,avg_aice)
        DEALLOCATE(sum_aice)
        avg_aice = MAX(avg_aice,c0)
        avg_aice = MIN(avg_aice,c1)

!-----------------------------------------------------------------------
!       write initial snapshot
!-----------------------------------------------------------------------

        isnap = 0
        call cidm_snap_writ(avg_calv)

      endif

      hgk  = hg
      hgkt = hg

      Hicek  = Hice
      Hicekt = Hice

      pice = rhoi*grav*Hice ! pressure, pice (Nm^-2)

!-----------------------------------------------------------------------
!     ice surface elevation, hi (m)
!     floating ice or grounded ice over the continental shelf,
!     ifloating [1/0]
!-----------------------------------------------------------------------

      WRITE(*,*) 'calculating surface elevation...'
      call cidm_get_surf(iland,hg,Hice,ifloating,hi)
      hik  = hi
      hikt = hi
#if !defined uvic_embm
      hi0  = hi
#endif

# if !defined uvic_embm
!-----------------------------------------------------------------------
!     initial topography and land mask
!-----------------------------------------------------------------------

      SELECT CASE(igrid)
      CASE (1,2,4,5,6,7,8)              ! Read in bed topography
        call loadtopo(hg0,hgk,hg,hgkt,igreen,landfrac)
      CASE (3)                          ! Synthetic topography
        call loadmask(hg0,hgk,hg,hgkt)
      CASE DEFAULT
        WRITE(45,*) 'invalid igrid case in bism: igrid = ', igrid
        STOP
      end SELECT

!-----------------------------------------------------------------------
!     ice thickness
!-----------------------------------------------------------------------

      call loadice(hg,hg0,hi,hik,hikt,Hice,Hicek,Hicekt,pice,landfrac,&
                   slope,slope0,hi0,Hocean,iland,ifloating,igreen)
# endif

!-----------------------------------------------------------------------
!     load initial temperature field
!-----------------------------------------------------------------------

      if (itherm == 0) then
        ltdcur = .false.
        tiso = 258.15_dp   ! Isothermal calculation temperature
        Tbar = tiso        ! Array assignment, used for calving
        Biso = B(tiso)
      elseif (itherm == 1) then
        ltdcur = .true.                                  ! Interpolate TD --> Dyn
        call loadtfld(tiso,tsole,hi,Hice)
        if (itdcoup == 1) then                        ! Thermomechanical flow
          ! Load interface temperatures for flux integration
          call fdtrans(tice,ticev,hi,Hice)
        endif
        Biso = B(tiso)
      endif

!-----------------------------------------------------------------------
!     Load table lookup for visco-elastic earth resposne
!-----------------------------------------------------------------------

      if (irebound == 2) then            ! Maxwell Earth model, table lookup
        call load_earth(vert,Hice,load,loadscale)
      endif

!-----------------------------------------------------------------------
!     load geothermal flux
!-----------------------------------------------------------------------

      if (itherm == 1) call loadgeo(geo)

# if !defined uvic_embm
      ! Load surface geology
      if (isurf > 0) then
        call loadsurf
        call sgprop
      endif

      ! Load bedrock geology
      if (ibedr > 0) then
        call loadbedr
      endif

      ! Sediment transport
      if (isilt > 0) then                 ! Load erodabilities
        call loaderode
        call loadsilt
      endif

      ! Sub-grid topographic information (pre-processed)
      if (isubtop > 0) then
        call loadterr(hg)
        if ((isubtop == 2).or.(isubtop == 3)) then
          call calhypso(hg,hg0,hgk,hgkt,igreen)
          WHERE ((igreen==1).or.(icemask==1)) iglacial = 1
        endif
      endif

      ! Bed thermal/groundwater specifications
      if (ibedtd > 0) then
        call loadbed
        call loadpf(hg,iland)
        call loadcov(Hice,icover)
      endif

      ! Climatological data
      if (imass < 5) then                  ! Parameterized climate
        call snowfall(precip)
      else                                ! Climate data
        iyear = 0
        call loadsst(tsea,hi0,hg0,delO,delsea,glacial,iyear)
        call loadsnow(precip,iyear)
      endif
      sealevel = c0

      ! Read in river basin map
      if (igingembre > 0) then
        ! Used in shydrol to sum river fluxes
        isumriv = 1
        if (nlat == 45) then
          call loadriver(hi,hg,hg0,Hice,iland,igreen,isumriv)
        elseif (nlat == 100) then
          call loadriver_fine(hi,hg,hg0,Hice,iland,igreen,isumriv)
        endif
      endif

      ! Write initial fields
      time = stime
      call writ(hg,Hice,time)
      if (nlon==120) then        ! Save N Amer startup information
        if (isubtop == 0) then
          call writna(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                      calve,tair,tsea,precip,igreen,time)
        elseif (isubtop == 2) then                ! subgrid diagnostics
          call writna_sub(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                          calve,tair,tsea,precip,igreen,time)
        endif
      endif

# endif

!-----------------------------------------------------------------------
!     calculate global ice volume
!-----------------------------------------------------------------------

      istart = 0
      if (init == 2) then   ! read previous ice volume data
        call cidm_vol_read(vol_SH,vol_NH)
      else
        stime  = c0
        call cidm_vol_calc(Hice,vol_SH,vol_NH)
        call cidm_vol_writ(vol_SH,vol_NH)
      endif

!-----------------------------------------------------------------------
!     diagnostics
!-----------------------------------------------------------------------

      idtvol  = INT(dtvol/delt)
      idtsnap = INT(dtsnap/delt)
      idtrest = INT(dtrest/delt)

      call cidm_summary
      WRITE(*,*) '*******************  SETCIDM end  *******************'

#endif
!=======================================================================
      end subroutine cidm_init
!=======================================================================
