!=======================================================================
!     Define physical and model constants
!     Declare model options and dimensions, for global accessibility
!=======================================================================

      MODULE global_param

      implicit none
      SAVE

!-----------------------------------------------------------------------
!     numerical precision
!-----------------------------------------------------------------------

      INTEGER, parameter :: sp = SELECTED_REAL_KIND(P=6)
#if defined uvic_embm
      INTEGER, parameter :: dp = 8
#else
      INTEGER, parameter :: dp = SELECTED_REAL_KIND(PRECISION(1.0d0))
#endif

!-----------------------------------------------------------------------
!     numerical constants
!-----------------------------------------------------------------------

      REAL(KIND=dp),PARAMETER :: c0 = 0.0_dp
      REAL(KIND=dp),PARAMETER :: c1 = 1.0_dp
      REAL(KIND=dp),PARAMETER :: c2 = 2.0_dp
      REAL(KIND=dp),PARAMETER :: c3 = 3.0_dp
      REAL(KIND=dp),PARAMETER :: c4 = 4.0_dp
      REAL(KIND=dp),PARAMETER :: c5 = 5.0_dp
      REAL(KIND=dp),PARAMETER :: c6 = 6.0_dp
      REAL(KIND=dp),PARAMETER :: c7 = 7.0_dp
      REAL(KIND=dp),PARAMETER :: c8 = 8.0_dp
      REAL(KIND=dp),PARAMETER :: c9 = 9.0_dp
      REAL(KIND=dp),PARAMETER :: c10 = 10.0_dp
      REAL(KIND=dp),PARAMETER :: c12 = 12.0_dp
      REAL(KIND=dp),PARAMETER :: c90 = 90.0_dp
      REAL(KIND=dp),PARAMETER :: c100 = 100.0_dp
      REAL(KIND=dp),PARAMETER :: r2 = c1/c2
      REAL(KIND=dp),PARAMETER :: r3 = c1/c3
      REAL(KIND=dp),PARAMETER :: r6 = c1/c6
      REAL(KIND=dp),PARAMETER :: r12 = c1/c12
      REAL(KIND=dp),PARAMETER :: p25 = c1/c4
      REAL(KIND=dp),PARAMETER :: p75 = c3/c4
      REAL(KIND=dp),PARAMETER :: spval = 1.E38_dp

      ! Universal physical constants
      REAL(KIND=dp), parameter :: grav = 9.81_dp
      REAL(KIND=dp), parameter :: erad = 6.370e6_dp !cAAA 04/25/01
      REAL(KIND=dp), parameter :: pi = 3.14159265358979_dp
      REAL(KIND=dp), parameter :: minute = 60._dp
      REAL(KIND=dp), parameter :: hour = 3600._dp
      REAL(KIND=dp), parameter :: day = hour*24._dp
      REAL(KIND=dp), parameter :: month = day*365.24_dp/c12
      REAL(KIND=dp), parameter :: year = day*365.24_dp
      REAL(KIND=dp), parameter :: degtrad = pi/180._dp
      REAL(KIND=dp), parameter :: shelfd = -120._dp    ! m below sea level
      INTEGER, parameter :: nmonth = 12                       ! Months
      INTEGER, parameter :: ncoord = 2                 ! Co-ordinate components
      INTEGER, parameter :: nface = 4                  ! Co-ordinate components

      ! Ice properties (modified SI - annum)
#if defined uvic_embm
      REAL(KIND=dp), parameter :: rhoi = 913._dp
      REAL(KIND=dp), parameter :: rhos = 330._dp
#else
      REAL(KIND=dp), parameter :: rhoi = 910._dp
#endif
      REAL(KIND=dp), parameter :: htci = 2009._dp
      REAL(KIND=dp), parameter :: tcondi = 2.1_dp*year
      REAL(KIND=dp), parameter :: tdiffi = tcondi/(rhoi*htci)
      REAL(KIND=dp), parameter :: tlat = 3.35e5_dp
      INTEGER, parameter :: gn  = 3
      INTEGER, parameter :: gnp1 = gn + 1
      INTEGER, parameter :: gnp2 = gn + 2
      INTEGER, parameter :: gnm1 = gn - 1
      INTEGER, parameter :: gnm2 = gn - 2
      INTEGER, parameter :: exponent = gnm1/2
      INTEGER, parameter :: derivexp = (gn-3)/2
      INTEGER, parameter :: strnhtexp = gnp1/2
      REAL(KIND=dp), parameter :: strinvexp = r3     ! (n-1)/2n
      REAL(KIND=dp), parameter :: Rgas  = 8.314_dp
      REAL(KIND=dp), parameter :: Qactwarm = 139000._dp
      REAL(KIND=dp), parameter :: Qactcold = 60000._dp
      REAL(KIND=dp), parameter :: Qactiso = 60700._dp
      REAL(KIND=dp), parameter :: Bowarm = 5.47e10_dp*c3
      REAL(KIND=dp), parameter :: Bocold = 1.14e-5_dp*c3
      REAL(KIND=dp), parameter :: Boiso = 4.0562e-5_dp*c3
      REAL(KIND=dp), parameter :: tnoice = 260.0_dp
      REAL(KIND=dp), parameter :: glen_min = 208.15_dp
      REAL(KIND=dp), parameter :: glen_max = 273.15_dp
      REAL(KIND=dp), parameter :: glen_inc = 0.01_dp

      ! Bed properties (modified SI - annum)
      REAL(KIND=dp), parameter :: rhoe = 3300._dp
!      REAL(KIND=dp), parameter :: tau = 5000._dp
      REAL(KIND=dp), parameter :: tau = 3000._dp
      REAL(KIND=dp), parameter :: Hbed = 2000._dp

      ! Water properties (modified SI - annum)
#if defined uvic_embm
      REAL(KIND=dp), parameter :: rhow = 1035._dp
#else
      REAL(KIND=dp), parameter :: rhow = 1028._dp
#endif
      REAL(KIND=dp), parameter :: htcw = 4183._dp
      REAL(KIND=dp), parameter :: tcondw = 1._dp*year
      REAL(KIND=dp), parameter :: tdiffw = tcondw/(rhow*htcw)
      REAL(KIND=dp), parameter :: iceberg = c1 - rhoi/rhow

      ! Mass balance model constants
#if defined uvic_embm
      REAL(KIND=dp), parameter :: lapse = .0065_dp      ! Deg C/m
#else
      REAL(KIND=dp), parameter :: lapse = .0075_dp      ! Deg C/m
#endif
      REAL(KIND=dp), parameter :: plapse = 1000.0_dp   ! m
      REAL(KIND=dp), parameter :: ablapse = 0.002_dp   ! (m/a) m^{-1}
      REAL(KIND=dp) :: kcalf,expcalf                   ! Calving parameters

      ! Define second order `constants'
      !   k(T) = 9.828*exp(-5.7d-3 T)
      !   dkdT(T) = -0.05602 exp(-5.7d-3 T)
      !   C(T) = 152.5 + 7.122 T
      REAL(KIND=dp), parameter :: triplept = 273.15_dp
      REAL(KIND=dp), parameter :: fptdep = 8.7e-4_dp   ! Deg C/metre of ice
      REAL(KIND=dp), parameter :: kice1 = 9.828_dp
      REAL(KIND=dp), parameter :: kice2 = -5.7e-3_dp
      REAL(KIND=dp), parameter :: kice3 = kice1*kice2
      REAL(KIND=dp), parameter :: cice1 = 152.5_dp
      REAL(KIND=dp), parameter :: cice2 = 7.122_dp

      ! Define model zero, physical zero
      REAL(KIND=dp), parameter :: zerop = 0.1_dp           ! physical zero
      REAL(KIND=dp), parameter :: zerom = 1.e-15_dp        ! machine zero
      REAL(KIND=dp), parameter :: fimp = c1                   ! TD implicitness
      REAL(KIND=dp), parameter :: afimp = c1 - fimp        ! " implicitness
      REAL(KIND=dp), parameter :: fimpd = r2
      REAL(KIND=dp), parameter :: afimpd = c1 - fimpd

      ! Define vertical co-ordinate system for dynamics
      ! zeta = EXP(aa(hi-z)/H);   *Use negative sense*
      REAL(KIND=dp), parameter :: zetatop = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: zeta

      ! Define vertical co-ordinate system for thermodynamics
      ! xi = (hi-z)/H;   *Use positive sense*
      REAL(KIND=dp), parameter :: xitop = c0
      REAL(KIND=dp), parameter :: xibot = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: xi

      ! Define vertical co-ordinate system for lower bed (unfrozen)
      ! xibc = (hg-Hbedf-z)/(Hbed-Hbedf);  *Use positive sense*
      REAL(KIND=dp), parameter :: xibctop = c0
      REAL(KIND=dp), parameter :: xibcbot = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: xibc

      ! Define vertical co-ordinate system for upper bed (frozen zone)
      ! xibhi = (hg-z)/Hbedf;    *Use positive sense*
      REAL(KIND=dp), parameter :: xibftop = c0
      REAL(KIND=dp), parameter :: xibfbot = c1
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: xibf

      ! Grid transformation dimensions
      REAL(KIND=dp) :: delzeta,delxi,delxibc,delxibf
      REAL(KIND=dp) :: aa          ! Dynamical grid transformation parameter

!***************************************************************************
      ! Declare grid properties and input parameters
      integer :: ntime,istart         ! Integration limits
      integer :: idelt,ideltt,idelth,ideltg,ideltb         ! Time steps
      integer :: ihydnest                                   ! Nested soln times
      REAL(KIND=dp) :: delt,deltt,delth,deltg,deltb        ! Time steps
      REAL(KIND=dp) :: hydnest                                   ! Nested soln times
      REAL(KIND=dp) :: themin,themax,phimin,phimax         ! Grid limits
      REAL(KIND=dp) :: delthed,delphid,deltheta,delphi     ! Grid resolution
      REAL(KIND=dp) :: dx,dy,dxy,dxdx,dydy,Biso            ! Grid parameters
      INTEGER, parameter :: nglen_table = 6500
      REAL(KIND=dp), ALLOCATABLE, DIMENSION(:) :: Btable   ! Glen parameter

!-----------------------------------------------------------------------
!     mask
!-----------------------------------------------------------------------

      INTEGER,ALLOCATABLE,DIMENSION(:,:) :: idomain
#if !defined uvic_embm
      INTEGER,ALLOCATABLE,DIMENSION(:,:) :: icemask
#endif

!-----------------------------------------------------------------------
!     geometry
!-----------------------------------------------------------------------

      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: lon      ! longitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: lat      ! latitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: area     ! Cell area (m^2)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: delx     ! Cell width  (m)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: delxhi   ! Interface width (m)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: sine     ! sine of latitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: sinehi   ! sine of inter. lat.
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: clat     ! colatitude
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: clathi   ! interface colat.
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: Bjvect   ! Bj (integration)
      REAL(KIND=dp),ALLOCATABLE,DIMENSION(:) :: Cjvect   ! Cj (integration)
#if !defined uvic_embm
      REAL(KIND=dp),ALLOCATABLE, DIMENSION(:,:) :: latitude  ! Cell latitude
#endif

!     REAL(KIND=dp), parameter :: themin = 7.644234_dp    ! EISMINT tests
!     REAL(KIND=dp), parameter :: themax = 7.644234_dp    ! Equatorial block
!     REAL(KIND=dp), parameter :: phimin = -7.644234_dp
!     REAL(KIND=dp), parameter :: phimax = 7.644234_dp
!     REAL(KIND=dp), parameter :: delthed = .449661_dp    ! 50 km
!     REAL(KIND=dp), parameter :: delphid = .449661_dp

!     REAL(KIND=dp), parameter :: themin = 35._dp         ! Laurentide
!     REAL(KIND=dp), parameter :: themax = 80._dp
!     REAL(KIND=dp), parameter :: phimin = 195._dp
!     REAL(KIND=dp), parameter :: phimax = 315._dp
!     REAL(KIND=dp), parameter :: delthed = 1._dp         ! 1 degree
!     REAL(KIND=dp), parameter :: delphid = 1._dp

!     REAL(KIND=dp), parameter :: themin = 53.0_dp     ! Alaskan peninsula
!     REAL(KIND=dp), parameter :: themax = 56.0_dp
!     REAL(KIND=dp), parameter :: phimin = -163.5_dp
!     REAL(KIND=dp), parameter :: phimax = -161._dp
!     REAL(KIND=dp), parameter :: delthed = 1.0_dp/24.0_dp    ! 4.6125 km
!     REAL(KIND=dp), parameter :: delphid = 1.0_dp/12.0_dp    ! 5.3715 km

!***************************************************************************
      ! Declare integer option flags, globally-accesible
      integer :: igrid           ! Grid type  1/2/3
      integer :: init            ! Initial field treatment 1/2/3
      integer :: imass           ! Mass balance treatment  1/2/3/4/5/6
      integer :: iclim           ! Climate data type/source  1/2/3/4/5/6
      integer :: itherm          ! Thermodynamic treatment  0/1
      integer :: itdcoup         ! Coupled dynamics-TD?  1/0
      integer :: itdptvar        ! T-dept ice properties?  1/0
      integer :: istrn           ! Strain heating discretization  0/1
      integer :: idisc           ! Discretization for TD (Up/centred) 3/4/5
      integer :: idyn            ! Dynamical treatment  1/2
      integer :: islid           ! Sliding?  1/0
      integer :: itrack          ! Particle tracking?  1/0
      integer :: isilt           ! Englacial debris?  1/0
      integer :: iheat           ! Geothermal heat treatment 1/2
      integer :: ibedtd          ! Bed TD treatment   0/1/2
      integer :: ibedmo          ! Bed TD solution method   1/2
      integer :: isurf           ! Surface geology treatment  0/1/2
      integer :: ibedr           ! Bed geology treatment  0/1
      integer :: isubtop         ! Subgrid topo treatment  0/1
      integer :: igingembre      ! Surface hydrological treatment  0/1/2
      integer :: isurfhyd        ! Surface water flux treatment  1/2
      integer :: irebound         ! Model isostatic compensation? 0/1
      integer :: ishelf                 ! Model floating ice and calving? 0/1
      integer :: ialbedo         ! Apply Susan Marshall albedo code? 0/1
      integer :: itest           ! result output test 0/1

      ! Array dimensions
      integer :: nlon,nlat,nvert        ! Grid dimensions
      integer :: nlonp,nlatp,nvertp     ! Physical grid dimensions
      integer :: nxi,nzeta              ! Transformed vertical grid dimensions
      integer :: njac1,njac2                 ! Jacobian matrix dimensions
      integer :: ncore                  ! Ice core climate proxy
      integer :: nducks                 ! Particle tracers
      integer :: ntrack                 ! r-t tracking variables
      integer :: nsilt                  ! Geologic debris units
      integer :: nsurf,nbedr            ! Surface, bedrock materials
      integer :: nsedtype               ! Sediment `type' divisions
      integer :: nvertb                 ! Crustal grid dimension
      integer :: nxibc,nxibf            ! Lower, upper crust grid size
      integer :: nbin                   ! Directional bins for striae
      integer :: ntable                 ! Number of entries, PDD lookup table
      integer :: npddamp                ! Cross-entries, PDD lookup table
      integer :: nhyps                  ! Number of hypsometric bins
      integer :: ndepw                  ! Dependencies in water sheet eqns
      integer :: njacw                  ! Subglacial water unknowns
      integer :: njw                    ! Subglacial water unknowns
      integer :: nwsys                  ! Sparse matrix dimension, water sheet
      integer :: ndepd                  ! Dependencies in dynamical eqns
      integer :: njacd                  ! Dynamical system unknowns
      integer :: ndsys                  ! Sparse matrix dimension, dynamics
      integer :: ntsys                        ! Full ice/bed TD system size
      integer :: ntfrz                        ! Vertical level of permafrost front
      integer :: ntisos

      integer :: itime                  ! current time

      REAL(KIND=dp) :: sealevel

! ... time
      REAL(KIND=dp) :: stime    ! start time
      REAL(KIND=dp) :: ctime    ! current time
      REAL(KIND=dp) :: etime    ! end time

! ... snapshot output
      integer :: isnap          ! current number of snapshot output
      integer :: nsnap          ! maximum number of snapshot output
      REAL(KIND=dp) :: dtsnap   ! snapshot output interval
      integer :: idtsnap

! ... ice volume output
      integer :: nvol           ! maximum number of ice volume output
      REAL(KIND=dp) :: dtvol    ! ice volume output interval
      integer :: idtvol

! ... restart
      REAL(KIND=dp) :: dtrest   ! restart file output interval
      integer :: idtrest

!=======================================================================
      end MODULE global_param
!=======================================================================
