      subroutine is_PCHEV(N,X,F,D,NVAL,XVAL,FVAL,DVAL,IERR)
!***BEGIN PROLOGUE  PCHEV
!***DATE WRITTEN   870828   (YYMMDD)
!***REVISION DATE  870828   (YYMMDD)
!***CATEGORY NO.  E3,H1
!***KEYWORDS  CUBIC HERMITE OR SPLINE DIFFERENTIATION,CUBIC HERMITE
!             EVALUATION,EASY TO USE SPLINE OR CUBIC HERMITE EVALUATOR
!***AUTHOR  KAHANER, D.K., (NBS)
!             SCIENTIFIC COMPUTING DIVISION
!             NATIONAL BUREAU OF STANDARDS
!             ROOM A161, TECHNOLOGY BUILDING
!             GAITHERSBURG, MARYLAND 20899
!             (301) 975-3808
!***PURPOSE  Evaluates the function and first derivative of a piecewise
!            cubic Hermite or spline function at an array of points XVAL,
!            easy to use.
!***DESCRIPTION

!          PCHEV:  Piecewise Cubic Hermite or Spline Derivative Evaluator,
!                  Easy to Use.

!     From the book "Numerical Methods and Software"
!          by  D. Kahaner, C. Moler, S. Nash
!                 Prentice Hall 1988

!     Evaluates the function and first derivative of the cubic Hermite
!     or spline function defined by  N, X, F, D, at the array of points XVAL.

!     This is an easy to use driver for the routines by F.N. Fritsch
!     described in reference (2) below. Those also have other capabilities.

! ----------------------------------------------------------------------

!  Calling sequence: call  PCHEV (N, X, F, D, NVAL, XVAL, FVAL, DVAL, IERR)

!     integer  N, NVAL, IERR
!     real  X(N), F(N), D(N), XVAL(NVAL), FVAL(NVAL), DVAL(NVAL)

!   Parameters:

!     N -- (input) number of data points.  (Error return if N.lt.2 .)

!     X -- (input) real array of independent variable values.  The
!           elements of X must be strictly increasing:
!             X(I-1) .lt. X(I),  I = 2(1)N. (Error return if not.)

!     F -- (input) real array of function values.  F(I) is
!           the value corresponding to X(I).

!     D -- (input) real array of derivative values.  D(I) is
!           the value corresponding to X(I).

!  NVAL -- (input) number of points at which the functions are to be
!           evaluated. ( Error return if NVAL.lt.1 )

!  XVAL -- (input) real array of points at which the functions are to
!           be evaluated.

!          NOTES:
!           1. The evaluation will be most efficient if the elements
!              of XVAL are increasing relative to X;
!              that is,   XVAL(J) .ge. X(I)
!              implies    XVAL(K) .ge. X(I),  all K.ge.J .
!           2. If any of the XVAL are outside the interval [X(1),X(N)],
!              values are extrapolated from the nearest extreme cubic,
!              and a warning error is returned.

!  FVAL -- (output) real array of values of the cubic Hermite function
!           defined by  N, X, F, D  at the points  XVAL.

!  DVAL -- (output) real array of values of the first derivative of
!           the same function at the points  XVAL.

!  IERR -- (output) error flag.
!           Normal return:
!              IERR = 0  (no errors).
!           Warning error:
!              IERR.gt.0  means that extrapolation was performed at
!                 IERR points.
!           "Recoverable" errors:
!              IERR = -1  if N.lt.2 .
!              IERR = -3  if the X-array is not strictly increasing.
!              IERR = -4  if NVAL.lt.1 .
!           (Output arrays have not been changed in any of these cases.)
!               NOTE:  The above errors are checked in the order listed,
!                   and following arguments have **NOT** been validated.
!              IERR = -5  if an error has occurred in the lower-level
!                         routine CHFDV.  NB: this should never happen.
!                         Notify the author **IMMEDIATELY** if it does.

! ----------------------------------------------------------------------
!***REFERENCES  1. F.N.FRITSCH AND R.E.CARLSON, 'MONOTONE PIECEWISE
!                 CUBIC INTERPOLATION,' SIAM J.NUMER.ANAL. 17, 2 (APRIL
!                 1980), 238-246.
!               2. F.N.FRITSCH, 'PIECEWISE CUBIC HERMITE INTERPOLATION
!                 PACKAGE, FINAL SPECIFICATIONS', LAWRENCE LIVERMORE
!                 NATIONAL LABORATORY, COMPUTER DOCUMENTATION UCID-30194,
!                 AUGUST 1982.
!***ROUTINES CALLED  PCHFD
!***END PROLOGUE  PCHEV
      integer  N, NVAL, IERR
      real  X(N), F(N), D(N), XVAL(NVAL), FVAL(NVAL), DVAL(NVAL)

!  DECLARE LOCAL VARIABLES.

      integer INCFD
      LOGICAL SKIP
      DATA SKIP /.true./
      DATA INCFD /1/

!***FIRST EXECUTABLE STATEMENT  PCHEV

      call IS_PCHFD(N,X,F,D,INCFD,SKIP,NVAL,XVAL,FVAL,DVAL,IERR)

 5000 CONTINUE
      return

!------------- LAST LINE OF PCHEV FOLLOWS ------------------------------
      end
      subroutine IS_PCHFD(N,X,F,D,INCFD,SKIP,NE,XE,FE,DE,IERR)
!***BEGIN PROLOGUE  PCHFD
!     THIS PROLOGUE HAS BEEN REMOVED FOR REASONS OF SPACE
!     FOR A COMPLETE COPY OF THIS ROUTINE CONTACT THE AUTHORS
!     From the book "Numerical Methods and Software"
!          by  D. Kahaner, C. Moler, S. Nash
!               Prentice Hall 1988
!***END PROLOGUE  PCHFD

!  DECLARE ARGUMENTS.

      integer  N, INCFD, NE, IERR
      real  X(N), F(INCFD,N), D(INCFD,N), XE(NE), FE(NE), DE(NE)
      LOGICAL  SKIP

!  DECLARE LOCAL VARIABLES.

      integer  I, IERC, IR, J, JFIRST, NEXT(2), NJ

!  VALIDITY-CHECK ARGUMENTS.

!***FIRST EXECUTABLE STATEMENT  PCHFD
      if (SKIP)  GO TO 5

      if ( N.lt.2 )  GO TO 5001
      if ( INCFD.lt.1 )  GO TO 5002
      do 1  I = 2, N
         if ( X(I).le.X(I-1) )  GO TO 5003
    1 CONTINUE

!  FUNCTION DEFINITION IS OK, GO ON.

    5 CONTINUE
      if ( NE.lt.1 )  GO TO 5004
      IERR = 0
      SKIP = .true.

!  LOOP OVER INTERVALS.        (   INTERVAL INDEX IS  IL = IR-1  . )
!                              ( INTERVAL IS X(IL).le.X.lt.X(IR) . )
      JFIRST = 1
      IR = 2
   10 CONTINUE

!     SKIP OUT OF LOOP if HAVE PROCESSED ALL EVALUATION POINTS.

         if (JFIRST .gt. NE)  GO TO 5000

!     LOCATE ALL POINTS IN INTERVAL.

         do 20  J = JFIRST, NE
            if (XE(J) .ge. X(IR))  GO TO 30
   20    CONTINUE
         J = NE + 1
         GO TO 40

!     HAVE LOCATED FIRST POINT BEYOND INTERVAL.

   30    CONTINUE
         if (IR .eq. N)  J = NE + 1

   40    CONTINUE
         NJ = J - JFIRST

!     SKIP EVALUATION if NO POINTS IN INTERVAL.

         if (NJ .eq. 0)  GO TO 50

!     EVALUATE CUBIC AT XE(I),  I = JFIRST (1) J-1 .

!       ----------------------------------------------------------------
        call IS_CHFDV (X(IR-1),X(IR), F(1,IR-1),F(1,IR), D(1,IR-1),
     *       D(1,IR),NJ, XE(JFIRST), FE(JFIRST), DE(JFIRST), NEXT, IERC)
!       ----------------------------------------------------------------
         if (IERC .lt. 0)  GO TO 5005

         if (NEXT(2) .eq. 0)  GO TO 42
!        if (NEXT(2) .gt. 0)  then
!           IN THE CURRENT SET OF XE-POINTS, THERE ARE NEXT(2) TO THE
!           RIGHT OF X(IR).

            if (IR .lt. N)  GO TO 41
!           if (IR .eq. N)  then
!              THESE ARE ACTUALLY EXTRAPOLATION POINTS.
               IERR = IERR + NEXT(2)
               GO TO 42
   41       CONTINUE
!           else
!              WE SHOULD NEVER HAVE GOTTEN HERE.
               GO TO 5005
!           endif
!        endif
   42    CONTINUE

         if (NEXT(1) .eq. 0)  GO TO 49
!        if (NEXT(1) .gt. 0)  then
!           IN THE CURRENT SET OF XE-POINTS, THERE ARE NEXT(1) TO THE
!           LEFT OF X(IR-1).

            if (IR .gt. 2)  GO TO 43
!           if (IR .eq. 2)  then
!              THESE ARE ACTUALLY EXTRAPOLATION POINTS.
               IERR = IERR + NEXT(1)
               GO TO 49
   43       CONTINUE
!           else
!              XE IS NOT ORDERED RELATIVE TO X, SO MUST ADJUST
!              EVALUATION INTERVAL.

!              FIRST, LOCATE FIRST POINT TO LEFT OF X(IR-1).
               do 44  I = JFIRST, J-1
                  if (XE(I) .lt. X(IR-1))  GO TO 45
   44          CONTINUE
!              NOTE-- CANNOT DROP THROUGH HERE UNLESS THERE IS AN ERROR
!                     IN CHFDV.
               GO TO 5005

   45          CONTINUE
!              RESET J.  (THIS WILL BE THE NEW JFIRST.)
               J = I

!              NOW FIND OUT HOW FAR TO BACK UP IN THE X-ARRAY.
               do 46  I = 1, IR-1
                  if (XE(J) .lt. X(I)) GO TO 47
   46          CONTINUE
!              NB-- CAN NEVER DROP THROUGH HERE, SINCE XE(J).lt.X(IR-1).

   47          CONTINUE
!              AT THIS POINT, EITHER  XE(J) .lt. X(1)
!                 OR      X(I-1) .le. XE(J) .lt. X(I) .
!              RESET IR, RECOGNIZING THAT IT WILL BE INCREMENTED BEFORE
!              CYCLING.
               IR = MAX0(1, I-1)
!           endif
!        endif
   49    CONTINUE

         JFIRST = J

!     end OF IR-LOOP.

   50 CONTINUE
      IR = IR + 1
      if (IR .le. N)  GO TO 10

!  NORMAL return.

 5000 CONTINUE
      return

!  ERROR returnS.

 5001 CONTINUE
!     N.lt.2 return.
      IERR = -1
      call IS_XERROR ('PCHFD -- NUMBER OF DATA POINTS LESS THAN TWO'
     *           , 44, IERR, 1)
      return

 5002 CONTINUE
!     INCFD.lt.1 return.
      IERR = -2
      call IS_XERROR ('PCHFD -- INCREMENT LESS THAN ONE'
     *           , 32, IERR, 1)
      return

 5003 CONTINUE
!     X-ARRAY NOT STRICTLY INCREASING.
      IERR = -3
      call IS_XERROR ('PCHFD -- X-ARRAY NOT STRICTLY INCREASING'
     *           , 40, IERR, 1)
      return

 5004 CONTINUE
!     NE.lt.1 return.
      IERR = -4
      call IS_XERROR ('PCHFD -- NUMBER OF EVALUATION POINTS LESS THAN
     * ONE' , 50, IERR, 1)
      return

 5005 CONTINUE
!     ERROR return FROM CHFDV.
!   *** THIS CASE SHOULD NEVER OCCUR ***
      IERR = -5
      call IS_XERROR ('PCHFD -- ERROR return FROM CHFDV -- FATAL'
     *           , 41, IERR, 2)
      return
!------------- LAST LINE OF PCHFD FOLLOWS ------------------------------
      end
      subroutine IS_CHFDV(X1,X2,F1,F2,D1,D2,NE,XE,FE,DE,NEXT,IERR)
!***BEGIN PROLOGUE  CHFDV
!     THIS PROLOGUE HAS BEEN REMOVED FOR REASONS OF SPACE
!     FOR A COMPLETE COPY OF THIS ROUTINE CONTACT THE AUTHORS
!     From the book "Numerical Methods and Software"
!          by  D. Kahaner, C. Moler, S. Nash
!               Prentice Hall 1988
!***END PROLOGUE  CHFDV

!  DECLARE ARGUMENTS.

      integer  NE, NEXT(2), IERR
      real  X1, X2, F1, F2, D1, D2, XE(NE), FE(NE), DE(NE)

!  DECLARE LOCAL VARIABLES.

      integer  I
      real  C2, C2T2, C3, C3T3, DEL1, DEL2, DELTA, H, X, XMI, XMA, ZERO
      DATA  ZERO /0./

!  VALIDITY-CHECK ARGUMENTS.

!***FIRST EXECUTABLE STATEMENT  CHFDV
      if (NE .lt. 1)  GO TO 5001
      H = X2 - X1
      if (H .eq. ZERO)  GO TO 5002

!  INITIALIZE.

      IERR = 0
      NEXT(1) = 0
      NEXT(2) = 0
      XMI = AMIN1(ZERO, H)
      XMA = AMAX1(ZERO, H)

!  COMPUTE CUBIC COEFFICIENTS (EXPANDED ABOUT X1).

      DELTA = (F2 - F1)/H
      DEL1 = (D1 - DELTA)/H
      DEL2 = (D2 - DELTA)/H
!                                           (DELTA IS NO LONGER NEEDED.)
      C2 = -(DEL1+DEL1 + DEL2)
      C2T2 = C2 + C2
      C3 = (DEL1 + DEL2)/H
!                               (H, DEL1 AND DEL2 ARE NO LONGER NEEDED.)
      C3T3 = C3+C3+C3

!  EVALUATION LOOP.

      do 500  I = 1, NE
         X = XE(I) - X1
         FE(I) = F1 + X*(D1 + X*(C2 + X*C3))
         DE(I) = D1 + X*(C2T2 + X*C3T3)
!          COUNT EXTRAPOLATION POINTS.
         if ( X.lt.XMI )  NEXT(1) = NEXT(1) + 1
         if ( X.gt.XMA )  NEXT(2) = NEXT(2) + 1
!        (NOTE REDUNDANCY--IF EITHER CONDITION IS TRUE, OTHER IS FALSE.)
  500 CONTINUE

!  NORMAL return.

      return

!  ERROR returnS.

 5001 CONTINUE
!     NE.lt.1 return.
      IERR = -1
      call IS_XERROR ('CHFDV -- NUMBER OF EVALUATION POINTS LESS THAN
     * ONE' , 50, IERR, 1)
      return

 5002 CONTINUE
!     X1.eq.X2 return.
      IERR = -2
      call IS_XERROR ('CHFDV -- INTERVAL endPOINTS EQUAL'
     *           , 33, IERR, 1)
      return
!------------- LAST LINE OF CHFDV FOLLOWS ------------------------------
      end
