!******************************************************************************
!   Solves 3D thermal balance over the ice sheet, including vertical
!   diffusion and advection (semi-implicit, CD) and horizontal advection
!   (explicit, upstream differencing).  Strain heating is explicit.
!   The equation is transformed to the uniformly-scaled vertical co-ordinate
!   system xi = (hi-z)/H.  Solves matrix system AT^{m+1} = BT^{m}+c, where
!   A, B are effectively tri-diagonal and c contains lots of explicit
!   T_k terms.  T(1) is the surface temperature, specified from the EBM,
!   and T(nvertp) is the basal ice temperature.  (Positive downwards in
!   the transformed grid: xi=0 to xi=1).  Heat flux into the ice is
!   dictated by  (a) constant geothermal flux (ibedtd=0)
!                (b) bed temperature gradients (ibedtd=1)

!   This module performs the full balance, calling on routine updatet.f90
!   to  invert the solution for each column of ice.  Work with local
!   column vectors tcolij, then reload into tice on completion.

!   Option itdnl=0 ensures solution from linear matrix system.
!   Where strain heating is high, a non-linear Newton iteration
!   may be more optimal; flag itdnl=1 enables this option.

!   Rapid nested time loops are included to promote stability in thin
!   ice.  Time steps must be a factor of the global timestep
!   delt.  Crank-Nicholson stability requirements for diffusion-dominant
!   heat flow give the rough time step requirements (nz=20):
!         ithick = 1      H > 1000 m   delt < 70 a
!         ithick = 2      H >  800 m   delt < 10 a
!         ithick = 3      H >  500 m   delt <  5 a
!         ithick = 4      H >  200 m   delt <  2 a
!         ithick = 5      H >  100 m   delt <  1 a
!         ithick = 6      H >   50 m   delt < .2 a
!   Ice less than 50 m thick is not modelled: linear temperature profiles
!   which conserve energy are defined, constrained by surface temperature.

!*****************************************************************************
    subroutine tfield(tair,ub,vb,hi,hikt,hg,hgkt,Hice,Hicekt,&
                      pice,geo,rest,dtdt,time,coimp,coexp,cocon,&
                      tiso,tsea,tsole,Tbar,bmelt,Hocean,ifloating)

!*****************************************************************************
      USE icetd_arrays
      implicit none
      INTEGER, INTENT(IN) :: time
      INTEGER, INTENT(IN), DIMENSION(nlon,nlat) :: ifloating
      REAL(dp), INTENT(IN) :: tiso
      REAL(dp), INTENT(IN), DIMENSION(nlon,nlat) :: hi,hg,Hice,tair,tsea,&
                                                    geo,pice,Hocean
      REAL(dp), INTENT(IN), DIMENSION(nlon-1,nlatp) :: ub
      REAL(dp), INTENT(IN), DIMENSION(nlonp,nlat-1) :: vb

      REAL(dp), INTENT(INOUT), DIMENSION(nlon,nlat) :: bmelt,hikt,hgkt,&
                                                       tsole,Hicekt,Tbar
      REAL(dp), INTENT(INOUT), DIMENSION(nvertp,nvertp) :: coimp,coexp
      REAL(dp), INTENT(INOUT), DIMENSION(nvertp) :: rest,cocon,dtdt

      ! Local variables
      integer :: i,j,k,ij,jj,nestt,mm,imelt,ktempl,ipass,l,isol
      REAL(dp) :: thinice,Hiceprev,Hicecur,tmelt,tsurf,tbase,timeinc,dtdz,&
                  delz,wsurf,deltn,ress,tcheck,rhscal,hipijcur,hitijcur,&
                  hipijk,hitijk,dhhdt,dhidt,dhgdt,dhndt,dhsdt,dhedt,dhwdt,&
                  dhtndt,dhtsdt,dhtedt,dhtwdt,dhipijdt,dhitijdt,Hhi,Hlo,&
                  Hrt,Hlt,htrt,htlt,hthi,htlo,Hsij,delhx,delhhx,delhy,&
                  delhhy,htij,hipij,hitij,gradij,taurt,tault,tauhi,taulo,&
                     htslid,qgeo,ratemelt,kice,strntm,uadtm,vadtm,uadgrtm,&
                  vadgrtm,difftmc,uadco,vadco,grttmm,grttm,grttmp,wadtmm,&
                  wadtm,wadtmp,difftm,dkdttm,basaltm,timecur,z,dhtnedt,&
                  dhtnwdt,dhtsedt,dhtswdt,gradrt,gradlt,gradhi,gradlo,&
                  excess,cice

      REAL(dp), EXTERNAL :: tlapse

      thinice = 10.0_dp

!****************************************************************************
      ! Calculate interface temperatures for strain heating calculation.
      ! Strain heating is treated explicitly, so the previous temperature
      ! field "tice" is used for a one-time calculation of ttt, tpt fields

      if (itdcoup == 1) then
        call strain_int(tice,ttt,tpt,vtt,vpt,hi,Hice)
      endif

!****************************************************************************
      ! Loop over entire domain, solving the vertical temperature balance
      ! at each point (i,j).  Split up the sheet and stream thermal balances:
      ! these can be solved independently in each cell since there is no
      ! ice advection between modes.  There is a transfer of energy (rho cT)H
      ! da/dt, with this source/sink term averaged over the cell component.
      ! This has to be done on a vertical level basis: this is equivalent
      ! to assuming instantaneous adjustment of the heights and temperature
      ! profile of the incoming ice to the new component properties.  There is
      ! conservation but no detail: averaging is instantaneous.

      ! Work in co-latitudes, from low lat to high.
      do j=3,nlatp
        jj = j-1

        do i=3,nlonp
          ij = i-1

!************************************************
          ! Jump to next (i,j) point if no ice
          Hicecur = Hice(i,j)
          if (Hicecur < zerop) CYCLE

          Hiceprev = Hicekt(i,j)

          ! Download tice into column vector for optimal efficiency
          do k=1,nvertp
            tcolijk(k) = tice(k,ij,jj)
            tcolij(k) = c0
          enddo

          ! Specify temperature profile for thin ice
          if (Hicecur < thinice) then
            tsurf = MIN(tair(i,j), triplept)
            if (ifloating(i,j)==1) then
              tbase = triplept - fptdep*Hicecur
            else
              tbase = 265.0_dp
            endif
            dtdz = (tbase-tsurf)/Hicecur
            delz = Hicecur/DBLE(nvert)
            do k=1,nvertp
              tice(k,ij,jj) = tsurf + dtdz*delz*(k-1)
!             tice(k,ij,jj) = tnoice
              ticek(k,ij,jj) = tice(k,ij,jj)
            enddo
            CYCLE                    ! Hop to next longitude
          endif

          ! Determine time step from ice thickness, advection rate
          wsurf = wvelt(1,ij,jj)
          call caldeltn(wsurf,Hicecur,deltn)
          nestt = ANINT(deltt/deltn)          ! Number of inner loops

!***************************************************************************
          ! Set background dynamic variables for linear interpolation
          ! in the nested time loop; if delt=deltn, use current values

          call setnest(hi,hikt,Hice,Hicekt,hg,hgkt,nestt,dhndt,dhsdt,&
                       dhedt,dhwdt,dhidt,dhgdt,dhhdt,hipijcur,hitijcur,&
                       hipijk,hitijk,dhipijdt,dhitijdt,dhtndt,dhtsdt,&
                       dhtedt,dhtwdt,dhtnedt,dhtnwdt,dhtsedt,dhtswdt,&
                       i,j,ij,jj)

!************************************************************************
          ! Internal time loop
          ! Integrate from t^m to t^{m+1}

          do mm = 1,nestt
            timeinc = mm*deltn
            timecur = DBLE(time-ideltt) + timeinc

            ! Assign this moment's dynamic variables
            call varnest(hi,hikt,Hice,Hicekt,nestt,timeinc,Hsij,Hicecur,&
                         Hiceprev,hipijcur,hitijcur,hipijk,hitijk,dhhdt,&
                         dhndt,dhsdt,dhedt,dhwdt,dhipijdt,dhitijdt,dhidt,&
                         dhtndt,dhtsdt,dhtedt,dhtwdt,Hhi,Hlo,Hrt,Hlt,hipij,&
                         hitij,gradij,htij,htrt,htlt,hthi,htlo,dhtnedt,&
                         dhtnwdt,dhtsedt,dhtswdt,gradrt,gradlt,&
                         gradhi,gradlo,i,j,ij,jj)

!********************************************************************
            ! Calculate sliding friction for the basal boundary
            ! equation -k dt/dz = Q + \tau_j v_j.  Convert
            ! basal stresses to transformed co-ordinate system.

            if (islid == 1) then
              taurt = r2*(pice(i,j)+pice(i+1,j))*(hi(i+1,j)-hi(i,j))/delx(j)
              tault = r2*(pice(i,j)+pice(ij,j))*(hi(i,j)-hi(ij,j))/delx(j)
              tauhi = r2*(pice(i,j)+pice(i,j+1))*(hi(i,j)-hi(i,j+1))/dy
              taulo = r2*(pice(i,j)+pice(i,jj))*(hi(i,jj)-hi(i,j))/dy
              htslid = r2*(ABS(taurt*ub(i,jj)) + ABS(tault*ub(ij,jj)) &
                         + ABS(tauhi*vb(ij,j)) + ABS(taulo*vb(ij,jj)))
            else
              htslid = c0
            endif

            ! Basal heat input
            qgeo = geo(i,j)

!*************************************************************************
            ! Solve full thermodynamic balance.  Vertical loop
            ! at point (ij,jj): set up system A(T^{m+1})= B(T^m)+c
            ! Include lower boundary through ``imaginary point'' below the
            ! bed, with temperature dictated by basal boundary condition.
            ! This point is allocated to vector c and points
            ! nvertp-1, nvertp.

            ! Grid advection terms (vertically-constant)
            delhx = (htrt-htlt)/dx
            delhhx = (Hrt-Hlt)/dx
            delhy = (htlo-hthi)/dy
            delhhy = (Hlo-Hhi)/dy
            uadco = -c1/dx
            vadco = -c1/dy
            delz = Hsij/DBLE(nvert)

            ! Surface boundary condition: atmospheric temperature
            k = 1
            tsurf = tlapse(tsea(i,j),htij)     ! Key flag for ieis
!           tsurf = tair(i,j)
            cocon(k) = MIN(tsurf,triplept)
            coimp(k,k) = c1
            coexp(k,k) = c0
            bmelt(i,j) = c0
            ktempl = nvertp

             ! Basal condition: look for floating ice
            if (ifloating(i,j)==1) then
              imelt = 1
              tmelt = triplept - fptdep*Hsij
            endif

            ! Interior points and bottom boundary
            do k=2,nvertp

              ! Calculate energy balance terms for coefficient matrices
              ipass = 1
              call calct0(hi,tiso,tice,ticek,wvelt,vtt,vpt,ttt,tpt,tcolijk,&
                          bmelt,Hsij,delhx,delhhx,delhy,delhhy,uadco,vadco,&
                          gradij,dhidt,dhhdt,uadtm,vadtm,uadgrtm,vadgrtm,&
                          wadtmm,wadtm,wadtmp,grttmm,grttm,grttmp,strntm,&
                          difftm,difftmc,dkdttm,basaltm,kice,qgeo,htslid,&
                          imelt,ratemelt,tmelt,gradrt,gradlt,gradhi,gradlo,&
                          Hrt,Hlt,Hhi,Hlo,delz,cice,ktempl,i,j,k,ij,jj)

              if (imelt == 1) then          ! Cell is at pressure melting pt
                coimp(k,k) = c1
                coexp(k,k) = c0
                cocon(k) = tmelt
                rest(k) = c0

              else                          ! Non-temperate point

                ! Load coefficients
                if (k == nvertp) then       ! Lower bndy: bed flux specified
                  coimp(k,k) = c1 + fimp*deltn*c2*difftm
                  coimp(k,k-1) = fimp*deltn*(-c2)*difftm
                  coimp(k,k-2) = c0
                  coexp(k,k) = c1 + afimp*deltn*(-c2)*difftm
                  coexp(k,k-1) = afimp*deltn*c2*difftm
                  coexp(k,k-2) = c0
                  cocon(k) = deltn*(strntm + uadtm + vadtm  + difftmc  &
                                  + dkdttm + basaltm)
                else                        ! Interior point
                  coimp(k,k-1) = fimp*deltn*(-difftm - grttmm -wadtmm  &
                                             + uadgrtm + vadgrtm)
                  coimp(k,k) = c1 + fimp*deltn*(c2*difftm -grttm -wadtm)
                  coimp(k,k+1) = fimp*deltn*(-difftm - grttmp - wadtmp  &
                                             - uadgrtm - vadgrtm)
                  coexp(k,k-1) = afimp*deltn*(difftm + grttmm + wadtmm  &
                                             - uadgrtm - vadgrtm)
                  coexp(k,k) = c1 + afimp*deltn*(-c2*difftm +grttm +wadtm)
                  coexp(k,k+1) = afimp*deltn*(difftm + grttmp + wadtmp  &
                                             + uadgrtm + vadgrtm)
                  cocon(k) = deltn*(strntm + uadtm + vadtm + dkdttm)
                endif

              endif

            enddo            ! Close vertical do loop

!****************************************************************
            ! Solve system for tice update
!           if (ktempl == nvertp) then
!             isol = 3
!           else
              isol = 2
!           endif

            call solvet(tcolij,tcolijk,coimp,coexp,cocon,ij,jj,isol)

            ! Correct for ice at the pressure melting point
            do k=2,nvertp
              tmelt = triplept - fptdep*Hsij*xi(k)
              if (tcolij(k) > tmelt) then
                excess = (tcolij(k)-tmelt)*cice*delz/(deltn*tlat)   ! m/yr
                if (k == nvertp) excess = r2*excess                 ! half-cell
                bmelt(i,j) = bmelt(i,j) + excess
                tcolij(k) = tmelt
              endif
            enddo

            ! Store current as previous solution for next inner timestep
            if (nestt > 1) then
              tcolijk = tcolij
            endif

!****************************************************************
            ! Calculate residual of sheet column vector T_ij
            ! Recalculate energy fluxes for updated tfield:
            !                           rest(k)=rhs(k)-lhs(k)

            if (MOD(INT(timecur),10000) == 0) then

              ipass = 2
              do k=2,ktempl
                call calct0(hi,tiso,tice,ticek,wvelt,vtt,vpt,ttt,tpt,tcolijk,&
                            bmelt,Hsij,delhx,delhhx,delhy,delhhy,uadco,vadco,&
                            gradij,dhidt,dhhdt,uadtm,vadtm,uadgrtm,vadgrtm,&
                            wadtmm,wadtm,wadtmp,grttmm,grttm,grttmp,strntm,&
                            difftm,difftmc,dkdttm,basaltm,kice,qgeo,htslid,&
                            imelt,ratemelt,tmelt,gradrt,gradlt,gradhi,gradlo,&
                            Hrt,Hlt,Hhi,Hlo,delz,cice,ktempl,i,j,k,ij,jj)

                if (k == nvertp) then            ! Lower boundary
                  rhscal =  deltn*(strntm + uadtm +vadtm +dkdttm +difftmc  &
                             + c2*difftm*(-tcolij(k)+tcolij(k-1)) + basaltm)

                else
                  rhscal =  deltn*(strntm + uadtm + vadtm + dkdttm   &
                              + (wadtmm + grttmm + difftm      &
                                      - uadgrtm - vadgrtm)*tcolij(k-1) &
                              + (wadtm + grttm -2.*difftm)*tcolij(k) &
                              + (wadtmp + grttmp + difftm  &
                                      + uadgrtm + vadgrtm)*tcolij(k+1))
                endif
                dtdt(k) = tcolij(k)-tcolijk(k)
                rest(k) = rhscal-dtdt(k)

                ! Run-time checks
                if ((i == 25).and.(j == 18).and.(INT(timecur)==20000)) then
                  WRITE(45,*) 'Point (25,18)  k= ', k
                  WRITE(45,*) 'strain', strntm
                  WRITE(45,*) 'uad, vad ', uadtm, vadtm
                  WRITE(45,*) 'grttm ', grttm*tcolij(k), grttmm*tcolij(k-1)
                  WRITE(45,*) 'difftm ' , difftm
                  WRITE(45,*) 'wadtm ', wadtm
                  if (k /= nvertp) then
                    WRITE(45,*) 'basaltm ', basaltm
                    WRITE(45,*) 'diff ', c2*difftm*(-tcolij(k)+tcolij(k-1))
                    WRITE(45,*) 'difftmc ', difftmc
                  endif
                  WRITE(45,*)
                  WRITE(45,*) 'rest(k)', rest(k)
                endif

              enddo             ! End vertical do loop

              call calres(rest,ress,nvertp,nvertp)

            endif          !! End occasional residual check

!**********************************************************************
            ! Empty matrices from last time step
            coexp = c0
            coimp = c0

!********************************************************************
            ! Damage control
            tcheck = tcolij(nvertp-1)
            if ((tcheck > triplept).or.(tcheck < 200.0_dp)) then
              WRITE(45,*) 'Temp field out of control:  point ',i,j
              WRITE(45,*) (tcolij(k),k=1,nvertp)
              WRITE(45,*) 'Time ', time, 'residuals ',rest
              WRITE(45,*) 'Heat fluxes: strain,uadtm,vadtm,uadgrtm,&
                            &vadgrtm,difftm,dkdttm,grttm,grttmm,grttmp,&
                            &wadtm,wadtmm,wadtmp,basaltm'
              WRITE(45,*) strntm,uadtm,vadtm,uadgrtm,vadgrtm,difftm,dkdttm,&
                          grttm,grttmm,grttmp,wadtm,wadtmm,wadtmp,basaltm
              STOP
            endif

          enddo            !! End internal time loop

!********************************************************************
          ! Upload new tcolij into tfield array
          ! Store current as previous solution for next global timestep
          do k=1,nvertp
            tice(k,ij,jj) = tcolij(k)
          enddo

!**************************************************************************
        enddo         !! End latitude loop
      enddo    !! End longitude loop

!*************************************************************************
      ! Load new tice into ticek array; load tsole
      ticek = tice

      do j=1,nlat
        do i=1,nlon
          ij = i-1
          jj = j-1
          if (Hice(i,j) > zerop) then
            tsole(i,j) = tice(nvertp,ij,jj)
            ! Calculate average column temperature for the calving law
            Tbar(i,j) = c0
            do k=1,nvert
              Tbar(i,j) = Tbar(i,j) + r2*(tice(k,i,j)+tice(k+1,i,j))
            enddo
            Tbar(i,j) = Tbar(i,j)*delxi
          else
            tsole(i,j) = tair(i,j)
            Tbar(i,j) = tiso
          endif
        enddo
      enddo

      ! Store current thicknesses, areal distributions as previous t-loop
      ! values for nested integrations.  Preserved between calls.
      hgkt = hg
      hikt = hi
      Hicekt = Hice

!***************************************************************************
    end subroutine tfield
!***************************************************************************
