      program sf_alb_nc

!=======================================================================
!     creates snow free albedo data file sf_alb.nc
!=======================================================================

      implicit none

      integer, allocatable :: maskv(:,:)
      real, allocatable :: avar(:,:), var(:,:,:), lat_t(:,:)
      real, allocatable :: xt(:), yt(:)

      integer id, jd
      parameter (id=360, jd=180)
      real data(id,jd,2), xd(id), yd(jd)

      integer i, imt, iou, j, jmt, k, n, id_xt, id_yt, id_time
      real psi, theta, phi, rad, alat, daymon(12)
      
      logical exists
      
      character(120) :: path
      
      data daymon(1:6)  / 15.5,  45.,   74.5, 105.,  135.5, 166. /
      data daymon(7:12) /196.5, 227.5, 258.,  288.5, 319.,  349.5/

!=======================================================================
!     set path and read path file if it exists
!=======================================================================

      path = '/usr/local/models/UVic_ESCM/data_source'
      inquire (file='../path', exist=exists)
      if (exists) then
        open (10,file='../path')
        read (10,'(a)') path
      endif
      path = trim(path)//'/veg/'
      rad = acos(-1.)/180.

!=======================================================================
!     read grid data
!=======================================================================

      call openfile ("../grid.nc", iou)
      call getdimlen ('xt', iou, imt)
      call getdimlen ('yt', iou, jmt)
      allocate ( xt(imt) )
      allocate ( yt(jmt) )
      allocate ( var(imt,jmt,12) )
      allocate ( avar(imt,jmt) )
      allocate ( maskv(imt,jmt) )
      allocate ( lat_t(imt,jmt) )
      call getvara ('xt', iou, imt, (/1/), (/imt/), xt, 1., 0.)
      call getvara ('yt', iou, jmt, (/1/), (/jmt/), yt, 1., 0.)
      call getvars ('psi', iou, 1, psi, 1., 0.)
      call getvars ('theta', iou, 1, theta, 1., 0.)
      call getvars ('phi', iou, 1, phi, 1., 0.)
      call getvara ('lat_t', iou, imt*jmt, (/1,1/), (/imt,jmt/), lat_t
     &, 1., 0.)
      call closefile (iou)

!=======================================================================
!     read kmt data
!=======================================================================

      call openfile ("../kmt.nc", iou)
      call getvara ('kmt', iou, imt*jmt, (/1,1/), (/imt,jmt/)
     &, var(:,:,1), 1., 0.)
      maskv(:,:) = nint(var(:,:,1))
      call closefile (iou)

!=======================================================================
!     read data
!=======================================================================

      call openfile (trim(path)//'sf_albedo.nc', iou)
      call getvara ('X', iou, id, (/1/), (/id/), xd, 1., 0.)
      call getvara ('Y', iou, jd, (/1/), (/jd/), yd, 1., 0.)
!     flip latitude
      yd(1:jd) = yd(jd:1:-1)
      call closefile (iou)

      do n=1,12

!=======================================================================
!       average 2 years of monthly albedo
!=======================================================================

        call openfile (trim(path)//'sf_albedo.nc', iou)
        call getvara ('albedo', iou, id*jd, (/1,1,n/), (/id,jd,1/)
     &,   data(:,:,1), 1., 0.)
        call getvara ('albedo', iou, id*jd, (/1,1,n+12/), (/id,jd,1/)
     &,   data(:,:,2), 1., 0.)
!       flip data in latitude
        data(:,1:jd,:) = data(:,jd:1:-1,:)
        call closefile (iou)
        data(:,:,1) = (data(:,:,1) + data(:,:,2))/2.

        do j=1,jd
          do i=1,id
            if (data(i,j,1) .eq. 0.) data(i,j,1) = 2.e20
          enddo
        enddo

!=======================================================================
!       rotate and interpolate data
!=======================================================================

        call rot_intrp_sclr (data(:,:,1), xd, yd, id, jd, var(:,:,n)
     &,   xt, yt, imt, jmt, phi, theta, psi, -1.e20, 0)
        call extrap (var(:,:,n), -1.e10, xt, maskv, imt, jmt, 1)

!=======================================================================
!       set albedo over the ocean (depends on latitude but not time)
!=======================================================================

        do j=1,jmt
          do i=1,imt
            if (maskv(i,j) .gt. 0.) then
              alat = abs(lat_t(i,j))
              if (alat .le. 30.) then
                var(i,j,n) = 0.06
              elseif (alat .ge. 70.) then
                var(i,j,n) = 0.17
              else
                var(i,j,n) = (1. - cos(((alat - 30)*rad)*90./40.))*
     &                       (0.17 - 0.06) + 0.06
              endif
            endif
          enddo
        enddo

      enddo

      avar(:,:) = 0.
      print*, 'Warning: setting albedo to 0.17 above 85 N'
      do n=1,12
        where (lat_t(:,:) .ge. 85) var(:,:,n) = 0.17
        avar(:,:) = avar(:,:) + var(:,:,n)
      enddo
      avar(:,:) = avar(:,:)/12

!=======================================================================
!     set cyclic boundary condition
!=======================================================================

      var(1,:,:) = var(imt-1,:,:)
      var(imt,:,:) = var(2,:,:)

!=======================================================================
!     write monthly netcdf data
!=======================================================================

      call opennew ("../sf_alb_mth.nc", iou)
      call redef (iou)
      call putglobal (iou, 'sf_alb_mth.nc', ' ', ' ', ' ')
      call defdim ('time', iou, 0, id_time)
      call defdim ('xt', iou, imt, id_xt)
      call defdim ('yt', iou, jmt, id_yt)
      call defvar ('time', iou, 1, (/id_time/), 0., 0., 'T', 'D'
     &, 'time', 'time', 'common_year since 1-1-1 00:00:0.0')
      call defvar ('xt', iou, 1, (/id_xt/), 0., 0., 'X', 'F'
     &, 'longitude of the t grid', 'longitude', 'degrees_east')
      call defvar ('yt', iou, 1, (/id_yt/), 0., 0., 'Y', 'F'
     &, 'latitude of the t grid', 'latitude', 'degrees_north')
      call defvar ('sf_alb', iou, 3, (/id_xt,id_yt,id_time/)
     &, 0., 1., ' ', 'F', 'snow_free albedo'
     &, 'surface_albedo_assuming_no_snow', '1')
      call enddef (iou)
      call putvara ('xt', iou, imt, (/1/), (/imt/), xt, 1., 0.)
      call putvara ('yt', iou, jmt, (/1/), (/jmt/), yt, 1., 0.)
      do n=1,12
        call putvars ('time', iou, n, daymon(n)/365., 1., 0.)
        call putvara ('sf_alb', iou, imt*jmt, (/1,1,n/)
     &,              (/imt,jmt,1/), var(1,1,n), 1., 0.)
      enddo
      call closefile (iou)

!=======================================================================
!     write annual netcdf data
!=======================================================================

      call opennew ("../sf_alb_ann.nc", iou)
      call redef (iou)
      call putglobal (iou, 'sf_alb_ann.nc', ' ', ' ', ' ')
      call defdim ('xt', iou, imt, id_xt)
      call defdim ('yt', iou, jmt, id_yt)
      call defvar ('xt', iou, 1, (/id_xt/), 0., 0., 'X', 'F'
     &, 'longitude of the t grid', 'longitude', 'degrees_east')
      call defvar ('yt', iou, 1, (/id_yt/), 0., 0., 'Y', 'F'
     &, 'latitude of the t grid', 'latitude', 'degrees_north')
      call defvar ('sf_alb', iou, 2, (/id_xt,id_yt,id_time/)
     &, 0., 1., ' ', 'F', 'snow_free albedo'
     &, 'surface_albedo_assuming_no_snow', '1')
      call enddef (iou)
      call putvara ('xt', iou, imt, (/1/), (/imt/), xt, 1., 0.)
      call putvara ('yt', iou, jmt, (/1/), (/jmt/), yt, 1., 0.)
      call putvara ('sf_alb', iou, imt*jmt, (/1,1/)
     &,              (/imt,jmt/), avar, 1., 0.)
      call closefile (iou)

      end
