      program rot_nc

!     "Rotates" fields from one netcdf file into another.

!     Variables must be defined in both the input and output files (fi
!     and fo) for at least one time slice. Fields are "rotated" from
!     the input file to the output file so the original contents of the
!     output file are lost. If times between the files differ new time
!     slices will be created in the output file. Any variables not 
!     defined in both the input and output files are ignored. Both 
!     components of vector fields are required. The rotation is defined
!     with the variables psi, theta and phi. 

!     Land masking may also be set for all variables (see ms2d, mv2d,
!     ms3d and mv3d). Land masks may be read from a file (fmski, fmsko)
!     or if the masking variable (vmski, vmsko) is defined, taken from
!     the input or output file. The output land mask may also be 
!     interpolated from the input mask (see intrp_mask). If the input 
!     or output mask is undefined, no land masking is done.

      implicit none

      integer maxv, max3d
      parameter (maxv = 100, max3d = 4)
      character (120) :: s2d(maxv), v2d(maxv,2), s3d(maxv,max3d)
      character (120) :: v3d(maxv,2), fi, fo, fmski, fmsko, vmski, vmsko

      logical verbose, exists, masking, intrp_mask, inqvardef

      integer i, iou, ii, im, io, is, j, ji, jm, jo, js, k, ki(max3d)
      integer ko(max3d), l, m,  n, ntrec, ns2d, nv2d, ns3d, nv3d
      integer :: is2d(maxv), iv2d(maxv), is3d(maxv,max3d), iv3d(maxv)
      integer :: ms2d(maxv), mv2d(maxv), ms3d(maxv,max3d), mv3d(maxv)

      real :: fs2d(maxv), fv2d(maxv), fs3d(maxv,max3d), fv3d(maxv)
      real time, psi, theta, phi, valmask
      real, allocatable :: dsi(:,:), dvi(:,:,:), dso(:,:), dvo(:,:,:)
      real, allocatable :: xsi(:), ysi(:), xvi(:), yvi(:)
      real, allocatable :: xso(:), yso(:), xvo(:), yvo(:)
      real, allocatable :: xm(:), ym(:)
      integer, allocatable :: msi(:,:), mvi(:,:), mso(:,:), mvo(:,:)
      
      include 'netcdf.inc'

!     initialise some stuff

!     define rotation in radians
!     rotate into "rotated" model
!      phi   = -2.268928028
!      theta = -0.226892803
!      psi   = 3.141592654
!     rotate out of "rotated" model
      psi   = 2.268928028
      theta = 0.226892803
      phi   = -3.141592654

!     set verbose to true to see field names as they are processed
      verbose = .false.
!     set intrp_mask to true to interpolate the mask from the input file
      intrp_mask = .false.
!     set masking to false if masking is not required
      masking = .true.

!     set file names
      fi = "in.nc"
      fo = "out.nc"
      fmski = "maski.nc"
      fmsko = "masko.nc"

!     set initial 2d and 2d vector and scalar fields to blank
      s2d(:) = " "
      v2d(:,:) = " "
      s3d(:,:) = " "
      v3d(:,:) = " "
!     set default masking (0=no, 1=yes, mask land values)
      ms2d(:) = 0
      mv2d(:) = 0
      ms3d(:,1:2) = 1
      ms3d(:,2:max3d) = 0
      mv3d(:) = 1
!     set default interpolation (0=bilinear, 1=nearest, interpolation)
      is2d(:) = 0
      iv2d(:) = 0
      is3d(:,:) = 0
      iv3d(:) = 0
!     set default missing values (used in masking land values)
      fs2d(:) = nf_fill_float
      fv2d(:) = nf_fill_float
      fs3d(:,:) = nf_fill_float
      fv3d(:) = nf_fill_float
      vmski = "kmt"
      vmsko = "kmt"
      time = 0.
!     mask anything over valmask
      valmask = 1.e18

!     set the names of 2d and 3d scalars and vectors to be rotated.
!     these may be commented out if not desired. if not found in the 
!     input file they are ignored

      s2d(1)  = "kmt"
        is2d(1) = 1
      s2d(2)  = "mskhr"
        is2d(2) = 1
        ms2d(2) = 1
        fs2d(2) = 0
      s2d(3)  = "tlat"
      s2d(4)  = "tlon"
      s2d(5)  = "ulat"
      s2d(6)  = "ulon"
      s2d(7)  = "flux_heat"
        ms2d(7) = 1
      s2d(8)  = "flux_salt"
        ms2d(8) = 1
      s2d(9)  = "ps"
        ms2d(9) = 1
      s2d(10)  = "psi"
        ms2d(10) = 1
      s2d(11) = "elev"  
      s2d(12) = "sat"
      s2d(13) = "shum"    
      s2d(14) = "slat"
      s2d(15) = "precip"
      s2d(16) = "evap"
      s2d(17) = "outlwr"
      s2d(18) = "uplwr"
      s2d(19) = "upsens"
      s2d(20) = "dnswr"
      s2d(21) = "upltnt"
      s2d(22) = "p_alb"
      s2d(23) = "a_alb"
      s2d(24) = "s_alb"
      s2d(25) = "tice"
      s2d(26) = "hice"
      s2d(27) = "aice"
      s2d(28) = "hsno"
      s2d(29) = "soilm"
      s2d(30) = "runoff"
      s2d(31) = "surf"
      s2d(32) = "flxadj_t"
      s2d(33) = "flxadj_s"
      s2d(34) = "psno"
      s2d(35) = "solins"
      s2d(36) = "ws"
      s2d(37) = "tmsk"
      s2d(38) = "aicel"
      s2d(39) = "hicel"
      s2d(40) = "tmsk"
      s2d(41) = "mskhr"
      s2d(42) = "lat_t"      
      s2d(43) = "lon_t"      
      s2d(44) = "lat_u"      
      s2d(45) = "lon_u"      
      
      s2d(46) = "TS1"
      s2d(47) = "TSTAR_GB"
      s2d(48) = "ALBLAND"
      s2d(49) = "ET"
      s2d(50) = "M"
      s2d(51) = "CS"
      s2d(52) = "RESP_S"

      v2d(1,1:2) = (/"taux","tauy"/)
        mv2d(1) = 1
      v2d(2,1:2) = (/"wx","wy"/)
      v2d(3,1:2) = (/"awx","awy"/)
      v2d(4,1:2) = (/"uice","vice"/)
        mv2d(4) = 1
      v2d(5,1:2) = (/"xint","yint"/)
        mv2d(5) = 1

!     3d in ocean depth
      s3d(1,1) = "temp"
      s3d(2,1) = "salinity"
      s3d(3,1) = "w"
      s3d(4,1) = "adv_vbtiso"

!     3d in ice category
      s3d(1,2) = "ticen"
      s3d(2,2) = "hicen"
      s3d(3,2) = "aicen"
      s3d(4,2) = "hsnon"

!     3d in plant funcional type
      s3d(1,3) = "GPP"
      s3d(2,3) = "NPP"
      s3d(3,3) = "HT"
      s3d(4,3) = "LAI"
      s3d(5,3) = "C_VEG"

!     3d in land type
      s3d(4,4) = "FRAC"

      v3d(1,1:2) = (/"u","v"/)
      v3d(2,1:2) = (/"adv_vetiso","adv_vntiso"/)

!     get input grid and allocate arrays
      inquire (file=trim(fo), exist=exists)
      if (.not. exists) then
        print*, 'can not open input file: ',fi
        stop
      endif
      call openfile (fi, iou)
      ntrec = 1
      ii = 1
      ji = 1
      ki(:) = 1
      if (inqvardef ('time', iou)) call getdimlen ('time', iou, ntrec)
      if (inqvardef ('xu', iou)) call getdimlen ('xu', iou, ii)
      if (inqvardef ('yu', iou)) call getdimlen ('yu', iou, ji)
      if (inqvardef ('zw', iou)) call getdimlen ('zw', iou, ki(1))
      if (inqvardef ('xt', iou)) call getdimlen ('xt', iou, ii)
      if (inqvardef ('yt', iou)) call getdimlen ('yt', iou, ji)
      if (inqvardef ('zt', iou)) call getdimlen ('zt', iou, ki(1))
      if (inqvardef ('cat', iou)) call getdimlen ('cat', iou, ki(2))
      if (inqvardef ('pft', iou)) call getdimlen ('pft', iou, ki(3))
      if (inqvardef ('type', iou)) call getdimlen ('type', iou, ki(4))
      allocate ( xsi(ii) )
      allocate ( ysi(ji) )
      allocate ( dsi(ii,ji) )
      allocate ( msi(ii,ji) )
      allocate ( xvi(ii) )
      allocate ( yvi(ji) )
      allocate ( dvi(ii,ji,2) )
      allocate ( mvi(ii,ji) )
      if (inqvardef ('xt', iou))
     &  call getvara ('xt', iou, ii, (/1/), (/ii/), xsi, 1., 0.)
      if (inqvardef ('yt', iou))
     &  call getvara ('yt', iou, ji, (/1/), (/ji/), ysi, 1., 0.)
      if (inqvardef ('xu', iou))
     &  call getvara ('xu', iou, ii, (/1/), (/ii/), xvi, 1., 0.)
      if (inqvardef ('yu', iou))
     &  call getvara ('yu', iou, ji, (/1/), (/ji/), yvi, 1., 0.)
      call closefile (iou)

!     get output grid and allocate arrays
      inquire (file=trim(fo), exist=exists)
      if (.not. exists) then
        print*, 'can not open ouput file: ',fo
        stop
      endif
      call openfile (fo, iou)
      io = 1
      jo = 1
      ko = 1
      if (inqvardef ('xu', iou)) call getdimlen ('xu', iou, io)
      if (inqvardef ('yu', iou)) call getdimlen ('yu', iou, jo)
      if (inqvardef ('zw', iou)) call getdimlen ('zw', iou, ko)
      if (inqvardef ('xt', iou)) call getdimlen ('xt', iou, io)
      if (inqvardef ('yt', iou)) call getdimlen ('yt', iou, jo)
      if (inqvardef ('zt', iou)) call getdimlen ('zt', iou, ko)
      if (inqvardef ('zt', iou)) call getdimlen ('zt', iou, ko(1))
      if (inqvardef ('cat', iou)) call getdimlen ('cat', iou, ko(2))
      if (inqvardef ('pft', iou)) call getdimlen ('pft', iou, ko(3))
      if (inqvardef ('type', iou)) call getdimlen ('type', iou, ko(4))
      allocate ( xso(io) )
      allocate ( yso(jo) )
      allocate ( dso(io,jo) )
      allocate ( mso(io,jo) )
      allocate ( xvo(io) )
      allocate ( yvo(jo) )
      allocate ( dvo(io,jo,2) )
      allocate ( mvo(io,jo) )
      if (inqvardef ('xt', iou))
     &  call getvara ('xt', iou, io, (/1/), (/io/), xso, 1., 0.)
      if (inqvardef ('yt', iou))
     &  call getvara ('yt', iou, jo, (/1/), (/jo/), yso, 1., 0.)
      if (inqvardef ('xu', iou))
     &  call getvara ('xu', iou, io, (/1/), (/io/), xvo, 1., 0.)
      if (inqvardef ('yu', iou))
     &  call getvara ('yu', iou, jo, (/1/), (/jo/), yvo, 1., 0.)
      call closefile (iou)

      if (masking) then
!       get input ocean mask from the mask or input file
        inquire (file=trim(fmski), exist=exists)
        if (exists) then
!         get input ocean mask from a mask file
          call openfile (fmski, iou)
        else
!         get input ocean mask from input file
          call openfile (fo, iou)
        endif        
        if (inqvardef (vmski, iou)) then
          im = 0
          jm = 0
          if (inqvardef ('xt', iou)) call getdimlen ('xt', iou, im)
          if (inqvardef ('yt', iou)) call getdimlen ('yt', iou, jm)
          if (im .gt. 0 .and. jm .gt. 0) then
            allocate ( xm(im) )
            allocate ( ym(jm) )
            call getvara ('xt', iou, im, (/1/), (/im/), xm, 1., 0.)
            call getvara ('yt', iou, jm, (/1/), (/jm/), ym, 1., 0.)
            is = 0
            do i=1,min(im, ii)
              if (abs(xm(i) - xsi(1)) .lt. 1.e-3) is = i
            enddo
            deallocate ( xm )
            js = 0
            do j=1,min(jm, ji)
              if (abs(ym(j) - ysi(1)) .lt. 1.e-3) js = j
            enddo
            deallocate ( ym )
            if (is .gt. 0 .and. js .gt. 0) then
              call getvara (vmski, iou, ii*ji, (/is,js/), (/ii,ji/)
     &,         dsi, 1., 0.)
            else
              masking = .false.
            endif 
          else
            masking = .false.
          endif
        else
          masking = .false.
        endif
        call closefile (iou)
      endif

      if (masking) then
!       get output ocean mask from the mask or output file
!       or interpolate it from input file mask
        inquire (file=trim(fmsko), exist=exists)
        if (exists .or. .not. intrp_mask) then
          if (exists) then
!           get output ocean mask from a mask file
            call openfile (fmsko, iou)
          else
!           get output ocean mask from output file
            call openfile (fo, iou)
          endif
          if (inqvardef (vmsko, iou)) then
            im = 0
            jm = 0
            if (inqvardef ('xt', iou)) call getdimlen ('xt', iou, im)
            if (inqvardef ('yt', iou)) call getdimlen ('yt', iou, jm)
            if (im .gt. 0 .and. jm .gt. 0) then
              allocate ( xm(im) )
              allocate ( ym(jm) )
              call getvara ('xt', iou, im, (/1/), (/im/), xm, 1., 0.)
              call getvara ('yt', iou, jm, (/1/), (/jm/), ym, 1., 0.)
              is = 0
              do i=1,min(im, io)
                if (abs(xm(i) - xso(1)) .lt. 1.e-3) is = i
              enddo
              deallocate ( xm )
              js = 0
              do j=1,min(jm, jo)
                if (abs(ym(j) - yso(1)) .lt. 1.e-3) js = j
              enddo
              deallocate ( ym )
              if (is .gt. 0 .and. js .gt. 0) then
                call getvara (vmsko, iou, io*jo, (/is,js/), (/io,jo/)
     &,           dso, 1., 0.)
              else
                masking = .false.
              endif 
            else
              masking = .false.
            endif
          else
            masking = .false.
          endif
          call closefile (iou)
        else
          call rot_intrp_sclr (dsi, xsi, ysi, ii, ji, dso, xso
     &,     yso, io, jo, phi, theta, psi, -abs(nf_fill_float), 1)
        endif
      endif

      if (masking) then
!       write mask to the output file
        call openfile (fo, iou)
        if (inqvardef (vmski, iou)) then
          call putvara (vmski, iou, io*jo, (/1,1/), (/io,jo/), dso
     &,     1., 0.)
        endif
        call closefile (iou)
!       calculate input velocity mask
        msi = dsi
        do i=1,ii-1
          do j=1,ji-1
            mvi(i,j) = min(msi(i,j),msi(i+1,j),msi(i,j+1),msi(i+1,j+1))
          enddo
        enddo
        mvi(ii,:) = mvi(2,:)
        mvi(:,ji) = 0 
!       calculate a output velocity mask
        mso = dso
        do i=1,io-1
          do j=1,jo-1
            mvo(i,j) = min(mso(i,j),mso(i+1,j),mso(i,j+1),mso(i+1,j+1))
          enddo
        enddo
        mvo(io,:) = mvo(2,:)
        mvo(:,jo) = 0
      else
!       if no masking set all masks to 1
        msi(:,:) = 1
        mvi(:,:) = 1
        mso(:,:) = 1
        mvo(:,:) = 1
      endif

!     loop through all time records
      do m=1,ntrec
        call openfile (fo, iou)
        if (inqvardef ('time', iou)) then
          call getvars ('time', iou, m, time, 1., 0.)    
          call closefile (iou)
          call openfile (fo, iou)
          call putvars ('time', iou, m, time, 1., 0.)    
        endif
        call closefile (iou)
        if (verbose) print*, 'time record: ', m, ' time: ', time

!       rotate and interpolate 2d scalar data
        do n=1,maxv
          call openfile (fi, iou)
          if (.not. inqvardef (s2d(n), iou)) s2d(n) = " "
          call closefile (iou)
          if (s2d(n) .ne. " " .and. trim(s2d(n)) .ne. vmski) then
            if (verbose) print*, '2d scalar: ',trim(s2d(n))
            call openfile (fi, iou)
            call getvara (s2d(n), iou, ii*ji, (/1,1,m/)          
     &,       (/ii,ji,1/), dsi, 1., 0.)
            call closefile (iou)

            if (ms2d(n) .eq. 1) then 
              call set_land (dsi, fs2d(n), msi, ii, ji, 1)
            endif
            call rot_intrp_sclr (dsi, xsi, ysi, ii, ji, dso, xso
     &,       yso, io, jo, phi, theta, psi, -abs(fs2d(n)), is2d(n))
            if (ms2d(n) .eq. 1) then
              call extrap2 (dso(:,:), fs2d(n), xso, io, jo)
              call set_land (dso, fs2d(n), mso, io, jo, 1)
            endif

            call openfile (fo, iou)
            if (inqvardef (s2d(n), iou)) then
              where (dso(:,:) .gt. valmask) dso(:,:) = fs2d(n)
              call putvara (s2d(n), iou, io*jo, (/1,1,m/)          
     &,         (/io,jo,1/), dso, 1., 0.)
            endif
            call closefile (iou)
          endif
        enddo

!       rotate and interpolate 2d vector data
        do n=1,maxv
          call openfile (fi, iou)
          if (.not. inqvardef (v2d(n,1), iou)) v2d(n,1) = " "
          call closefile (iou)
          if (v2d(n,1) .ne. " " .and. trim(v2d(n,1)) .ne. vmski) then
            if (verbose) print*, '2d vector: ',trim(v2d(n,1)), ' '
     &,       trim(v2d(n,2))
            call openfile (fi, iou)
            call getvara (v2d(n,1), iou, ii*ji, (/1,1,m/)          
     &,       (/ii,ji,1/), dvi(1,1,1), 1., 0.)    
            call getvara (v2d(n,2), iou, ii*ji, (/1,1,m/)          
     &,       (/ii,ji,1/), dvi(1,1,2), 1., 0.)    
            call closefile (iou)

            if (mv2d(n) .eq. 1) then 
              call set_land (dvi(:,:,1), fv2d(n), mvi, ii, ji, 1)
              call set_land (dvi(:,:,2), fv2d(n), mvi, ii, ji, 1)
            endif
            call rot_intrp_vctr (dvi, xvi, yvi, ii, ji, dvo, xvo
     &,       yvo, io, jo, phi, theta, psi, -abs(fv2d(n)), iv2d(n))
            if (mv2d(n) .eq. 1) then 
              call extrap2 (dvo(:,:,1), fv2d(n), xvo, io, jo)
              call extrap2 (dvo(:,:,2), fv2d(n), xvo, io, jo)
              call set_land (dvo(:,:,1), fv2d(n), mvo, io, jo, 1)
              call set_land (dvo(:,:,2), fv2d(n), mvo, io, jo, 1)
            endif

            call openfile (fo, iou)
            if (inqvardef (v2d(n,1), iou)) then
              where (dvo(:,:,1) .gt. valmask) dvo(:,:,1) = fv2d(n)
              call putvara (v2d(n,1), iou, io*jo, (/1,1,m/)          
     &,         (/io,jo,1/), dvo(1,1,1), 1., 0.)
            endif   
            if (inqvardef (v2d(n,2), iou)) then
              where (dvo(:,:,2) .gt. valmask) dvo(:,:,2) = fv2d(n)
              call putvara (v2d(n,2), iou, io*jo, (/1,1,m/)          
     &,         (/io,jo,1/), dvo(1,1,2), 1., 0.)    
            endif   
            call closefile (iou)
          endif
        enddo

        do l=1,max3d

          do k=1,ki(l)
          if (verbose) print*, l, 'level: ',k

!           rotate and interpolate 3d scalar data
            do n=1,maxv
              call openfile (fi, iou)
              if (.not. inqvardef (s3d(n,l), iou)) s3d(n,l) = " "
              call closefile (iou)
              if (s3d(n,l).ne." " .and.  trim(s3d(n,l)).ne.vmski) then
                if (verbose) print*, '3d scalar: ',trim(s3d(n,l))
                call openfile (fi, iou)
                call getvara (s3d(n,l), iou, ii*ji, (/1,1,k,m/)          
     &,          (/ii,ji,1,1/), dsi, 1., 0.)    
                call closefile (iou)

                if (ms3d(n,l) .eq. 1) then 
                  call set_land (dsi, fs3d(n,l), msi, ii, ji, k)
                endif
                call rot_intrp_sclr (dsi, xsi, ysi, ii, ji, dso, xso
     &,           yso, io, jo, phi, theta, psi, -abs(fs3d(n,l))
     &,           is3d(n,l))
                if (ms3d(n,l) .eq. 1) then 
                  call extrap2 (dso(:,:), fs3d(n,l), xso, io, jo)
                  call set_land (dso, fs3d(n,l), mso, io, jo, k)
                endif

                call openfile (fo, iou)
                if (inqvardef (s3d(n,l), iou)) then
                  where (dso(:,:) .gt. valmask) dso(:,:) = fs3d(n,l)
                  call putvara (s3d(n,l), iou, io*jo, (/1,1,k,m/)          
     &,             (/io,jo,1,1/), dso, 1., 0.)    
                endif
                call closefile (iou)
              endif
            enddo

          enddo
          
        enddo

        do k=1,ki(1)

!         rotate and interpolate 3d vector data
          do n=1,maxv
            call openfile (fi, iou)
            if (.not. inqvardef (v3d(n,1), iou)) v3d(n,1) = " "
            call closefile (iou)
            if (v3d(n,1) .ne. " " .and. trim(v3d(n,1)) .ne. vmski) then
              if (verbose) print*, '3d vector: ',trim(v3d(n,1)), ' '
     &,         trim(v3d(n,2))
              call openfile (fi, iou)
              if (inqvardef (v3d(n,1), iou)) then
                call getvara (v3d(n,1), iou, ii*ji, (/1,1,k,m/)          
     &,           (/ii,ji,1,1/), dvi(1,1,1), 1., 0.)
              endif
              if (inqvardef (v3d(n,2), iou)) then
                call getvara (v3d(n,2), iou, ii*ji, (/1,1,k,m/)          
     &,           (/ii,ji,1,1/), dvi(1,1,2), 1., 0.)    
              endif
              call closefile (iou)

              if (mv3d(n) .eq. 1) then 
                call set_land (dvi(:,:,1), fv3d(n), mvi, ii, ji, k)
                call set_land (dvi(:,:,2), fv3d(n), mvi, ii, ji, k)
              endif
              call rot_intrp_vctr (dvi, xvi, yvi, ii, ji, dvo, xvo
     &,         yvo, io, jo, phi, theta, psi, -abs(fv3d(n)), iv3d(n))
              if (mv3d(n) .eq. 1) then 
                call extrap2 (dvo(:,:,1), fv3d(n), xvo, io, jo)
                call extrap2 (dvo(:,:,2), fv3d(n), xvo, io, jo)
                call set_land (dvo(:,:,1), fv3d(n), mvo, io, jo, k)
                call set_land (dvo(:,:,2), fv3d(n), mvo, io, jo, k)
              endif

              call openfile (fo, iou)
              if (inqvardef (v3d(n,1), iou)) then
                where (dvo(:,:,1) .gt. valmask) dvo(:,:,1) = fv2d(n)
                call putvara (v3d(n,1), iou, io*jo, (/1,1,k,m/)          
     &,           (/io,jo,1,1/), dvo(1,1,1), 1., 0.)    
              endif
              if (inqvardef (v3d(n,2), iou)) then
                where (dvo(:,:,2) .gt. valmask) dvo(:,:,2) = fv2d(n)
                call putvara (v3d(n,2), iou, io*jo, (/1,1,k,m/)          
     &,           (/io,jo,1,1/), dvo(1,1,2), 1., 0.)    
              endif
              call closefile (iou)              
            endif
          enddo

        enddo

      enddo

      end
