!***************************************************************************
!   Solves system Jx=b using Gaussian elimination of the
!   lower diagonal from top down, then back-substitution from the
!   bottom up.  Returns solution xvect(njac).  nsys is the size
!   of the system to be solved.

!   Flag isol identifies which problem is being solved:
!         isol = 1      Latitudinal ADI solver: l=2,nlat-1
!         isol = 2      Longitudinal ADI solver: l=2,nlon-1
!         isol = 3      Temperature solver **off tri-diag on bdy**
!         isol = 4      Bed TD solver **off tri-diag on bdy**
!   Flag isys identifies size of system (for isol=1,2)
!         isys = 1      Only sheet flow (nsys=nlatp)
!         isys = 2      Sheet + stream flow (nsys=2*nlatp)
!         isys = 3      Sheet + stream + shelf flow (nsys=3*nlatp)

    subroutine calsol(jac,bvect,xvect,njac,nsys,isol)

!***************************************************************************
      USE global_param
      implicit none
      INTEGER, INTENT(IN) :: njac,nsys,isol
      REAL(KIND=dp), INTENT(INOUT) :: jac(njac,njac)
      REAL(KIND=dp), INTENT(OUT) :: bvect(njac),xvect(njac)

      ! Local variables
      integer :: i,j
      REAL(KIND=dp) :: fact

      ! Case isol=1,2: tridiagonal

      if ((isol == 1).or.(isol == 2)) then
        ! Row-reduce jac to upper triangular, modify bvect(i) accordingly
        do i=2,nsys
          fact = jac(i,i-1)/jac(i-1,i-1)
          do j=i-1,i
            jac(i,j) = jac(i,j) - fact*jac(i-1,j)
          enddo
          bvect(i) = bvect(i) - fact*bvect(i-1)
        enddo

        ! Solve from bottom up
        xvect(nsys) = bvect(nsys)/jac(nsys,nsys)
        do i=nsys-1,1,-1
           xvect(i) = (bvect(i)-jac(i,i+1)*xvect(i+1))/jac(i,i)
        enddo

      ! Case isol=3: lower boundary gives three elements in upper row

      elseif (isol == 3) then
        ! Row-reduce jac to upper triangular, modify bvect(i) accordingly
        i = nsys
        fact = jac(i,i-2)/jac(i-1,i-2)      ! Lower 3-pt boundary
        do j=i-2,i
          jac(i,j) = jac(i,j) - fact*jac(i-1,j)
        enddo
        bvect(i) = bvect(i) - fact*bvect(i-1)

        do i=2,nsys                        ! Standard reduction, top down
          fact = jac(i,i-1)/jac(i-1,i-1)
          do j=i-1,i
            jac(i,j) = jac(i,j) - fact*jac(i-1,j)
          enddo
          bvect(i) = bvect(i) - fact*bvect(i-1)
        enddo

        ! Solve from bottom up
        xvect(nsys) = bvect(nsys)/jac(nsys,nsys)
        do i=nsys-1,1,-1
           xvect(i) = (bvect(i)-jac(i,i+1)*xvect(i+1))/jac(i,i)
        enddo

      endif

!***************************************************************************
    end subroutine calsol
!***************************************************************************
