!=======================================================================
!                      UBC Basic Ice Sheet Model

!     Program bism models ice flow over a 2D vertically-integrated ice
!     sheet, using the biconjugate gradient method on a spherical grid.
!     Thermomechanical coupling is optional. When invoked, the vertical
!     dimension is added in the dynamic solution and 3D velocity fields
!     are calculated.  Thermodynamics are solved after Jenssen (1977) in
!     the style of Huybrechts (1988) and followers. Dynamics and
!     thermodynamics are solved on vertically-transformed co-ordinates
!     with linear interpolation between grids.

!     This is a stream-lined, pure ice-sheet version of the general
!     mixture model which includes ice-stream dynamics.

!     MODULE global_vars declares global arrays and variables.
!     MODULE global_param defines global model and physical constants.
!     All code is implicit none; though I usually use (a-h,o-z)
!     explicitly for double precision variables and (i-n) explicitly for
!     integers.
!     External to MAIN bism, all variables are locally defined.
!     Global variables are passed as arguments in the subroutine calls,
!     including explicit array dimensions. Global constants are passed
!     through MODULE global_param.

!     based on code by: S. Marshall, M. Yoshimori, A. Schmittner
!=======================================================================

#if defined uvic_embm
      subroutine cidm(e_timein,e_timeout,avg_Hice,avg_hi,avg_calv,&
                      avg_aice)
#else
      subroutine cidm(e_timein,e_timeout)
#endif

#if defined ubc_cidm
!-----------------------------------------------------------------------
!     declare constants and variables
!-----------------------------------------------------------------------

! ... global
      USE global_vars
      USE subgrid_arrays
# if !defined uvic_embm
      USE deg_day_vars
# endif
      USE hydrol_arrays
      USE geol_arrays
      USE icetd_arrays
      USE bedtd_arrays
      USE shelf_arrays
# if defined uvic_embm
      USE cidm_mod_embm
# endif
      implicit none

# if defined uvic_embm
! ... subroutine arguments
      REAL(KIND=dp),INTENT(IN) :: e_timein,e_timeout
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:iimt,1:jjmt) :: avg_Hice,&
                                            avg_hi,avg_calv,avg_aice

! ... local
      REAL(KIND=dp),DIMENSION(1:nlon,1:nlat) :: fwater,sum_Hice,sum_hi,&
                                                   bal,sum_calv,sum_aice
      REAL(KIND=dp),DIMENSION(1) :: vol1, vol2, bal2, calv2
# endif
      integer :: i,j,ndcur,iprint,isloop,ieloop,icloop,nloop
      LOGICAL :: ltdcur,ldiverge
      REAL(KIND=dp) :: res

!-----------------------------------------------------------------------
!     set integration time and number of loops
!-----------------------------------------------------------------------

      stime = e_timein
      etime = e_timeout
      isloop = INT(stime/delt)
      ieloop = INT(etime/delt)
      nloop = ieloop-isloop

# if defined uvic_embm
!-----------------------------------------------------------------------
!     initialize SBC arrays for other components of the climate system
!     model
!-----------------------------------------------------------------------

      sum_Hice = c0
      sum_hi   = c0
      sum_calv = c0

# endif
      vol1 = c0 ! global ice volume before update
      do j=1,nlat
        do i=2,nlon-1
          vol1 = vol1 + Hice(i,j)*area(j)
        enddo
      enddo
!-----------------------------------------------------------------------
!     S T A R T   CIDM   M A I N   L O O P
!-----------------------------------------------------------------------

      ctime = stime
      do icloop = isloop+1,ieloop
        ctime = ctime + delt

!-----------------------------------------------------------------------
!       isostatic adjustment hg(t)
!-----------------------------------------------------------------------

        if (irebound == 1) then
          call cidm_isos(hg0,hg,hi,Hice,Hicekt,dhgdt,hgk,hgkt,hik,hikt,&
                    load,vert,loadscale,alinear,aquad,bquad,curvature,&
                    time_maxwell,hg_maxwell,iglacial,ifloating,iland)
        endif

# if defined uvic_embm
!-----------------------------------------------------------------------
!       surface mass balance, balance (m yr^{-1} i.e.)
!       (accumuration by snowfall and ablation by surface melting)
!-----------------------------------------------------------------------

        call cidm_mass_acc
        call cidm_mass_smelt
        call cidm_mass_balance
        bal = balance
# else
!-----------------------------------------------------------------------
!       calculate glacial index, based on GRIP del18 record
!-----------------------------------------------------------------------

        if ((iclim >= 6).and.(iclim <= 10)) then
          if ((MOD(time,100)==0).or.(time==(stime+idelt))) then
            call glacindex(glacial,tsea,precip,hi,hi0,hg,iglacial,&
                           delsea,sealevel,time)
          endif
        endif

!-----------------------------------------------------------------------
!       monthly albedo
!-----------------------------------------------------------------------

        if (ialbedo == 1) then                ! SM albedo code
          call surfalb(albedo,bckgrnd,soot,snowfrac,snowdep,snowage,&
                        snowrad,coszrs,hi)
        endif

        ! Calculate current climate variables
        ! Surface temperature
        call airtemp(tsea,tseaC,hi,hg,hi0,tair,Hice,delO,time)

        ! Surface mass balance
        call massbal(acc,smelt,balance,Hice,hi,hg,slope,slope0,tair,&
                     tsea,tseaC,precip,rainfall,pddnet,annamp,delO,&
                     delsea,hi0,hg0,sealevel,iland,iglacial,icelevel,&
                     ideep,ifloating,igreen,landfrac,time)

        ! Treat mass loss from calving if enabled
        if (ishelf >= 2) then
          call calving(calve,Hocean,Hice,Tbar,hg,landfrac,sealevel,iland)
          balance = balance - calve                ! Adjust mass balance
        endif

        ! Write initial mass balance data
        if (time==(stime+idelt)) then
          call writmass(balance,acc,smelt,bmelt,pddnet,annamp,tsea,tair,&
                        precip,time)
        endif
# endif

!-----------------------------------------------------------------------
!       flag ltdcur tracks time for asynchronous T-D runs
!-----------------------------------------------------------------------

        if ((itherm == 1).and.(MOD((ctime-stime),deltt) == c0)) then
          ltdcur = .true.
        else
          ltdcur = .false.
        endif

!-----------------------------------------------------------------------
!       solve dynamic equations
!-----------------------------------------------------------------------

        if (itdcoup == 1) then   ! thermodynamics option
          ! Load interface temperatures for flux integration
          call interface_temps(ticev,tt,tp,hi,Hice)
        endif

        ! Conjugate gradient 2D solution
        call is_dyn (hg,hgk,hi,hik,dhgdt,Hice,Hicek,wvel,vt,vp,tiso,&
                      ub,vb,ubk,vbk,iub,ivb,Bbar,Tbar,Hocean,balance,&
                      bmelt,acc,smelt,calve,dhidt,fpshtk,ftshtk,fpsht,&
                      ftsht,pice,icover,iland,ifloating,&
                      ltdcur,ldiverge,res,itime,fwater)

# if !defined uvic_embm
        if ((isubtop == 2).or.(isubtop == 3)) then
          ! Subgrid ice nucleation and decay
          call classify_glacial_cells(hi,hg,Hice,Hicek,sealevel,&
                                      ifloating,iglacial,icelevel,igreen)
        endif

!-----------------------------------------------------------------------
!       solve surface hydrology evolution at intervals hydnest
!-----------------------------------------------------------------------

        if ((igingembre > 0).and.(MOD((time-stime),ihydnest) == 0)) then

          if (igingembre == 1) then
            call surf_hydrol(hi,hg,hg0,Hice,smelt,rainfall,iland,&
                             ifloating,igreen,time)

          elseif (igingembre == 2) then
            call shydrol_newt(hi,hg,hg0,Hice,smelt,rainfall,iland,&
                              ifloating,igreen,time)
          endif
        endif

!-----------------------------------------------------------------------
!       particle tracking
!-----------------------------------------------------------------------

        if ((itrack == 1).and.(MOD((time-stime),ideltg) == 0)) then

          ! Particle tracking, spawn subglacial ducks
          call spawn(ndcur,time)
          call track(ndcur,vt,vp,hi,hg,Hice,time)

          ! Erode the bed
          call striae(vt,vp,wvel,tsole,hg,Hice,time)

        endif

        if ((isilt==1).and.(MOD(time,ideltg)==0)) then
          call debris(vt,vp,hi,hg,tsole,bmelt,time)

        endif

# endif
!-----------------------------------------------------------------------
!       solve thermal balance at current time step
!-----------------------------------------------------------------------

        if (ltdcur) then

          ! Transform velocity field to temperature grid
          call bktrans(vt,vp,wvel,vtt,vpt,wvelt,hi,Hice,itime,iland)

          ! Solve temperature equation
          call tfield(tair,ub,vb,hi,hikt,hg,hgkt,Hice,Hicekt,&
                      pice,geo,rest,dtdt,itime,coimp,coexp,cocon,&
                      tiso,tsea,tsole,Tbar,bmelt,Hocean,ifloating)

# if !defined uvic_embm
          ! Write temperature/velocity fields
          iprint = 10000
          if ((MOD((time-stime),iprint)==0).or.(time==etime)) then
            call writt(time)
          endif

# endif
          ! Transform temperature field to dynamical grid
          call fdtrans(tice,ticev,hi,Hice)

# if !defined uvic_embm
!-----------------------------------------------------------------------
!         solve bed thermal balance at current time step
!-----------------------------------------------------------------------

          if ((ibedtd > 0).and.(MOD((time-stime),ideltb)==0)) then

            call bedtd(tsole,tair,hg,hgkt,Hice,geo,icover,iland,time)

            ! Write results
            call writb(hg,time)

          endif

# endif
        endif      ! End itherm ifblock

# if !defined uvic_embm
!-----------------------------------------------------------------------
!       convergence or iterout; write current fields
!-----------------------------------------------------------------------

        if (ntime <= 20000) iprint = 500
        if (ntime > 20000) iprint = 10000        ! Maintenance output
        if ((MOD(time,iprint)==0).or.(time==etime).or.(time==-12000) &
                .or. (time==-10000) .or. (time==-9000)) then
          call writ(hg,Hice,time)
          call writv(vt,vp,wvel,fpsht,ftsht,time)
          call writmass(balance,acc,smelt,bmelt,pddnet,annamp,tsea,tair,&
                        precip,time)
          if (igingembre > 0) then
            call writhyd(hg,hg0,hi,smelt,bmelt,rainfall,iland,time)
          endif
        endif
        if ((MOD(time,100)==0).and.(nlon==63)) then       ! EISMINT output
          call writeis(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,time)
        elseif ((MOD(time,100)==0).and.(nlon==120)) then         ! N America
          if (isubtop==0) then
            call writna(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                        calve,tair,tsea,precip,igreen,time)
          elseif (isubtop == 2) then                ! subgrid diagnostics
            call writna_sub(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,acc,smelt,&
                            calve,tair,tsea,precip,igreen,time)
          endif
        elseif ((MOD(time,200)==0).and.(nlon==30)) then          ! Cold Bay
          call writeis(hg,hi,Hice,fpsht,ftsht,vp,vt,wvel,time)
        endif

# endif
!-----------------------------------------------------------------------
!       store current as previous solution for next timestep
!-----------------------------------------------------------------------

        hgk   = hg
        Hicek = Hice
        hik   = hi
        pice  = rhoi*grav*Hice

# if defined uvic_embm
!-----------------------------------------------------------------------
!       sum handshake values for subroutine "bism_embm"
!-----------------------------------------------------------------------

        sum_Hice = sum_Hice + Hice
        sum_hi   = sum_hi   + hi
        sum_calv = sum_calv + fwater

!-------for output of global error in moisture conservation due to
!       ice sheet dynamics uncomment following lines
!        vol2 = c0 ! global ice volume after update
!        bal2 = c0 ! global mass balance
!        calv2 = c0 ! global calving rate
!        do j=1,nlat
!                 do i=2,nlon-1
!            vol2 = vol2 + Hice(i,j)*area(j)
!            bal2 = bal2 + bal(i,j)*area(j)
!            calv2 = calv2 + sum_calv(i,j)*area(j)
!          enddo
!              enddo
!        write(*,*)'vol1,vol2,bal2,calv2'
!        write(*,1) vol1,vol2,bal2,calv2
!    1   FORMAT(4E16.8)
!        write(*,2)'        vol2-vol1 = ',vol2-vol1
!    2   FORMAT(A20,E16.8)
!        write(*,2)'(bal2-calv2)*delt = ',(bal2-calv2)*delt
# endif

!-----------------------------------------------------------------------
!       calculate and write ice volume
!-----------------------------------------------------------------------

        if (MOD(icloop,idtvol) == 0) then
          call cidm_vol_calc(Hice,vol_SH,vol_NH)
          call cidm_vol_writ(vol_SH,vol_NH)
        endif

!-----------------------------------------------------------------------
!       write snapshot
!-----------------------------------------------------------------------

        if (MOD(icloop,idtsnap) == 0) then
          call cidm_snap_writ(sum_calv)
        endif

!-----------------------------------------------------------------------
!     E N D   BISM   M A I N   L O O P
!-----------------------------------------------------------------------

      enddo

# if defined uvic_embm
!-----------------------------------------------------------------------
!     average surface boundary conditions over the coupling period
!-----------------------------------------------------------------------

      sum_Hice  = sum_Hice / DBLE(nloop)
      sum_hi    = sum_hi   / DBLE(nloop)
      sum_calv  = sum_calv / (etime-stime)

!-----------------------------------------------------------------------
!     convert SBC grids for other components of the climate system model
!-----------------------------------------------------------------------

      call cidm_sbc_hl(sum_Hice,avg_Hice)
      call cidm_sbc_hl(sum_hi,avg_hi)
      call cidm_sbc_hl(sum_calv,avg_calv)
      sum_aice = sum_Hice*rhoi/rhos
      sum_aice = MAX(sum_aice,c0)
      sum_aice = MIN(sum_aice,c1)
      call cidm_sbc_hl(sum_aice,avg_aice)
      avg_aice = MAX(avg_aice,c0)
      avg_aice = MIN(avg_aice,c1)

# endif
!-----------------------------------------------------------------------
!     write restart
!-----------------------------------------------------------------------

      if (MOD(ieloop,idtrest) == 0) then
        call cidm_rest_writ(avg_Hice,avg_hi,avg_calv,avg_aice)
      endif

#endif
!=======================================================================
      end subroutine cidm
!=======================================================================
