!=======================================================================
!     This subroutine passes the surface boundary conditions averaged
!     over coupling interval between the land ice model and the
!     atmosphere model.

!     Variables passed to the atmospheric model from the land ice model:
!       1. ice thickness (for surface mass balance)
!       2. surface elevation (for surface air temperature)

!     N.B.
!       1. Surface altitude can be changed by ice thickness, directly
!          (via ice growth or decay) or indirectly (via isostatic
!          adjustment), so this should be calculated in the land ice
!          model.
!       2. The energy used for surface melting is supplied directly
!          from the atmosphere (and not by ice).
!       3. No refreezing
!       4. The latent heat difference between snow and ice melting is
!          not accounted in EMBM.
!       5. Sublimation and evaporation are not accounted so far in BISM.

!     based on code by: M. Yoshimori and A. Schmittner
!=======================================================================

      subroutine cidm_sbc_atm(atm_Hice,atm_hi)

#if defined ubc_cidm && defined uvic_embm
      USE global_param    ! dp,c0,c100
      USE cidm_mod_embm   ! iimt,jjmt
      implicit none

      REAL(KIND=dp),INTENT(INOUT),DIMENSION(1:iimt,1:jjmt) :: atm_Hice,&
                                                              atm_hi

!-----------------------------------------------------------------------
!     change unit and avoid negative value
!-----------------------------------------------------------------------

      atm_Hice = MAX(atm_Hice*c100,c0)
      atm_hi   = MAX(atm_hi*c100,c0)

#endif
!=======================================================================
      end subroutine cidm_sbc_atm
!=======================================================================

!=======================================================================
!     This subroutine, sbc_bm, passes the surface boundary
!     conditions averaged over coupling interval to the ocean model from
!     the land ice model (bism->embm).

!     Variables passed to the ocean model from the land ice model:
!       1. Energy flux into the ocean
!            latent: calving
!       2. Moisture flux into the ocean
!            calving

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_ocn(ocn_calv,ocn_ltnt)

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE global_param    ! dp,rhow,rhoi
      USE cidm_mod_embm   ! imt,jmt,flice
      implicit none

! ... subroutine arguments
      REAL(KIND=dp),INTENT(INOUT),DIMENSION(1:iimt,1:jjmt) :: ocn_calv
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:iimt,1:jjmt) :: ocn_ltnt

!-----------------------------------------------------------------------
!     ice thickness, avg_calv (m/a) to water-equivalent ice ablation,
!     avg_calv (cm/s)
!-----------------------------------------------------------------------
!AAA
# if defined andreas_no_mc
      ocn_calv = c0
# else
      ocn_calv = ocn_calv*c100*rhoi/rhow/year
# endif
!AAA
!-----------------------------------------------------------------------
!     latent heat flux, bism_ltnt (g/s**2/s)

!     N.B. Since unit of energy is (kg*m**2/s**2) or (g*cm**2/s**2),
!          latent heat flux per unit area and time should have the unit
!          of  (kg/s**2/s) or (g/s**2/s).
!-----------------------------------------------------------------------

      ocn_ltnt = ocn_calv*flice

#endif
!=======================================================================
      end subroutine cidm_sbc_ocn
!=======================================================================

!=======================================================================
!     This subroutine allocates arrays for the surface boundary
!     conditons.

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_allocate

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE global_param    ! c0
      USE cidm_mod_embm   ! iimt,jjmt,atm_s_ann,atm_m_ann,atm_r_ann
      implicit none

      ALLOCATE (atm_s_ann(1:iimt,1:jjmt)); atm_s_ann = c0
      ALLOCATE (atm_m_ann(1:iimt,1:jjmt)); atm_m_ann = c0
# if defined ubc_cidm_refreeze
      ALLOCATE (atm_r_ann(1:iimt,1:jjmt)); atm_r_ann = c0
# endif

#endif
!=======================================================================
      end subroutine cidm_sbc_allocate
!=======================================================================

!=======================================================================
!     This subroutine initializes the surface boundary conditons.

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_init

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE global_param    ! c0
      USE cidm_mod_embm   ! nsum,atm_s_ann,atm_m_ann,atm_r_ann
      implicit none

      nsum = 0
      atm_s_ann = c0
      atm_m_ann = c0
# if defined ubc_cidm_refreeze
      atm_r_ann = c0
# endif

#endif
!=======================================================================
      end subroutine cidm_sbc_init
!=======================================================================

!=======================================================================
!     This subroutine sums up the surface boundary conditions.

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_sum(subpsno,subprecip)

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE global_param    ! dp
      USE cidm_mod_embm   ! iimt,jjmt,nsum,atm_s_ann,atm_m_ann,atm_r_ann
      implicit none

! ... subroutine arguments
      REAL(KIND=dp),INTENT(IN),DIMENSION(1:iimt,1:jjmt) :: subpsno
      REAL(KIND=dp),INTENT(IN),DIMENSION(1:iimt,1:jjmt) :: subprecip

      nsum = nsum + 1
      WHERE (subpsno > c0)
        atm_s_ann = atm_s_ann + subpsno
      elseWHERE
        atm_m_ann = atm_m_ann - subpsno
# if defined ubc_cidm_refreeze
        atm_r_ann = atm_r_ann + subprecip
# endif
      end WHERE

#endif
!=======================================================================
      end subroutine cidm_sbc_sum
!=======================================================================

!=======================================================================
!     This subroutine averages the surface boundary conditions.

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_avg

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE cidm_mod_embm   ! nsum,atm_s_ann,atm_m_ann,atm_r_ann
      implicit none

      if (nsum == 0) then
        WRITE(*,*) 'error in cidm_sbc_avg: invalid value for nsum'
        WRITE(*,*) 'program will stop'
        STOP
      endif

      atm_s_ann = atm_s_ann/DBLE(nsum)
      atm_m_ann = atm_m_ann/DBLE(nsum)
# if defined ubc_cidm_refreeze
      atm_r_ann = atm_r_ann/DBLE(nsum)
# endif

#endif
!=======================================================================
      end subroutine cidm_sbc_avg
!=======================================================================

!=======================================================================
!     This subroutine, sbc_eb, passes the surface boundary
!     conditions averaged over coupling period to the land ice model
!     from the atmospheric model (embm->bism).

!     Variables passed to the ice-sheet model from the atmospheric model:
!       1. Moisture flux into the the land ice model
!            snowfall, freezed rainfall, evaporation and sublimation
!       2. Sea-level air temperature

!     N.B.
!       1. Neglect energy flux into the ice-sheet
!            radiation: shortwave radiation through the atmosphere
!                       into the ice-sheet
!                       longwave radiation from the atmosphere
!                       into the ice-sheet
!            sensible: turbulent
!            others: precipitation (rainfall and snowfall)
!       2. Sublimation and evaporation are not accounted so far in BISM

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_lice

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE cidm_mod_embm   ! iimt,jjmt,atm_s_ann,atm_m_ann,atm_r_ann
      USE cidm_mod_mass   ! s_ann,m_ann,r_ann,"global_param"
                          ! dp,nlon,nlat,c100,year
      implicit none

!-----------------------------------------------------------------------
!     unit transformation
!-----------------------------------------------------------------------

      atm_s_ann = atm_s_ann*year/c100
      atm_m_ann = atm_m_ann*year/c100
# if defined ubc_cidm_refreeze
      atm_r_ann = atm_r_ann*year/c100
# endif

!-----------------------------------------------------------------------
!     grid transformation
!-----------------------------------------------------------------------

      call cidm_sbc_lh(atm_s_ann,s_ann)
      call cidm_sbc_lh(atm_m_ann,m_ann)
# if defined ubc_cidm_refreeze
      call cidm_sbc_lh(atm_r_ann,r_ann)
# endif

#endif
!=======================================================================
      end subroutine cidm_sbc_lice
!=======================================================================

!=======================================================================
!     This subroutine deallocates arrays for the surface boundary
!     conditons.

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_deallocate

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE cidm_mod_embm   ! atm_s_ann,atm_m_ann,atm_r_ann
      implicit none

      DEALLOCATE (atm_s_ann)
      DEALLOCATE (atm_m_ann)
# if defined ubc_cidm_refreeze
      DEALLOCATE (atm_r_ann)
# endif

#endif
!=======================================================================
      end subroutine cidm_sbc_deallocate
!=======================================================================

!=======================================================================
!     This subroutine converts the grids for surface boundary conditions
!     from one model with lower horizontal resolution (imt,jmt) to
!     another with higher horizontal resolution (nlon,nlat).

!     embm -> cidm
!     (imt-2)*ni = nlon-2
!     (jmt-4)*nj = nlat-2

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_lh(data1,data2)

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE global_param     ! nlon,nlat,lon,lat,spval
      USE cidm_mod_embm    ! iimt,jjmt,xt,yt
      implicit none

! ... subroutine arguments
      REAL(KIND=dp),INTENT(IN),DIMENSION(1:iimt,1:jjmt) :: data1
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:nlon,1:nlat) :: data2

! ... local
      REAL(KIND=dp),DIMENSION(1:nlon*nlat) :: vector

      call cidm_tool_interp(iimt,jjmt,xt,yt,data1,&
                            nlon,nlat,lon,lat,data2,spval)
      call cidm_tool_convert1d(nlon,nlat,data2,vector)
      call cidm_tool_fill(nlon,nlat,vector,spval)
      call cidm_tool_convert2d(nlon,nlat,vector,data2)

!-----------------------------------------------------------------------
!     lateral boundary condition
!-----------------------------------------------------------------------

      call cidm_lbc_2d_real(data2)

#endif
!=======================================================================
      end subroutine cidm_sbc_lh
!=======================================================================

!=======================================================================
!     This subroutine converts the grids for surface boundary conditions
!     from one model with higher horizontal resolution (nlon,nlat) to
!     another with lower horizontal resolution (imt,jmt).

!     cidm -> embm
!     nlon-2 = (imt-2)*ni
!     nlat-2 = (jmt-4)*nj

!     based on code by: M. Yoshimori
!=======================================================================

      subroutine cidm_sbc_hl(data1,data2)

#if defined ubc_cidm && defined uvic_embm
! ... global
      USE global_param     ! nlon,nlat,spval
      USE cidm_mod_embm    ! iimt,jjmt
      implicit none

! ... subroutine arguments
      REAL(KIND=dp),INTENT(IN),DIMENSION(1:nlon,1:nlat) :: data1
      REAL(KIND=dp),INTENT(OUT),DIMENSION(1:iimt,1:jjmt) :: data2

! ... local
      REAL(KIND=dp),DIMENSION(1:iimt*jjmt) :: vector

      call cidm_tool_interp(nlon,nlat,lon,lat,data1,&
                            iimt,jjmt,xt,yt,data2,spval)
      call cidm_tool_convert1d(iimt,jjmt,data2,vector)
      call cidm_tool_fill(iimt,jjmt,vector,spval)
      call cidm_tool_convert2d(iimt,jjmt,vector,data2)

!-----------------------------------------------------------------------
!     lateral boundary condition
!-----------------------------------------------------------------------

      call cidm_lbc_embm(data2)

#endif
!=======================================================================
      end subroutine cidm_sbc_hl
!=======================================================================
