!=======================================================================
!     Calculate current ice sheet fluxes over the entire grid
!     Fluxes computed at the cell interface, Crank-Nicholson weighting
!     on the explicit and implicit contributions from ice thickness

!     Save a great deal of information for use in the Jacobian
!     calculation.

!     *_n  : * on the north face of a cell
!     vb   : southward basal sliding velocity (2D)
!     v    : southward velocity (3D)
!     flux : southward advective flux
!     Hice : ice thickness

!     N.B. General sign convention: eastward positive, southward
!            positive
!=======================================================================

      subroutine is_dyn_nflux(time,ifloating,tiso,Dcoeff,Hvimp,hivimp, &
                              vb_n,scale_n,v_n,flux_n,Hice_n,gradx_n,  &
                              grady_n,div_n,ganma_n,dbar_n)

      USE icetd_arrays
      USE shelf_arrays
      implicit none

      INTEGER, INTENT(IN) :: time
      INTEGER, INTENT(IN), DIMENSION(nlon,nlat) :: ifloating
      REAL(dp), INTENT(IN) :: tiso,Dcoeff
      REAL(dp), INTENT(IN), DIMENSION(ndsys) :: Hvimp,hivimp
      REAL(dp), INTENT(IN), DIMENSION(nlonp,nlat-1) :: vb_n
      REAL(dp), INTENT(INOUT), DIMENSION(nlonp,nlat-1) :: scale_n
      REAL(dp), INTENT(INOUT), DIMENSION(nvertp,nlonp,nlat-1) :: v_n
      REAL(dp), INTENT(OUT), DIMENSION(nlonp,nlat-1) :: flux_n,Hice_n, &
                                 gradx_n,grady_n,div_n,dbar_n,ganma_n

      ! Local variables
      integer :: i,j,im,ip,k,kz,ktabrt,ktabhi
      REAL(dp) :: zsumhi,zetacur,vhicur,Binthi,hi_N
      REAL(dp), DIMENSION(nvert) :: Bzhi

      REAL(dp), EXTERNAL :: B

!-----------------------------------------------------------------------
!     Loop over the interior (physical) model grid.
!     Calculate eastern (rt) and northern (hi) interface fluxes
!-----------------------------------------------------------------------

      do j=2,nlat-2
long:   do i=2,nlon-1
          im = i-1
          ip = i+1
          k = i + nlon*(j-1)

!-----------------------------------------------------------------------
!         ice thickness at the north cell interface: Hice(i,j+1/2)
!-----------------------------------------------------------------------

          Hice_n(im,j) = (Hvimp(k)+Hvimp(k+nlon))/c2

!-----------------------------------------------------------------------
!         jump to next point when no ice available
!-----------------------------------------------------------------------

          if (Hice_n(im,j) < zerop) then
            flux_n(im,j) = c0
            CYCLE long
          endif

!-----------------------------------------------------------------------
!         if both floating, no vertical shear flux, i.e., basal sliding
!         only
!-----------------------------------------------------------------------

          if (ishelf > 0) then   ! floating ice option
            if ((ifloating(i,j) == 1) .and. (ifloathi(im,j) == 1)) then
              flux_n(im,j) = -vb_n(im,j)*Hice_n(im,j)
              scale_n(im,j) = c0
              CYCLE long
            endif
          endif

!-----------------------------------------------------------------------
!         gradient in  x direction: 1/sin(theta)*(d_hi/d_ramuda)i,j+1/2
!-----------------------------------------------------------------------

          gradx_n(im,j) = ((hivimp(k+1+nlon)-hivimp(k-1+nlon))/sine(j+1) &
                        +  (hivimp(k+1     )-hivimp(k-1     ))/sine(j))  &
                        / (c4*delphi)

!-----------------------------------------------------------------------
!         gradient in -y direction: (d_hi/d_theta)i,j+1/2
!-----------------------------------------------------------------------

          grady_n(im,j) = - (hivimp(k+nlon)-hivimp(k))/deltheta

!-----------------------------------------------------------------------
!         divergence
!-----------------------------------------------------------------------

          div_n(im,j) = gradx_n(im,j)**2 + grady_n(im,j)**2

!-----------------------------------------------------------------------
!         ((ganma)i,j+1/2)**(n-1)
!-----------------------------------------------------------------------

          ganma_n(im,j) = (SQRT(div_n(im,j)))**(gn-1)

!-----------------------------------------------------------------------
!         Sheet Fluxes out of control volume
!         out:  fluxhi, fluxlt (i.e. positive fluxhi = out) (follow slopes)
!         in:   fluxlo, fluxrt

!         Numerical integration over ice thickness to calculate
!           velocities and fluxes;
!         use scaled vertical co-ordinate zeta=exp(a(hi-z)/Hice), with
!           zeta=exp(a) at the bed and zeta=1 at the surface;
!         delzeta(nvert) is fixed, as defined in global_param

!         N.B. Flux sign convention: counter-intuitive (following the
!                slope), such that velocity signs need reversed
!-----------------------------------------------------------------------

          if ((itherm == 0) .or. (itdcoup ==0)) then

!-----------------------------------------------------------------------
!           isothermal treatment
!-----------------------------------------------------------------------

            ! column ice stiffness Bbar
            dbar_n(im,j) = Dcoeff*Biso/DBLE(gnp2)

            ! southward flux at north cell interface
            flux_n(im,j) = sinehi(j)*(-vb_n(im,j)*Hice_n(im,j)         &
           +dbar_n(im,j)*ganma_n(im,j)*grady_n(im,j)*Hice_n(im,j)**gnp2)

            if (ctime == etime) then   ! velocities for output
              hi_N = (hivimp(k)+hivimp(k+nlon))/c2
              v_n(1,im,j) = vb_n(im,j)
              do kz=1,nvert
                zetacur = (zeta(kz+1)+zeta(kz))/c2
                v_n(kz+1,im,j) = v_n(1,im,j)                           &
                             + dbar_n(im,j)*DBLE(gnp2)/DBLE(gnp1)      &
                               *ganma_n(im,j)*grady_n(im,j)            &
                               *((Hice_n(im,j)*LOG(zetacur)/aa)**gnp1  &
                                 -Hice_n(im,j)**gnp1)
              enddo
            endif

          else

!-----------------------------------------------------------------------
!           polythermal treatment
!-----------------------------------------------------------------------

            ! First integration: velocities
            v_n(1,im,j) = vb_n(im,j)
            Binthi = c0
            do kz=1,nvert
              zetacur = (zeta(kz+1)+zeta(kz))/c2
              Bzhi(kz) = B(tt(kz,im,j))*1.67_dp
              Binthi = Binthi &
                     - LOG(zetacur)**gn*Bzhi(kz)*delzeta/zetacur
              v_n(kz+1,im,j) = v_n(1,im,j) - Dcoeff*grady_n(im,j) &
                        *(Hice_n(im,j)/aa)**gnp1*ganma_n(im,j)*Binthi
            enddo

            ! Second integration: fluxes
            ! Take linear average of variables in cell centre
            zsumhi = c0
            do kz=1,nvert
              zetacur = (zeta(kz+1)+zeta(kz))/c2
              vhicur = (v_n(kz,im,j)+v_n(kz+1,im,j))/c2
              zsumhi = zsumhi + delzeta/zetacur*vhicur
            enddo

            flux_n(im,j) = sinehi(j)*Hice_n(im,j)*(-vb_n(im,j) + zsumhi/aa)

            ! Vertically-integrated ice stiffness coefficient
            dbar_n(im,j) = Bzhi(3)*Dcoeff/gnp2*aa**gnp2

          endif

        enddo long   ! End longitude loop
      enddo          ! End latitude loop

      v_n(1:nvertp,1:nlonp,1) = c0
      scale_n(1:nlonp,1) = c0
      flux_n (1:nlonp,1) = c0
      Hice_n (1:nlonp,1) = c0
      gradx_n(1:nlonp,1) = c0
      grady_n(1:nlonp,1) = c0
      div_n  (1:nlonp,1) = c0
      ganma_n(1:nlonp,1) = c0
      dbar_n (1:nlonp,1) = c0

      v_n(1:nvertp,1:nlonp,nlat-1) = c0
      scale_n(1:nlonp,nlat-1) = c0
      flux_n (1:nlonp,nlat-1) = c0
      Hice_n (1:nlonp,nlat-1) = c0
      gradx_n(1:nlonp,nlat-1) = c0
      grady_n(1:nlonp,nlat-1) = c0
      div_n  (1:nlonp,nlat-1) = c0
      ganma_n(1:nlonp,nlat-1) = c0
      dbar_n (1:nlonp,nlat-1) = c0

!=======================================================================
      end subroutine is_dyn_nflux
!=======================================================================
