      subroutine mom_rest_in (fname, ids, ide, jds, jde)
!=======================================================================
!     input routine for atmospheric restarts

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, ib, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain

!   local variables
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     based on code by: M. Eby
!=======================================================================
#if defined uvic_mom

!      implicit none

# include "param.h"
# if defined trajectories
#  include "ptraj.h"
# endif
# include "coord.h"
# include "emode.h"
# include "grdvar.h"
# include "iounit.h"
# include "levind.h"
# include "mw.h"
# include "task_on.h"
# include "tmngr.h"
# include "switch.h"

      character(*) :: fname
      character(120) :: name, new_file_name
      character(3) :: a3

      integer i, iou, j, ln, n, ntrec, ids, ide, jds, jde
      integer jils, ile, jls, jle, kls, kle
      integer ib(10), ic(10), it(10), iu(10), id_time, id_xt, id_xu
      integer id_yt, id_yu, id_zt, id_zw, id_xt_e, id_xu_e, id_yt_e
      integer id_yu_e, id_zt_e, id_zw_e, id_part
      integer nyear, nmonth, nday, nhour, nmin, nsec

      logical inqvardef

      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)
      real zt_e(km+1), zw_e(km+1), time, tmp
      real, allocatable :: tmpij(:,:),tmpik(:,:)

!-----------------------------------------------------------------------
!     open file
!-----------------------------------------------------------------------
      name = new_file_name (fname)
      call openfile (name, iou)
      ntrec = 1

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global read domain)
!-----------------------------------------------------------------------
      ils = max(ids,1)
      ile = min(ide,imt)
      jls = max(jds,1)
      jle = min(jde,jmt)
      kls = 1
      kle = km

      allocate ( tmpij(ils:ile,jls:jle) )
      allocate ( tmpik(ils:ile,kls:kle) )

!-----------------------------------------------------------------------
!     read 1d data (t)
!-----------------------------------------------------------------------
      tmp = itt
      call getvars ('itt', iou, 1, tmp, c1, c0)
      itt = tmp
      tmp = irstdy
      call getvars ('irstdy', iou, 1, tmp, c1, c0)
      irstdy = tmp
      tmp = msrsdy
      call getvars ('msrsdy', iou, 1, tmp, c1, c0)
      msrsdy = tmp
# if defined trajectories
      tmp = 0.
      call getvars ('reltim', iou, 1, tmp, c1, c0)
      relyr = tmp
      tmp = 0.
      call getvars ('nptraj', iou, 1, tmp, c1, c0)
      npart = tmp
      write (stdout, '(a,i8,a)')
     &'===> also read trajectory restart for ',npart,' particles'
      if (npart .eq. 0) then
        call ptraji
        npart = nptraj
      endif
      if (npart .ne. nptraj) then
        write (stdout,*) ' => Particle trajectory restart error:'
        write (stdout,*) '    read ',npart,' particles but expected '
     &,                     nptraj
        stop "=>mom_rest_in"
      endif
      ln = 3*nptraj
      pxyz(:) = 0.
      call getvara ('pxyz', iou, ln, (/1/), (/ln/), pxyz, c1, c0)
      pijk(:) = 0.
      call getvara ('pijk', iou, ln, (/1/), (/ln/), pijk, c1, c0)
#  if defined lyapunov
      ln = 2*2*nptraj
      em(:) = 0.
      call getvara ('em', iou, ln, (/1/), (/ln/), em, c1, c0)
#  endif
# endif
      tmp = year0
      call getvars ('year', iou, 1, tmp, c1, c0)
      nyear = tmp
      tmp = month0
      call getvars ('month', iou, 1, tmp, c1, c0)
      nmonth = tmp
      tmp = day0
      call getvars ('day', iou, 1, tmp, c1, c0)
      nday = tmp
      tmp = hour0
      call getvars ('hour', iou, 1, tmp, c1, c0)
      nhour = tmp
      tmp = min0
      call getvars ('minute', iou, 1, tmp, c1, c0)
      nmin = tmp
      tmp = sec0
      call getvars ('second', iou, 1, tmp, c1, c0)
      nsec = tmp
      call mkstmp (stamp, nyear, nmonth, nday, nhour, nmin, nsec)
      if (init_time_in) then
        itt = 0
        irstdy = 0
        msrsdy = 0
        relyr = 0.
        call mkstmp (stamp, year0, month0, day0, hour0, min0, sec0)
      endif

!-----------------------------------------------------------------------
!     read 2d data (x,y)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ile-ils+1
      ib(2) = 1
      ic(2) = jle-jls+1
      ln = ic(1)*ic(2)
      tmpij(:,:) = kmt(ils:ile,jls:jle)
      call getvara ('kmt', iou, ln, ib, ic, tmpij, c1, c0)
      kmt(ils:ile,jls:jle) = tmpij(ils:ile,jls:jle)

!-----------------------------------------------------------------------
!     read 3d data (x,y,t)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ile-ils+1
      ib(2) = 1
      ic(2) = jle-jls+1
      ib(3) = 1
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)
# if defined coarse_grained_parallelism
      taum1disk = mod(itt+2,3) + 1
      taudisk   = mod(itt  ,3) + 1
      taup1disk = mod(itt+1,3) + 1
# else
      taum1disk = mod(itt+1,2) + 1
      taudisk   = mod(itt  ,2) + 1
      taup1disk = taum1disk
# endif
!     update pointers to tau-1, tau, & tau+1 data in the MW based on itt
      if (wide_open_mw) then
!       rotate time levels instead of moving data
        taum1 = mod(itt+0,3) - 1
        tau   = mod(itt+1,3) - 1
        taup1 = mod(itt+2,3) - 1
      endif

# if defined rigid_lid_surface_pressure || defined implicit_free_surface
      tmpij(ils:ile,jls:jle) = ps(ils:ile,jls:jle,1)
      call getvara ('ps1', iou, ln, ib, ic, tmpij, c1, c0)
      ps(ils:ile,jls:jle,1) = tmpij(ils:ile,jls:jle)
      tmpij(ils:ile,jls:jle) = ps(ils:ile,jls:jle,2)
      call getvara ('ps2', iou, ln, ib, ic, tmpij, c1, c0)
      ps(ils:ile,jls:jle,2) = tmpij(ils:ile,jls:jle)
!     (ubarm, ubar) is at (tau, tau+1)
      tmpij(ils:ile,jls:jle) = ubarm1(ils:ile,jls:jle,1)
      call getvara ('ubarm1', iou, ln, ib, ic, tmpij, c1, c0)
      ubarm1(ils:ile,jls:jle,1) = tmpij(ils:ile,jls:jle)
      tmpij(ils:ile,jls:jle) = ubarm1(ils:ile,jls:jle,2)
      call getvara ('ubarm2', iou, ln, ib, ic, tmpij, c1, c0)
      ubarm1(ils:ile,jls:jle,2) = tmpij(ils:ile,jls:jle)
      tmpij(ils:ile,jls:jle) = ubar(ils:ile,jls:jle,1)
      call getvara ('ubar1', iou, ln, ib, ic, tmpij, c1, c0)
      ubar(ils:ile,jls:jle,1) = tmpij(ils:ile,jls:jle)
      tmpij(ils:ile,jls:jle) = ubar(ils:ile,jls:jle,2)
      call getvara ('ubar2', iou, ln, ib, ic, tmpij, c1, c0)
      ubar(ils:ile,jls:jle,2) = tmpij(ils:ile,jls:jle)

!     initialize the kflds disk which is used for surface pressure.
      pguess(:,:) = 0.
      ptd(:,:) = ps(:,:,1)
      call oput (kflds, nwds, nkflds-1, ptd)
      call oput (kflds, nwds, nkflds, ptd)

# endif
# if defined stream_function
!     first do psi at "tau" then at "tau+1"
      tmpij(ils:ile,jls:jle) = tmpij(ils:ile,jls:jle)
      call getvara ('psi1', iou, ln, ib, ic, tmpij, c1, c0)
      psi(ils:ile,jls:jle,1) = tmpij(ils:ile,jls:jle)
      tmpij(ils:ile,jls:jle) = psi(ils:ile,jls:jle,2)
      call getvara ('psi2', iou, ln, ib, ic, tmpij, c1, c0)
      psi(ils:ile,jls:jle,2) = tmpij(ils:ile,jls:jle)
!     guess fields
      tmpij(ils:ile,jls:jle) = ptd(ils:ile,jls:jle)
      call getvara ('ptd1', iou, ln, ib, ic, tmpij, c1, c0)
      ptd(ils:ile,jls:jle) = tmpij(ils:ile,jls:jle)
      call oput (kflds, nwds, nkflds-1, ptd)
      tmpij(ils:ile,jls:jle) = ptd(ils:ile,jls:jle)
      call getvara ('ptd2', iou, ln, ib, ic, tmpij, c1, c0)
      ptd(ils:ile,jls:jle) = tmpij(ils:ile,jls:jle)
      call oput (kflds, nwds, nkflds, ptd)
# endif

# if defined uvic_mom
!     construct depth arrays associated with "u" cells

      call depth_u (kmt, imt, jmt, zw, km, kmu, h, hr)

# endif
!-----------------------------------------------------------------------
!     read 4d data (x,y,z,t)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ile-ils+1
      ib(2) = 1
      ic(2) = 1
      ib(3) = 1
      ic(3) = kle-kls+1
      ib(4) = 1
      ic(4) = 1
      ln = ic(1)*ic(2)*ic(3)*ic(4)

!     read the "tau" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
          call getrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
        endif
        ib(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          tmpik(ils:ile,kls:kle) = t(ils:ile,kls:kle,j,n,tau)
          call getvara('tracer1_'//trim(a3), iou, ln, ib, ic
     &,     tmpik, c1, c0)
          t(ils:ile,kls:kle,j,n,tau) = tmpik(ils:ile,kls:kle)

        enddo
        tmpik(ils:ile,kls:kle) = u(ils:ile,kls:kle,j,1,tau)
        call getvara('u1', iou, ln, ib, ic, tmpik, c1, c0)
        u(ils:ile,kls:kle,j,1,tau) = tmpik(ils:ile,kls:kle)
        tmpik(ils:ile,kls:kle) = u(ils:ile,kls:kle,j,2,tau)
        call getvara('v1', iou, ln, ib, ic, tmpik, c1, c0)
        u(ils:ile,kls:kle,j,2,tau) = tmpik(ils:ile,kls:kle)
!       initialize every latitude
        if (wide_open_mw) then
!         do nothing since "tau" data is in place in the MW
        else
          call putrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
        endif
      enddo
      if (wide_open_mw) then
!       Initialze 1st and last latitude row for tau-1 to prevent
!       use of uninitialized values on boundary row.
        do j=1,jmt,jmt-1
          do k=1,km
            do i=1,imt
              u(i,k,j,1,taum1) = u(i,k,j,1,tau)
              u(i,k,j,2,taum1) = u(i,k,j,2,tau)
              do n=1,nvar-2
                t(i,k,j,n,taum1) = t(i,k,j,n,tau)
              enddo
            enddo
          enddo
        enddo
      endif

!     read the "tau+1" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
          call getrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif
        ib(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          tmpik(ils:ile,kls:kle) = t(ils:ile,kls:kle,j,n,taup1)
          call getvara('tracer2_'//trim(a3), iou, ln, ib, ic
     &,     tmpik, c1, c0)
          t(ils:ile,kls:kle,j,n,taup1) = tmpik(ils:ile,kls:kle)
        enddo
        tmpik(ils:ile,kls:kle) = u(ils:ile,kls:kle,j,1,taup1)
        call getvara('u2', iou, ln, ib, ic, tmpik, c1, c0)
        u(ils:ile,kls:kle,j,1,taup1) = tmpik(ils:ile,kls:kle)
        tmpik(ils:ile,kls:kle) = u(ils:ile,kls:kle,j,2,taup1)
        call getvara('v2', iou, ln, ib, ic, tmpik, c1, c0)
        u(ils:ile,kls:kle,j,2,taup1) = tmpik(ils:ile,kls:kle)
!       initialize every latitude
        if (wide_open_mw) then
!         do nothing since "tau+1" data is in place in the MW
        else
          call putrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif
      enddo

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      print*, '=> Ocn restart read from ',trim(fname),' on ', stamp

      deallocate (tmpij)
      deallocate (tmpik)

      call closefile (iou)

      return
      end

      subroutine mom_rest_out (fname, ids, ide, jds, jde)
!=======================================================================
!     output routine for atmospheric restarts

!     data may be sized differently in x and y from the global fields.
!     fields may be written with or without a time dimension. data
!     should be defined with the routine defvar and written with putvar.
!     if no time dimension, then data is only written once per file.
!     make sure the it, iu, ib, and ic arrays and are defining the
!     correct dimensions. ln may also need to be recalculated.

!   inputs:
!     fname              = file name
!     ids, ide ...       = start and end index for data domain

!   local variables
!     igs, ige, jgs, jge = global write domain start and end indicies
!     ig, jg             = global write domain size
!     ils, ile, jls, jle = local domain start and end indicies
!     it                 = t grid axis definitions (x,y,t default)
!     iu                 = u grid axis definitions (x,y,t default)
!     is                 = start for write on each axis (x,y,t default)
!     ic                 = count for write on each axis (x,y,t default)
!     id_...             = id's for axis (used for it, iu or defvar)
!     iou                = io unit (ncid)
!     ln                 = length of data to be written

!     based on code by: M. Eby
!=======================================================================

!      implicit none

# include "param.h"
# if defined trajectories
#  include "ptraj.h"
# endif
# include "coord.h"
# include "emode.h"
# include "grdvar.h"
# include "iounit.h"
# include "levind.h"
# include "mw.h"
# include "switch.h"
# include "task_on.h"
# include "tmngr.h"

      character(*) :: fname
      character(120) :: name, new_file_name, title
      character(3) :: a3
      character(32) :: lstamp

      integer i, iou, j, ln, n, ntrec, ids, ide, jds, jde, igs, ige, ig
      integer jgs, jge, jg, kgs, kge, kg, ils, ile, jls, jle, kls, kle
      integer ib(10), ic(10), it(10), iu(10), id_time, id_xt, id_xu
      integer id_yt, id_yu, id_zt, id_zw, id_xt_e, id_xu_e, id_yt_e
      integer id_yu_e, id_zt_e, id_zw_e, id_part
      integer nyear, nmonth, nday, nhour, nmin, nsec

      real xt_e(imt+1), xu_e(imt+1), yt_e(jmt+1), yu_e(jmt+1)
      real zt_e(km+1), zw_e(km+1), time, tmp, bufsl(imt,km,2)
      real ext(imt,2), c100, c1e3, c1e20
      real, allocatable :: tmpij(:,:), tmpik(:,:)
      real, allocatable :: tmpi(:), tmpj(:), tmpk(:)
      real, allocatable :: tmpie(:), tmpje(:), tmpke(:)

      c100 = 100.
      c1e3 = 1.e3
      c1e20 = 1.e20
      lstamp = stamp

!-----------------------------------------------------------------------
!     open file
!-----------------------------------------------------------------------
      name = new_file_name (fname)
      call openfile (name, iou)
      ntrec = 1

!-----------------------------------------------------------------------
!     set global write domain size
!-----------------------------------------------------------------------
      igs = 1
      ige = imt
      ig  = ige-igs+1
      jgs = 1
      jge = jmt
      jg  = jge-jgs+1
      kgs = 1
      kge = km
      kg  = kge-kgs+1

!-----------------------------------------------------------------------
!     start definitions
!-----------------------------------------------------------------------
      call redef (iou)

!-----------------------------------------------------------------------
!     set global attributes
!-----------------------------------------------------------------------
      title = 'Restart'
      call putglobal (iou, name, title, expnam, timunit)

!-----------------------------------------------------------------------
!     define dimensions
!-----------------------------------------------------------------------
      call defdim ('time', iou, 0, id_time)
      call defdim ('xt', iou, ig, id_xt)
      call defdim ('yt', iou, jg, id_yt)
      call defdim ('zt', iou, kg, id_zt)
      call defdim ('xu', iou, ig, id_xu)
      call defdim ('yu', iou, jg, id_yu)
      call defdim ('zw', iou, kg, id_zw)
      call defdim ('xt_edges', iou, ig+1, id_xt_e)
      call defdim ('yt_edges', iou, jg+1, id_yt_e)
      call defdim ('zt_edges', iou, kg+1, id_zt_e)
      call defdim ('xu_edges', iou, ig+1, id_xu_e)
      call defdim ('yu_edges', iou, jg+1, id_yu_e)
      call defdim ('zw_edges', iou, kg+1, id_zw_e)
# if defined trajectories
      call defdim ('part', iou, nptraj*4, id_part)
# endif

!-----------------------------------------------------------------------
!     define 1d data (t)
!-----------------------------------------------------------------------
      it(1) = id_time
      call defvar ('time', iou, 1, it, c0, c0, 'T', 'D'
     &, 'time since initial condition', 'time', trim(timunit))
      call defvar ('itt', iou, 1, it, c0, c0, ' ', 'D'
     &, 'itt', ' ',' ')
      call defvar ('irstdy', iou, 1, it, c0, c0, ' ', 'D'
     &, 'irstdy', ' ',' ')
      call defvar ('msrsdy', iou, 1, it, c0, c0, ' ', 'D'
     &, 'msrsdy', ' ',' ')
# if defined trajectories
      iu(1) = id_part
      call defvar ('reltim', iou, 1, it, c0, c0, ' ', 'D'
     &, 'reltim', ' ',' ')
      call defvar ('nptraj', iou, 1, it, c0, c0, ' ', 'D'
     &, 'nptraj', ' ',' ')
      call defvar ('pxyz', iou, 1, iu, c0, c0, ' ', 'D'
     &, 'pxyz', ' ',' ')
      call defvar ('pijk', iou, 1, iu, c0, c0, ' ', 'D'
     &, 'pijk', ' ',' ')
#  if defined lyapunov
      call defvar ('em', iou, 1, iu, c0, c0, ' ', 'D'
     &, 'em', ' ',' ')
#  endif
# endif
      call defvar ('year', iou, 1, it, c0, c0, ' ', 'D'
     &, 'year', ' ',' ')
      call defvar ('month', iou, 1, it, c0, c0, ' ', 'D'
     &, 'month', ' ',' ')
      call defvar ('day', iou, 1, it, c0, c0, ' ', 'D'
     &, 'day', ' ',' ')
      call defvar ('hour', iou, 1, it, c0, c0, ' ', 'D'
     &, 'hour', ' ',' ')
      call defvar ('minute', iou, 1, it, c0, c0, ' ', 'D'
     &, 'minute', ' ',' ')
      call defvar ('second', iou, 1, it, c0, c0, ' ', 'D'
     &, 'second', ' ',' ')

!-----------------------------------------------------------------------
!     define 1d data (x, y or z)
!-----------------------------------------------------------------------
      it(1) = id_xt
      call defvar ('xt', iou, 1, it, c0, c0, 'X', 'D'
     &, 'longitude of the t grid', 'grid_longitude', 'degrees_east')
      it(1) = id_yt
      call defvar ('yt', iou, 1, it, c0, c0, 'Y', 'D'
     &, 'latitude of the t grid', 'grid_latitude', 'degrees_north')
      it(1) = id_zt
      call defvar ('zt', iou, 1, it, c0, c0, 'Z', 'D'
     &, 'depth of the t grid', 'depth', 'm')
      it(1) = id_xu
      call defvar ('xu', iou, 1, it, c0, c0, 'X', 'D'
     &, 'longitude of the u grid', 'grid_longitude', 'degrees_east')
      it(1) = id_yu
      call defvar ('yu', iou, 1, it, c0, c0, 'Y', 'D'
     &, 'latitude of the u grid', 'grid_latitude', 'degrees_north')
      it(1) = id_zw
      call defvar ('zw', iou, 1, it, c0, c0, 'Z', 'D'
     &, 'depth of the w grid', 'depth', 'm')
      it(1) = id_xt_e
      call defvar ('xt_edges', iou, 1, it, c0, c0, ' ', 'D'
     &, 'longitude of t grid edges', ' ', 'degrees')
      it(1) = id_yt_e
      call defvar ('yt_edges', iou, 1, it, c0, c0, ' ', 'D'
     &, 'latitude of t grid edges', ' ', 'degrees')
      it(1) = id_zt_e
      call defvar ('zt_edges', iou, 1, it, c0, c0, ' ', 'D'
     &, 'depth of t grid edges', ' ', 'm')
      it(1) = id_xu_e
      call defvar ('xu_edges', iou, 1, it, c0, c0, ' ', 'D'
     &, 'longitude of u grid edges', ' ', 'degrees')
      it(1) = id_yu_e
      call defvar ('yu_edges', iou, 1, it, c0, c0, ' ', 'D'
     &, 'latitude of u grid edges', ' ', 'degrees')
      it(1) = id_zw_e
      call defvar ('zw_edges', iou, 1, it, c0, c0, ' ', 'D'
     &, 'depth of w grid edges', ' ', 'm')

!-----------------------------------------------------------------------
!     define 2d data (x,y)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      call defvar ('kmt', iou, 2, it, c0, c1e3, ' ', 'I'
     &,  'kmt', ' ' ,' ')

!-----------------------------------------------------------------------
!     define 3d data (x,y,t)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      it(3) = id_time
      iu(3) = id_time
# if defined rigid_lid_surface_pressure || defined implicit_free_surface
      call defvar ('ps1', iou, 3, it,  -c1e20, c1e20, ' ', 'D'
     &, 'ps1', ' ', ' ')
      call defvar ('ps2', iou, 3, it,  -c1e20, c1e20, ' ', 'D'
     &, 'ps2', ' ', ' ')
      call defvar ('ubarm1', iou, 3, iu,  -c1e20, c1e20, ' ', 'D'
     &, 'ubarm1', ' ', ' ')
      call defvar ('ubarm2', iou, 3, iu,  -c1e20, c1e20, ' ', 'D'
     &, 'ubarm2', ' ', ' ')
      call defvar ('ubar1', iou, 3, iu,  -c1e20, c1e20, ' ', 'D'
     &, 'ubar1', ' ', ' ')
      call defvar ('ubar2', iou, 3, iu,  -c1e20, c1e20, ' ', 'D'
     &, 'ubar2', ' ', ' ')
# endif
# if defined stream_function
      call defvar ('psi1', iou, 3, it,  -c1e20, c1e20, ' ', 'D'
     &, 'psi1', ' ', ' ')
      call defvar ('psi2', iou, 3, it,  -c1e20, c1e20, ' ', 'D'
     &, 'psi2', ' ', ' ')
      call defvar ('ptd1', iou, 3, it,  -c1e20, c1e20, ' ', 'D'
     &, 'ptd1', ' ', ' ')
      call defvar ('ptd2', iou, 3, it,  -c1e20, c1e20, ' ', 'D'
     &, 'ptd2', ' ', ' ')
# endif

!-----------------------------------------------------------------------
!     define 4d data (x,y,z,t)
!-----------------------------------------------------------------------
      it(1) = id_xt
      iu(1) = id_xu
      it(2) = id_yt
      iu(2) = id_yu
      it(3) = id_zt
      iu(3) = id_zt
      it(4) = id_time
      iu(4) = id_time
      do n=1,nt
        if (n .lt. 1000) write(a3,'(i3)') n
        if (n .lt. 100) write(a3,'(i2)') n
        if (n .lt. 10) write(a3,'(i1)') n
        call defvar ('tracer1_'//trim(a3), iou , 4, it, -c1e20, c1e20
     &,   ' ', 'D', 'tracer1_'//trim(a3), ' ', ' ')
        call defvar ('tracer2_'//trim(a3), iou , 4, it, -c1e20, c1e20
     &,   ' ', 'D', 'tracer2_'//trim(a3), ' ', ' ')
      enddo
      call defvar ('u1', iou , 4, iu, -c1e20, c1e20, ' '
     &, 'D', 'u1', ' ', ' ')
      call defvar ('u2', iou , 4, iu, -c1e20, c1e20, ' '
     &, 'D', 'u2', ' ', ' ')
      call defvar ('v1', iou , 4, iu, -c1e20, c1e20, ' '
     &, 'D', 'v1', ' ', ' ')
      call defvar ('v2', iou , 4, iu, -c1e20, c1e20, ' '
     &, 'D', 'v2', ' ', ' ')

!-----------------------------------------------------------------------
!     end definitions
!-----------------------------------------------------------------------
      call enddef (iou)

!-----------------------------------------------------------------------
!     local domain size (minimum of data domain and global write domain)
!-----------------------------------------------------------------------
      ils = max(ids,igs)
      ile = min(ide,ige)
      jls = max(jds,jgs)
      jle = min(jde,jge)
      kls = max(1,kgs)
      kle = min(km,kge)

      allocate ( tmpij(ils:ile,jls:jle) )
      allocate ( tmpik(ils:ile,kls:kle) )
      allocate ( tmpi(igs:ige) )
      allocate ( tmpj(jgs:jge) )
      allocate ( tmpk(kgs:kge) )
      allocate ( tmpie(igs:ige+1) )
      allocate ( tmpje(jgs:jge+1) )
      allocate ( tmpke(kgs:kge+1) )

!-----------------------------------------------------------------------
!     write 1d data (t)
!-----------------------------------------------------------------------
      if (init_time_out) then
        tmp = 0.
        call putvars ('time', iou, ntrec, 0., c1, c0)
        tmp = 0.
        call putvars ('itt', iou, ntrec, tmp, c1, c0)
        tmp = 0.
        call putvars ('irstdy', iou, ntrec, tmp, c1, c0)
        tmp = 0.
        call putvars ('msrsdy', iou, ntrec, tmp, c1, c0)
        call mkstmp (lstamp, year0, month0, day0, hour0, min0, sec0)
      else
        tmp = relyr
        call putvars ('time', iou, ntrec, 0., c1, c0)
        tmp = itt
        call putvars ('itt', iou, ntrec, tmp, c1, c0)
        tmp = iday(imodeltime)
        call putvars ('irstdy', iou, ntrec, tmp, c1, c0)
        tmp = msday(imodeltime)
        call putvars ('msrsdy', iou, ntrec, tmp, c1, c0)
      endif
# if defined trajectories
      if (init_time_out) then
        tmp = 0.
      else
        tmp = relyr
      endif
      call putvars ('reltim', iou, 1, tmp, c1, c0)
      tmp = nptraj
      call putvars ('nptraj', iou, 1, tmp, c1, c0)
      ln = 3*nptraj
      ib(:) = 1
      ic(:) = ln
      call putvara ('pxyz', iou, ln, ib, ic, pxyz, c1, c0)
      call putvara ('pijk', iou, ln, ib, ic, pijk, c1, c0)
#  if defined lyapunov
      ln = 2*2*nptraj
      call putvara ('em', iou, ln, ib, ic, em, c1, c0)
#  endif
# endif
      call rdstmp (lstamp, nyear, nmonth, nday, nhour, nmin, nsec)
      tmp = nyear
      call putvars ('year', iou, 1, tmp, c1, c0)
      tmp = nmonth
      call putvars ('month', iou, 1, tmp, c1, c0)
      tmp = nday
      call putvars ('day', iou, 1, tmp, c1, c0)
      tmp = nhour
      call putvars ('hour', iou, 1, tmp, c1, c0)
      tmp = nmin
      call putvars ('minute', iou, 1, tmp, c1, c0)
      tmp = nsec
      call putvars ('second', iou, 1, tmp, c1, c0)

!-----------------------------------------------------------------------
!     write 1d data (x, y or z)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ig
      tmpi(igs:ige) = xt(igs:ige)
      call putvara ('xt', iou, ig, ib, ic, tmpi, c1, c0)
      tmpi(igs:ige) = xu(igs:ige)
      call putvara ('xu', iou, ig, ib, ic, tmpi, c1, c0)

      ic(1) = jg
      tmpj(jgs:jge) = yt(jgs:jge)
      call putvara ('yt', iou, jg, ib, ic, tmpj, c1, c0)
      tmpj(jgs:jge) = yu(jgs:jge)
      call putvara ('yu', iou, jg, ib, ic, tmpj, c1, c0)

      ic(1) = kg
      tmpk(kgs:kge) = zt(kgs:kge)
      call putvara ('zt', iou, kg, ib, ic, tmpk, c100, c0)
      tmpk(kgs:kge) = zw(kgs:kge)
      call putvara ('zw', iou, kg, ib, ic, tmpk, c100, c0)

      ic(1) = ig + 1
      call edge_maker (1, xt_e, xt, dxt, xu, dxu, imt)
      tmpie(igs:ige+1) = xt_e(igs:ige+1)
      call putvara ('xt_edges', iou, ig+1, ib, ic, tmpie, c1, c0)
      call edge_maker (2, xu_e, xt, dxt, xu, dxu, imt)
      tmpie(igs:ige+1) = xu_e(igs:ige+1)
      call putvara ('xu_edges', iou, ig+1, ib, ic, tmpie, c1, c0)

      ic(1) = jg + 1
      call edge_maker (1, yt_e, yt, dyt, yu, dyu, jmt)
      tmpje(jgs:jge+1) = yt_e(jgs:jge+1)
      call putvara ('yt_edges', iou, jg+1, ib, ic, tmpje, c1, c0)
      call edge_maker (2, yu_e, yt, dyt, yu, dyu, jmt)
      tmpje(jgs:jge+1) = yu_e(jgs:jge+1)
      call putvara ('yu_edges', iou, jg+1, ib, ic, tmpje, c1, c0)

      ic(1) = kg + 1
      call edge_maker (1, zt_e, zt, dzt, zw, dzw, km)
      tmpke(kgs:kge+1) = zt_e(kgs:kge+1)
      call putvara ('zt_edges', iou, kg+1, ib, ic, tmpke, c100, c0)
      call edge_maker (2, zw_e, zt, dzt, zw, dzw, km)
      tmpke(kgs:kge+1) = zw_e(kgs:kge+1)
      call putvara ('zw_edges', iou, kg+1, ib, ic, tmpke, c100, c0)

!-----------------------------------------------------------------------
!     write 2d data (x,y)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ile-ils+1
      ib(2) = 1
      ic(2) = jle-jls+1
      ln = ic(1)*ic(2)
      tmpij(ils:ile,jls:jle) = kmt(ils:ile,jls:jle)
      call putvara ('kmt', iou, ln, ib, ic, tmpij, c1, c0)

!-----------------------------------------------------------------------
!     write 3d data (x,y,t)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ile-ils+1
      ib(2) = 1
      ic(2) = jle-jls+1
      ib(3) = ntrec
      ic(3) = 1
      ln = ic(1)*ic(2)*ic(3)
# if defined rigid_lid_surface_pressure || defined implicit_free_surface
      tmpij(ils:ile,jls:jle) = ps(ils:ile,jls:jle,1)
      call putvara ('ps1', iou, ln, ib, ic, tmpij, c1, c0)
      tmpij(ils:ile,jls:jle) = ps(ils:ile,jls:jle,2)
      call putvara ('ps2', iou, ln, ib, ic, tmpij, c1, c0)
!     (ubarm, ubar) is at (tau, tau+1)
      tmpij(ils:ile,jls:jle) = ubarm1(ils:ile,jls:jle,1)
      call putvara ('ubarm1', iou, ln, ib, ic, tmpij, c1, c0)
      tmpij(ils:ile,jls:jle) = ubarm1(ils:ile,jls:jle,2)
      call putvara ('ubarm2', iou, ln, ib, ic, tmpij, c1, c0)
      tmpij(ils:ile,jls:jle) = ubar(ils:ile,jls:jle,1)
      call putvara ('ubar1', iou, ln, ib, ic, tmpij, c1, c0)
      tmpij(ils:ile,jls:jle) = ubar(ils:ile,jls:jle,2)
      call putvara ('ubar2', iou, ln, ib, ic, tmpij, c1, c0)
# endif
# if defined stream_function
!     first do psi at "tau" then at "tau+1"
      tmpij(ils:ile,jls:jle) = psi(ils:ile,jls:jle,1)
      call putvara ('psi1', iou, ln, ib, ic, tmpij, c1, c0)
      tmpij(ils:ile,jls:jle) = psi(ils:ile,jls:jle,2)
      call putvara ('psi2', iou, ln, ib, ic, tmpij, c1, c0)
!     guess fields
      call oget (kflds, nwds, nkflds-1, ptd)
      tmpij(ils:ile,jls:jle) = ptd(ils:ile,jls:jle)
      call putvara ('ptd1', iou, ln, ib, ic, tmpij, c1, c0)
      call oget (kflds, nwds, nkflds, ptd)
      tmpij(ils:ile,jls:jle) =  ptd(ils:ile,jls:jle)
      call putvara ('ptd2', iou, ln, ib, ic, tmpij, c1, c0)
# endif

!-----------------------------------------------------------------------
!     write 4d data (x,y,z,t)
!-----------------------------------------------------------------------
      ib(1) = 1
      ic(1) = ile-ils+1
      ib(2) = 1
      ic(2) = 1
      ib(3) = 1
      ic(3) = kle-kls+1
      ib(4) = 1
      ic(4) = 1
      ln = ic(1)*ic(2)*ic(3)*ic(4)
# if defined coarse_grained_parallelism
      taum1disk = mod(itt+2,3) + 1
      taudisk   = mod(itt  ,3) + 1
      taup1disk = mod(itt+1,3) + 1
# endif

!     save the "tau" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
!         remove external mode from "tau". since psi has been updated
!         psi(,,2) is at "tau"
          if (jrow .lt. jmt) then
# if defined stream_function
            do i=2,imt-1
              diag1 = psi(i+1,jrow+1,2) - psi(i  ,jrow,2)
              diag0 = psi(i  ,jrow+1,2) - psi(i+1,jrow,2)
              ext(i,1) = -(diag1+diag0)*dyu2r(jrow)*hr(i,jrow)
              ext(i,2) =  (diag1-diag0)*dxu2r(i)*hr(i,jrow)*csur(jrow)
             enddo
            do k=1,km
              do i=2,imt-1
                if (k .le. kmu(i,jrow)) then
                  bufsl(i,k,1) = (u(i,k,j,1,tau) - ext(i,1))
                  bufsl(i,k,2) = (u(i,k,j,2,tau) - ext(i,2))
                else
                  bufsl(i,k,1) = c0
                  bufsl(i,k,2) = c0
                endif
              enddo
            enddo
# endif
# if defined rigid_lid_surface_pressure || defined implicit_free_surface
!           since external mode has been updated, ubarm1 is at "tau"
            do k=1,km
              do i=2,imt-1
                if (k .le. kmu(i,jrow)) then
                  bufsl(i,k,1) = (u(i,k,j,1,tau) - ubarm1(i,jrow,1))
                  bufsl(i,k,2) = (u(i,k,j,2,tau) - ubarm1(i,jrow,2))
                else
                  bufsl(i,k,1) = c0
                  bufsl(i,k,2) = c0
                endif
              enddo
            enddo
# endif
            call setbcx (bufsl(1,1,1), imt, km)
            call setbcx (bufsl(1,1,2), imt, km)
          else
            do k=1,km
              do i=1,imt
                bufsl(i,k,1) = c0
                bufsl(i,k,2) = c0
              enddo
            enddo
          endif
        else
          j = jmw
          call getrow (latdisk(taudisk), nslab, jrow, u(1,1,j,1,tau)
     &,                                               t(1,1,j,1,tau))
          do k=1,km
            do i=1,imt
              bufsl(i,k,1) = u(i,k,j,1,tau)
              bufsl(i,k,2) = u(i,k,j,2,tau)
            enddo
          enddo
        endif

        ib(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          tmpik(ils:ile,kls:kle) = t(ils:ile,kls:kle,j,n,tau)
          call putvara('tracer1_'//trim(a3), iou, ln, ib, ic
     &,     tmpik, c1, c0)
        enddo
        tmpik(ils:ile,kls:kle) = bufsl(ils:ile,kls:kle,1)
        call putvara('u1', iou, ln, ib, ic, tmpik, c1, c0)
        tmpik(ils:ile,kls:kle) = bufsl(ils:ile,kls:kle,2)
        call putvara('v1', iou, ln, ib, ic, tmpik, c1, c0)
      enddo

!     save the "tau+1" latitude rows
      do jrow=1,jmt
        if (wide_open_mw) then
          j = jrow
        else
          j = jmw
          call getrow (latdisk(taup1disk), nslab, jrow
     &,                u(1,1,j,1,taup1), t(1,1,j,1,taup1))
        endif
        ib(2) = jrow
        do n=1,nt
          if (n .lt. 1000) write(a3, '(i3)') n
          if (n .lt. 100) write(a3, '(i2)') n
          if (n .lt. 10) write(a3, '(i1)') n
          tmpik(ils:ile,kls:kle) = t(ils:ile,kls:kle,j,n,taup1)
          call putvara('tracer2_'//trim(a3), iou, ln, ib, ic
     &,     tmpik, c1, c0)
        enddo
        tmpik(ils:ile,kls:kle) = u(ils:ile,kls:kle,j,1,taup1)
        call putvara('u2', iou, ln, ib, ic, tmpik, c1, c0)
        tmpik(ils:ile,kls:kle) = u(ils:ile,kls:kle,j,2,taup1)
        call putvara('v2', iou, ln, ib, ic, tmpik, c1, c0)
      enddo

!-----------------------------------------------------------------------
!     close the file
!-----------------------------------------------------------------------
      print*, '=> Ocn restart written to ',trim(fname),' on ', lstamp

      deallocate ( tmpij )
      deallocate ( tmpik )
      deallocate ( tmpi )
      deallocate ( tmpj )
      deallocate ( tmpk )
      deallocate ( tmpie )
      deallocate ( tmpje )
      deallocate ( tmpke )

      call closefile (iou)
#endif

      return
      end
