      subroutine npzd_src
#if defined uvic_npzd
     $                   (bioin,depth,theta,temp,dnswr,impo,dtts
     $                   ,dzt,dayfrac,wwd,nud,dndt
     $                   ,expoout,nppout,grazout,morpout,morptout
     $                   ,morzout,remiout,excrout
# if defined uvic_nitrogen
     $                   ,npp_Dout,graz_Dout,morp_Dout
# endif
     $                    )
!=======================================================================
!     computes source terms of the NPZD model
!     initial version of code adapted from Xavier Giraud:
!     Giraud et al. 2000, J Mar Res, 58, 609-630
!     original model reference:
!     Oeschlies and Garcon 1999, Global Biogeochem. Cycles 13, 135-160
!     Schmittner et al. 2005,  Global Biogeochem. Cycles 19, GB3004,
!     doi:10.1029/2004GB002283.
!     This is the optimized model version of Schartau & Oschlies, 2003,
!     J. Mar. Res. 62, 765-793 and Oschlies & Schartau, in prep.
!     as described in Schmittner and Oschlies, in prep.
!     note that nutrient "N" now represents phosphate
!     Andreas Schmittner Sept 2005 (aschmitt@coas.oregonstate.edu)

!     input variables:

!         bioin(1:4) = N,P,Z,D [mmol/m^3]
# if defined uvic_nitrogen
!          bioin(5)  = nitrate [mmol/m^3]
!          bioin(6)  = diazotrophs [mmol/m^3]
# endif
!              depth = depth of grid box boundaries [cm]
!              theta = pi/2 minus angle of incidence at noon [radian]
!               temp = temperature [deg C]
!              dnswr = downward shortwave radiation [W/m^2]
!               impo = import of detritus from above [mmol/(m^3 s)]
!               dtts = tracer time step [s]
!                dzt = depth of grid box [cm]
!            dayfrac = day length (fraction: 0 < dayfrac < 1)
!                wwd = sinking speed of detritus [cm/s]
!                nud = remineralisation rate of detritus [1/s]

!     output variables:

!               dndt = source terms [mmol/(m^3 s)]
!               nppout  = net primary production [mmol/(m^3 s)]
!               grazout = grazing [mmol/(m^3 s)]
!               morpout = quadratic mortality of phytoplankton [mmol/(m^3 s)]
!               morptout = specific mortality of phytoplankton [mmol/(m^3 s)]
!               morzout = mortality of zooplankton [mmol/(m^3 s)]
!               remiout = remineralisation [mmol/(m^3 s)]
!               excrout = excretion [mmol/(m^3 s)]
!               expoout = detrital export [mmol/(m^3 s)]
# if defined uvic_nitrogen
!               npp_Dout = NPP of diazotrophs
!               graz_Dout = grazing of diazotrophs
!               morp_Dout = mortality of diazotrophs
# endif
!=======================================================================
!      implicit none
      include "param.h"
      include "calendar.h"
      include "npzd.h"
      integer n,nint,l
      real   bioin(ntnpzd),dndt(ntnpzd)
      real   depth,theta,temp,dnswr,rexp
      real   bion,biop,bioz,biod
      real   light,avej,jmax,u_P,g_P
      real   npp,graz,morp,morpt,morz,remi,excr,expo,impo
      real   nppout,grazout,morpout,morptout,morzout,remiout,excrout
      real   expoout
      real   dtbio,dtts,dzt,costhe
      real   nflag,pflag,zflag,dflag,wwd,gl,gd,dayfrac
      real   bct,nupt,nud
# if defined uvic_nitrogen
      real   biono3,u_D,npp_D,npp_Dout,no3flag
      real   biodiaz,diazflag,g_D,graz_D
      real   morp_D,jmax_D,gd_D,avej_D,no3upt_D
      real   morp_Dout,graz_Dout
# endif

      trcmin = 5e-12
!     # of integrations per tracer time step
      nint = dtts/dtnpzd ! dtnpzd is time step in seconds
      dtbio = dtts/nint   ! time step [s]

      costhe = sqrt(1.-(1.-cos(theta)**2.)/1.33**2.) ! refraction index = 1.33
!     calculate light at upper grid box boundary z = depth + dzt/2
      light = dnswr*par*EXP((kw*(depth+dzt/2.))/costhe)
      bct = bbio**(cbio*temp)
      jmax = abio*bct
      nupt = nupt0*bct
!     photosynthesis after Evans & Parslow (1985)
!     notation as in JGOFS report No. 23 p. 6
      gd = jmax*dayfrac
      gl = alpha*light
      avej = gd/(kw*dzt)*(phi(2.*gl/gd)
     &                   -phi(2.*gl/gd*exp((-kw-kc*bioin(2))*dzt)))
# if defined uvic_nitrogen
      jmax_D = max(0.,abio*(bct-2.6))*jdiar
      gd_D = max(1.e-14,jmax_D*dayfrac)
      avej_D = gd_D/(kw*dzt)*(phi(2.*gl/gd_D)
     &                   -phi(2.*gl/gd_D*exp((-kw-kc*bioin(2))*dzt)))
# endif
      bion =  bioin(1)
      biop =  bioin(2)
      bioz =  bioin(3)
      biod =  bioin(4)
      do n=1,ntnpzd
         dndt(n)=0.0
      enddo
# if defined uvic_nitrogen
      biono3 = bioin(5)
      biodiaz = bioin(6)
      npp_Dout = 0.0
      graz_Dout = 0.0
      morp_Dout = 0.0
# endif
      nppout = 0.0
      grazout = 0.0
      morpout = 0.0
      morptout = 0.0
      morzout = 0.0
      remiout = 0.0
      excrout = 0.0
      expoout = 0.0
      do l=1,nint
!     growth rate of phytoplankton
        u_P = min(avej,jmax*bion/(k1p+bion))
# if defined uvic_nitrogen
        u_P = min(u_P,jmax*biono3/(k1+biono3)) ! nitrate limitation
        u_D = min(avej_D,jmax_D*bion/(k1p+bion)) ! growth rate of
                            ! diazotrophs
                            ! smaller than other phytoplankton and not
                            ! nitrate limited
# endif
        npp = u_P*biop
# if defined uvic_nitrogen
        npp_D = max(0.,u_D*biodiaz)
!    grazing function for diazotrophs
        g_D = gbio*epsbio*biodiaz*biodiaz
     &          /(gbio+epsbio*biodiaz*biodiaz)
        graz_D = g_D*bioz
        morp_D = nupt*biodiaz ! linear mortality
        no3upt_D = biono3/(k1+biono3)*npp_D ! nitrate uptake
# endif
!    grazing function
        g_P = gbio*epsbio*biop*biop/(gbio+epsbio*biop*biop)
        graz = g_P*bioz
        morp = nup*biop*biop
        morpt = nupt*biop
        morz = nuz*bioz*bioz
        remi = nud*bct*biod
        excr = gamma2*bct*bioz
        expo = wwd*biod/dzt
!    flags prevent negative values by setting outgoing fluxes to
!    zero if tracers are lower than trcmin
        nflag = 0.5+sign(0.5,bion-trcmin)
        pflag = 0.5+sign(0.5,biop-trcmin)
        zflag = 0.5+sign(0.5,bioz-trcmin)
        dflag = 0.5+sign(0.5,biod-trcmin)
# if defined uvic_nitrogen
        no3flag = 0.5+sign(0.5,biono3-trcmin)
        diazflag = 0.5+sign(0.5,biodiaz-trcmin)
# endif
        npp = npp*nflag
# if defined uvic_nitrogen
     $        *no3flag
        npp_D = npp_D*nflag
        graz_D = graz_D*diazflag
        morp_D = morp_D*diazflag
        no3upt_D = no3upt_D*no3flag
        npp_Dout = npp_Dout + npp_D
        graz_Dout = graz_Dout + graz_D
        morp_Dout = morp_Dout + morp_D
# endif
        graz = graz*pflag
        morp = morp*pflag
        morpt = morpt*pflag
        morz = morz*zflag
        remi = remi*dflag
        excr = excr*zflag
        expo = expo*dflag
        expoout = expoout + expo
        nppout = nppout + npp
        grazout = grazout + graz
        morpout = morpout + morp
        morptout = morptout + morpt
        morzout = morzout + morz
        remiout = remiout + remi
        excrout = excrout + excr

!  !!!!!!!!!!!!!!!!!  nutrients equation
        dndt(1) = dndt(1) + remi + excr - npp + morpt
# if defined uvic_nitrogen
     1            - npp_D
!  !!!!!!!!!!!!!!!!!  nitrate (NO3) equation
        dndt(5) = dndt(5) + remi + excr - npp + morpt
     &              - no3upt_D
!  !!!!!!!!!!!!!!!!!  diazotroph equation
        dndt(6) = dndt(6) + npp_D - morp_D - graz_D
# endif
!  !!!!!!!!!!!!!!!!!  phytoplankton equation
        dndt(2) = dndt(2) + npp - morp - graz - morpt
!  !!!!!!!!!!!!!!!!!  zooplankton equation
        dndt(3) = dndt(3) + gamma1*graz - excr - morz
# if defined uvic_nitrogen
     &                    + gamma1*graz_D
# endif
!  !!!!!!!!!!!!!!!!!  detritus equation
        dndt(4) = dndt(4) + (1.-gamma1)*graz + morp
# if defined uvic_nitrogen
     &              + (1.-gamma1)*graz_D + morp_D
# endif
     1              + morz - remi
     2              - expo + impo
        bion = bion + dtbio*redptn*(remi + excr - npp + morpt)
# if defined uvic_nitrogen
     &              - dtbio*redptn*npp_D
        biono3 = biono3 + dtbio*(remi + excr - npp + morpt
     &              - no3upt_D)

        biodiaz = biodiaz + dtbio*(npp_D - morp_D - graz_D
     &                             - morpt)
# endif
        biop = biop + dtbio*(npp - morp - graz - morpt)
        bioz = bioz + dtbio*(gamma1*graz - excr - morz)
# if defined uvic_nitrogen
     &              + dtbio*gamma1*graz_D
# endif
        biod = biod + dtbio*((1.-gamma1)*graz + morp
# if defined uvic_nitrogen
     &              + (1.-gamma1)*graz_D + morp_D
# endif
     1              + morz - remi
     2              - expo + impo)
      enddo
      do n=1,ntnpzd
         dndt(n)=dndt(n)/nint
      enddo
# if defined uvic_nitrogen
      npp_Dout = npp_Dout/nint
      graz_Dout = graz_Dout/nint
      morp_Dout = morp_Dout/nint
# endif
      expoout = expoout/nint
      nppout = nppout/nint
      remiout = remiout/nint
      excrout = excrout/nint
      grazout = grazout/nint
      morpout = morpout/nint
      morptout = morptout/nint
      morzout = morzout/nint
      return

      contains
       function phi(u)
       real phi,u
       phi = (0.555588*u+0.04926*u**2)/(1.+0.188721*u)
       end function phi
#endif
      end
