      program UVic_ESCM

!=======================================================================

!     UNIVERSITY OF VICTORIA EARTH SYSTEM CLIMATE MODEL

!     A climate model developed by researchers in the Climate Research
!     Group, in the School of Earth and Ocean Sciences, located at the
!     University of Victoria, Victoria, B.C., Canada.

!     Many people have contributed to the development of this code
!     and attempts are made to indicate originators of code where
!     possible or appropriate. Please direct problems or questions
!     to the code contact person at:
!     http://climate.uvic.ca/climate-lab/model

!     Requirements:

!     Standard fortran 90 is used

!     Disclaimer:

!     The UVic Earth System Climate Model (UVic_ESCM) is a climate
!     modeling research tool developed at the University of Victoria.
!     Others may use it freely but we assume no responsibility for
!     problems or incorrect use. It is left to the user to ensure that
!     a particular configuration is working correctly.
!=======================================================================

!     This is the main driver. Integration time is divided into a number
!     of equal time segments and SBC are held fixed for each time
!     segment. When coupling, SBC are supplied each time segment (the
!     coupling period) and held fixed for that period.

!     based on code by: R. C. Pacanowski, A. Rosati and M. Eby
!=======================================================================

#include "param.h"
#if defined uvic_embm
# include "atm.h"
# include "cembm.h"
#endif
#include "coord.h"
#include "csbc.h"
#include "iounit.h"
#include "levind.h"
#if defined uvic_mom
# include "mw.h"
# include "task_on.h"
#endif
#include "scalar.h"
#include "switch.h"
#include "tmngr.h"
#if defined ubc_cidm
# include "ice.h"
# include "subgrid.h"
# include "calendar.h"
      real dtcidm
#endif

      logical   mk_out
      namelist /uvic/ mk_out

      print*, '== UNIVERSITY OF VICTORIA EARTH SYSTEM CLIMATE MODEL =='
      print*, '                                            '

#if defined uvic_mk_out
!-----------------------------------------------------------------------
!     mk options
!-----------------------------------------------------------------------

      mk_out = .false.
      call getunit (ioun, 'control.in', 'f s r')
      read  (ioun, uvic, end=100)
100   continue
      call relunit (ioun)
      if (mk_out) then
!       print mk options
        print*, ' '
        print*, 'Mk Options are:'
        include "uvic_mk.h"
        print*, ' '
      endif

#endif
!-----------------------------------------------------------------------
!     initialize i/o units
!-----------------------------------------------------------------------

      call ioinit

!-----------------------------------------------------------------------
!     setup file renaming
!-----------------------------------------------------------------------

      call file_names

!-----------------------------------------------------------------------
!     Initialize S.B.C. indices
!-----------------------------------------------------------------------

      do n=1,numsbc
        sbc(:,:,n) = 0.0
        mapsbc(n) = " "
      enddo

      m = 1
      call set (itaux, m, mapsbc(m), 'taux', m)
      call set (itauy, m, mapsbc(m), 'tauy', m)
      call set (iws, m, mapsbc(m), 'ws', m)
      call set (iaca, m, mapsbc(m), 'a_calb', m)
      call set (isca, m, mapsbc(m), 's_calb', m)
      call set (ihflx, m, mapsbc(m), 'hflx', m)
      call set (isflx, m, mapsbc(m), 'sflx', m)
      call set (isst, m, mapsbc(m), 'sst', m)
      call set (isss, m, mapsbc(m), 'sss', m)
#if defined uvic_embm_awind && defined uvic_embm
      call set (iwa, m, mapsbc(m), 'wa', m)
#endif
#if defined uvic_embm_adv_q && defined uvic_embm
      call set (iwxq, m, mapsbc(m), 'wx_q', m)
      call set (iwyq, m, mapsbc(m), 'wy_q', m)
#endif
#if defined uvic_embm_adv_t && defined uvic_embm
      call set (iwxt, m, mapsbc(m), 'wx_t', m)
      call set (iwyt, m, mapsbc(m), 'wy_t', m)
#endif
      call set (iro, m, mapsbc(m), 'ro', m)
#if defined shortwave
      call set (ipsw, m, mapsbc(m), 'psw', m)
#endif
#if defined uvic_ice_evp
      call set (isu, m, mapsbc(m), 'su', m)
      call set (isv, m, mapsbc(m), 'sv', m)
      call set (igu, m, mapsbc(m), 'gu', m)
      call set (igv, m, mapsbc(m), 'gv', m)
#endif
#if defined uvic_carbon
      call set (issc, m, mapsbc(m), 'ssc', m)
      call set (icflx, m, mapsbc(m), 'cflx', m)
#endif
#if defined uvic_alk
      call set (issa, m, mapsbc(m), 'ssa', m)
#endif
#if defined uvic_c14
      call set (issc14, m, mapsbc(m), 'ssc14', m)
      call set (ic14flx, m, mapsbc(m), 'c14flx', m)
#endif
#if defined uvic_cfc11
      call set (isscfc11, m, mapsbc(m), 'sscfc11', m)
      call set (icfc11flx, m, mapsbc(m), 'cfc11flx', m)
#endif
#if defined uvic_cfc12
      call set (isscfc12, m, mapsbc(m), 'sscfc12', m)
      call set (icfc12flx, m, mapsbc(m), 'cfc12flx', m)
#endif
#if defined uvic_o2
      call set (isso2, m, mapsbc(m), 'sso2', m)
      call set (io2flx, m, mapsbc(m), 'o2flx', m)
#endif
#if defined uvic_mtlm
      call set (iat, m, mapsbc(m), 'at', m)
      call set (irh, m, mapsbc(m), 'rh', m)
      call set (ipr, m, mapsbc(m), 'pr', m)
      call set (ips, m, mapsbc(m), 'ps', m)
      call set (iaws, m, mapsbc(m), 'aws', m)
      call set (iswr, m, mapsbc(m), 'swr', m)
      call set (ilwr, m, mapsbc(m), 'lwr', m)
      call set (isens, m, mapsbc(m), 'sens', m)
      call set (ievap, m, mapsbc(m), 'evap', m)
      call set (idtr, m, mapsbc(m), 'dtr', m)
#endif
#if defined uvic_mtlm && defined uvic_carbon
      call set (isr, m, mapsbc(m), 'sr', m)
      call set (inpp, m, mapsbc(m), 'npp', m)
#endif

      if ( m-1 .gt. numsbc) then
        print*, '==> Error: increase numsbc in csbc.h to ', m-1
        stop '=>UVic_ESCM'
      endif

#if defined uvic_mom
!-----------------------------------------------------------------------
!     Initialize ocean tracer names
!-----------------------------------------------------------------------

      do n=1,nt
        mapt(n) = " "
      enddo

      m = 1
      call set (itemp, m, mapt(m), 'temp', m)
      call set (isalt, m, mapt(m), 'salt', m)
# if defined uvic_carbon
      call set (idic, m, mapt(m), 'dic', m)
# endif
# if defined uvic_c14
      call set (ic14, m, mapt(m), 'c14', m)
# endif
# if defined uvic_cfc11
      call set (icfc11, m, mapt(m), 'cfc11', m)
# endif
# if defined uvic_cfc12
      call set (icfc12, m, mapt(m), 'cfc12', m)
# endif
# if defined uvic_o2
      call set (io2, m, mapt(m), 'o2', m)
# endif
# if defined uvic_alk
      call set (ialk, m, mapt(m), 'alk', m)
# endif
# if defined uvic_npzd
      call set (inutr, m, mapt(m), 'nutr', m)
#  if defined uvic_nitrogen
      call set (ino3, m, mapt(m), 'no3', m)
#  endif
#  if !defined uvic_npzd_simple
      call set (iphyt, m, mapt(m), 'phyt', m)
      call set (izoop, m, mapt(m), 'zoop', m)
      call set (idetr, m, mapt(m), 'detr', m)
#   if defined uvic_nitrogen
      call set (idiaz, m, mapt(m), 'diaz', m)
#   endif
#  endif
# endif
      if ( m-1 .gt. nt) then
        print*, '==> Error: increase nt for tracers in size.h'
        stop '=>UVic_ESCM'
      endif

!-----------------------------------------------------------------------
!     Initialize ocean source tracer names, must have equivalent tracer
!-----------------------------------------------------------------------

      do n=1,nt
        mapst(n) = " "
        itrc(n) = 0
      enddo

      m = 1
# if defined uvic_carbon
      call set (isdic, m, mapst(m), 'sdic', m)
      itrc(idic) = m-1
# endif
# if defined uvic_c14
      call set (isc14, m, mapst(m), 'sc14', m)
      itrc(ic14) = m-1
# endif
# if defined uvic_o2
      call set (iso2, m, mapst(m), 'so2', m)
      itrc(io2) = m-1
# endif
# if defined uvic_alk
      call set (isalk, m, mapst(m), 'salk', m)
      itrc(ialk) = m-1
# endif
# if defined uvic_npzd
      call set (isnutr, m, mapst(m), 'snutr', m)
      itrc(inutr) = m-1
#  if defined uvic_nitrogen
      call set (isno3, m, mapst(m), 'sno3', m)
      itrc(ino3) = m-1
#  endif
#  if !defined uvic_npzd_simple
      call set (isphyt, m, mapst(m), 'sphyt', m)
      itrc(iphyt) = m-1
      call set (iszoop, m, mapst(m), 'szoop', m)
      itrc(izoop) = m-1
      call set (isdetr, m, mapst(m), 'sdetr', m)
      itrc(idetr) = m-1
#   if defined uvic_nitrogen
      call set (isdiaz, m, mapst(m), 'sdiaz', m)
      itrc(idiaz) = m-1
#   endif
#  endif
# endif
      if ( m-1 .gt. nt) then
        print*, '==> Error: increase nt for tracer sources in size.h'
        stop '=>UVic_ESCM'
      endif

#endif
#if defined uvic_embm
!-----------------------------------------------------------------------
!     Initialize atmosphere tracer names
!-----------------------------------------------------------------------

      do n=1,nat
        mapat(n) = " "
      enddo

      m = 1
      call set (isat, m, mapat(m), 'sat', m)
      call set (ishum, m, mapat(m), 'shum', m)
# if defined uvic_carbon && defined uvic_carbon_co2_2d
      call set (ico2, m, mapat(m), 'co2', m)
# endif

      if ( m-1 .gt. nat) then
        print*, '==> Error: increase nat in size.h'
        stop '=>UVic_ESCM'
      endif
#endif

!-----------------------------------------------------------------------
!     do the introductory ocean setup once per run
!-----------------------------------------------------------------------

      call setocn

#if defined ubc_cidm
      print*, ' option ubc_cidm is not ready for use in this version'
      stop
!-----------------------------------------------------------------------
!    do the introductory icesheet setup once per run
!-----------------------------------------------------------------------

      degrad = pi/180.
      do j=2,jmt-1
        phij = -90. - dytdeg(j)/2. + dytdeg(j)*float(j-1)
        area(j) = radius*radius*cos(phij*degrad)
     &   *(dxtdeg(1)*degrad)*(dytdeg(j)*degrad)
        if ((j.ge.jmint).and.(j.le.jmaxt)) then
          do m=1,mj
            jj = (j-jmint)*mj + m
            phijj = phij - dytdeg(j)/2. - dytdeg(j)/float(mj)/2.
     &            + dytdeg(j)/float(mj)*float(m)

            subarea(jj) = radius*radius*cos(phijj*degrad)
     &       *(dxtdeg(1)/float(mi)*degrad)*(dytdeg(j)/float(mj)*degrad)
          enddo
        endif
      enddo
      area(1)   = area(2)
      area(jmt) = area(jmt-1)
      if (jmint.eq.1) then
        do m=1,mj
          jj = (1-jmint)*mj + m
          subarea(jj) = subarea(jj+mj)
        enddo
      endif
      if (jmaxt.eq.jmt) then
        do m=1,mj
          jj = (jmt-jmint)*mj + m
          subarea(jj) = subarea(jj-mj)
        enddo
      endif

      call cidm_init (subhice, subelev, avg_calv, subaice, dtcidm)
#endif

!-----------------------------------------------------------------------
!     do the introductory atmosphere setup once per run
!-----------------------------------------------------------------------

      write (stdout,'(/a36/)') ' ==> Note: the atmos setup follows:'

!     "setatm" must do the following:
!       1) set up the atmospheric S.B.C. grid definition
!       2) define the atmosphere land/sea mask
!       3) set the atmosphere time step "dtatm" {seconds}

#if defined uvic_embm
      call setembm
#else
      call setatm

!     when the MOM S.B.C. come from a dataset, force the segment time
!     and atmospheric time step to one MOM time step. This will force
!     the number of segments to one and the number of time steps per
!     segment to represent the length of the run in days.

      dtatm = dtocn
      segtim = dtocn*secday
#endif
#if defined uvic_mtlm

!-----------------------------------------------------------------------
!     do the introductory land setup once per run
!-----------------------------------------------------------------------
      call setmtlm
#endif

!-----------------------------------------------------------------------
!     compute the number of ocean time steps "numots" for this run and
!     the number of ocean time steps per ocean segment "ntspos".
!     compute the number of atmos time steps "numats" for this run and
!     the number of atmos time steps per atmos segment "ntspas".
!     divide the integration time "days" into "numseg" segments.
!     each will be length "segtim" days. Surface boundary conditions
!     are supplied every "segtim" days.
!-----------------------------------------------------------------------

      numots = nint(rundays/(dtocn*secday))
      ntspos = nint(segtim/(dtocn*secday))
      numats = nint(rundays/(dtatm*secday))
      ntspas = nint(segtim/(dtatm*secday))
      numseg = numots/ntspos
#if defined uvic_mtlm
# if defined uvic_mtlm_segday
      if (segtim .gt. 1.) then
        ntspls = nint(c1/(dtlnd*secday))
      else
        ntspls = nint(segtim/(dtlnd*secday))
      endif
# else
      ntspls = nint(segtim/(dtlnd*secday))
# endif
#endif

#if !defined uvic_embm && defined uvic_mom
!     load the tracers (SST & SSS) for each row "j". (zero on land)
!     load from the MW if fully opened otherwise load from disk

      if (wide_open_mw) then
        do j=1,jmt
          do i=1,imt
# if !defined uvic_replacst
            if (isst .ne. 0) sbc(i,j,isst) = t(i,1,j,itemp,taup1)
            if (isss .ne. 0) sbc(i,j,isss) = t(i,1,j,isalt,taup1)
# endif
# if defined uvic_carbon
            if (issc .ne. 0) sbc(i,j,issc) = t(i,1,j,idic,taup1)
# endif
# if defined uvic_c14
            if (issc14 .ne. 0) sbc(i,j,issc14) = t(i,1,j,ic14,taup1)
# endif
# if defined uvic_o2
            if (isso2 .ne. 0) sbc(i,j,isso2) = t(i,1,j,io2,taup1)
# endif
# if defined uvic_alk
            if (issa .ne. 0) sbc(i,j,issa) = t(i,1,j,ialk,taup1)
# endif
          enddo
        enddo
      else
        do j=1,jmt
# if !defined uvic_replacst
          if (isst .ne. 0) call getst (j, sbc(1,1,isst), itemp)
          if (isss .ne. 0) call getst (j, sbc(1,1,isss), isalt)
# endif
# if defined uvic_carbon
          if (issc .ne. 0) call getst (j, sbc(1,1,issc), idic)
# endif
# if defined uvic_c14
          if (issc14 .ne. 0) call getst (j, sbc(1,1,issc14), ic14)
# endif
# if defined uvic_o2
          if (isso2 .ne. 0) call getst (j, sbc(1,1,isso2), io2)
# endif
# if defined uvic_alk
          if (issa .ne. 0) call getst (j, sbc(1,1,issa), ialk)
# endif
        enddo
      endif

#endif
!-----------------------------------------------------------------------
!     check for consistency in the S.B.C. setup
!-----------------------------------------------------------------------

      call chkcpl

#if defined ubc_cidm
!-----------------------------------------------------------------------
!     get inital conditions from the icesheet model
!-----------------------------------------------------------------------

      call cidm_sbc_allocate
      call cidm_sbc_atm (subhice, subelev)
      call cidm_sbc_ocn (avg_calv, avg_ltnt)
      call cidm_sbc_init

#endif
#if defined uvic_global_sums
!-----------------------------------------------------------------------
!     get global sums at the start of the run
!-----------------------------------------------------------------------

      dtoih = 0.
      call globalsum (1)

#endif
!-----------------------------------------------------------------------
!     S T A R T    S E G M E N T    L O O P
!-----------------------------------------------------------------------

      do n=1,numseg

!-----------------------------------------------------------------------
!       get the atmospheric S.B.C.
!-----------------------------------------------------------------------

        call gasbc (1, imt, 1, jmt)

!-----------------------------------------------------------------------
!       call the atmospheric model once for each time step until one
!       segment of "segtim" days is complete. hold atmos S.B.C. fixed
!       during each segment and predict average S.B.C. for ocean
!-----------------------------------------------------------------------

        do loop=1,ntspas
#if defined uvic_embm
          call embm (1, imt, 1, jmt)

# if !defined uvic_mom
#  if defined ubc_cidm
          call cidmout (snapts, timavgts, tsits, timunit
     &,                 expnam, relyr, stamp)
#  endif
#  if defined uvic_mtlm
          call mtlmout (1, imt, 1, jmt)
#  endif
          if (tsits .and. iotsi .ne. stdout .and. iotsi .gt. 0) then
            write (*,'(1x, a3, i7, 1x, a32)') 'ts=',itt, stamp
          endif
# elif defined ubc_cidm
!         integrate land-ice surface boundary conditions
          if (addflux)  call cidm_sbc_sum (subpsno, subprecip)
# endif
#else
          call atmos
#endif
        enddo

#if defined ubc_cidm
        if (mod(relyr,dtcidm) .lt. 1.e-6) then
          call cidm_sbc_avg
          call cidm_sbc_lice
          call cidm(0., dtcidm, subhice, subelev, avg_calv, subaice)
          call cidm_sbc_atm (subhice, subelev)
          call cidm_sbc_ocn (avg_calv, avg_ltnt)
          call cidm_sbc_init
        endif

#endif
#if defined uvic_mtlm
!-----------------------------------------------------------------------
!       get land S.B.C.s
!-----------------------------------------------------------------------

        call glsbc

!----------------------------------------------------------------------
!       call the land-surface and vegetation  model once for each time
!       step until one segment of "segtim" days is complete.
!       hold land S.B.C. fixed
!       during each segment and predict average S.B.C. for the EMBM
!-----------------------------------------------------------------------

        do loop=1,ntspls
           call mtlm
        enddo

#endif
!-----------------------------------------------------------------------
!       get ocean S.B.C.s
!-----------------------------------------------------------------------

        call gosbc

#if defined uvic_mom
!-----------------------------------------------------------------------
!       call the ocean model once for each time step until one
!       segment of "segtim" days is complete. hold ocean S.B.C. fixed
!       during each segment and predict average S.B.C. for atmos
!-----------------------------------------------------------------------

        do loop=1,ntspos
          call mom
# if defined uvic_embm
          call embmout (1, imt, 1, jmt)
# endif
# if defined ubc_cidm
          call cidmout (snapts, timavgts, tsits, timunit
     &,                 expnam, relyr, stamp)
# endif
# if defined uvic_mtlm
          call mtlmout (1, imt, 1, jmt)
# endif
          if (tsits .and. iotsi .ne. stdout .and. iotsi .gt. 0) then
            write (*,'(1x, a3, i7, 1x, a32)') 'ts=',itt, stamp
          endif
        enddo

#endif
#if defined uvic_global_sums
!-----------------------------------------------------------------------
!       write change in global sums for heat and fresh water
!-----------------------------------------------------------------------

        if (tsits) call globalsum (2)

#endif
      enddo

#if defined uvic_global_sums
!-----------------------------------------------------------------------
!     get global sums at the end of the run
!-----------------------------------------------------------------------

      call globalsum (3)

#endif
!-----------------------------------------------------------------------
!     E N D    S E G M E N T    L O O P
!-----------------------------------------------------------------------

      print*, ' ==>  UVIC_ESCM integration is complete.'

      call release_all

      stop
      end

      subroutine chkcpl
      logical errorc
#include "param.h"
#include "csbc.h"
#include "switch.h"

!-----------------------------------------------------------------------
!     do consistency checks before allowing model to continue
!-----------------------------------------------------------------------

      errorc = .false.
      write (stdout,*) ' '
      write (stdout,*) '    (checking S.B.C. setup)'

      if (dtatm .eq. c0) then
          write (stdout,9000)
     & '==> Error: the atmospheric time step must be set in "setatm"  '
          errorc = .true.
          dtatm = 1.e-6
      endif
!      critv = 1.e-6
      critv = 1.e-4
      if (segtim .ne. c0) then
        r1 = rundays/segtim
      else
        r1 = 0.5
      endif
      r2 = segtim/(dtocn*secday)
      r3 = segtim/(dtatm*secday)
      if (segtim .eq. c0) then
          write (stdout,9000)
     & '==> Error: coupling period "segtim" must be specified         '
          errorc = .true.
      elseif (abs(r1-nint(r1)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of segments       '
     &,'    "segtim"  within "rundays" (the length of the run)        '
          errorc = .true.
      elseif (abs(r2-nint(r2)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of density time   '
     &,'    steps "dtocn"  within "segtim" (the segment time)         '
          errorc = .true.
      elseif (abs(r3-nint(r3)) .gt. critv) then
          write (stdout,9000)
     & '==> Error: there must be an integral number of atmos time     '
     &,'    steps "dtatm"  within "segtim" (the segment time)         '
          errorc = .true.
      endif
#if defined uvic_mom && defined uvic_embm && defined restorst
        write (stdout,9000)
     & '==> Warning: restoring to surface tracers (restorst)          '
     &,'    with coupled model                                        '
#endif
#if defined uvic_mom && defined uvic_embm && defined uvic_replacst
        write (stdout,9000)
     & '==> Warning: replacing surface tracers (uvic_replacst)        '
     &,'    with coupled model                                        '
#endif
#if defined restorst && defined uvic_replacst
        write (stdout,9000)
     & '==> Warning: both restoring and replacing surface tracers     '
     &,'    (restorst, uvic_replacst ). Replacing takes precidence    '
#endif
      write (stdout,*) '    (End of S.B.C. checks) '
      write (stdout,*) ' '
      if (errorc) stop '=>chkcpl'

9000  format (/,(1x,a80))
      return
      end

      subroutine set (index, num, name, text, inc)

!-----------------------------------------------------------------------
!     increment counter, set index and text
!-----------------------------------------------------------------------

      character(*) :: name, text

      name = text
      index = num
      inc = index + 1

      return
      end

      subroutine getst (jrow, ocnout, ntabc)

#if defined uvic_mom
!-----------------------------------------------------------------------
!     read surface tracers from disk row "jrow"
!-----------------------------------------------------------------------

# include "param.h"
# include "iounit.h"
# include "mw.h"
# include "tmngr.h"

      dimension ocnout(imt,jmt)

      call getrow (latdisk(taup1disk), nslab, jrow
     &,          u(1,1,jmw,1,taup1), t(1,1,jmw,1,taup1))
      do i=1,imt
        ocnout(i,jrow) = t(i,1,jmw,ntabc,taup1)
      enddo
#endif

      return
      end
