      subroutine npzd_src (biota, depth, theta, temp, dnswr, impo, dtts
     &,                    dzt, dayfrac, dndt, expoout, nppout, remiout
     &,                    excrout, grazout, morpout, morzout)
!=======================================================================
!     computes source terms of the NPZD model
!     code adapted from Xavier Giraud:
!     Giraud et al. 2000, J Mar Res, 58, 609-630
!     original model reference:
!     Oeschlies and Garcon 1999, Global Biogeochem. Cycles 13, 135-160

!     input variables:

!         biota(1:4) = N,P,Z,D [mmol/m^3]
!              depth = depth of grid box boundaries [m]
!              theta = pi/2 minus angle of incidence at noon [radian]
!               temp = temperature [deg C]
!              dnswr = downward shortwave radiation [W/m^2]
!               impo = import of detritus from above [mmol/(m^3 s)]
!               dtts = tracer time step [s]
!                dzt = depth of grid box [m]
!            dayfrac = day length (fraction: 0 < dayfrac < 1)

!     output variables:

!               dndt = source terms [mmol/(m^3 s)]
!               impo = import of detritus from above [mmol/(m^3 s)]
!               nppout  = net primary production [mmol/(m^3 s)]
!               grazout = grazing [mmol/(m^3 s)]
!               morpout = mortality of phytoplankton [mmol/(m^3 s)]
!               morzout = mortality of zooplankton [mmol/(m^3 s)]
!               remiout = remineralisation [mmol/(m^3 s)]
!               excrout = excretion [mmol/(m^3 s)]
!               expoout = detrital export [mmol/(m^3 s)]

!     based on code by: A. Schmittner
!=======================================================================
#if defined uvic_npzd

      implicit none

      include "param.h"
      include "calendar.h"
      include "npzd.h"

      integer n, nint, l
      real biota(4), dndt(4)
      real depth, theta, temp, dnswr, rexp
      real bion, biop, bioz, biod
      real light, avej, jmax, u_npz, g_npz
      real npp, graz, morp, morz, remi, excr, expo, impo
      real nppout, grazout, morpout, morzout, remiout, excrout, expoout
      real dtbio, dtts, dzt, costhe
      real nflag, pflag, zflag, dflag, wwd, dayfrac, gl, gd, trcmin5

!     # of integrations per tracer time step
      nint = dtts/(dtnpzd*3600)     ! dtnpzd is time step in hours
      dtbio = dtts/(nint*daylen)    ! time step [days]

      costhe = sqrt(1.-(1.-cos(theta)**2.)/1.33**2.) ! refraction index = 1.33
!     calculate light at upper grid box boundary z = depth + dzt/2
      light = dnswr*par*EXP((kw*(depth+dzt/2.))/costhe)
      jmax = abio*bbio**(cbio*temp)
!     photosynthesis after Evans & Parslow (1985)
!     notation as in JGOFS report No. 23 p. 6
      gd = jmax*dayfrac
      gl = alpha*light
      avej = gd/(kw*dzt)*(phi(2.*gl/gd)
     &     - phi(2.*gl/gd*exp((-kw-kc*biota(2))*dzt)))
      impo = impo*daylen ! convert to [mmol/(m^3 day)]
      wwd = wd  !*(1.-1.8e-3*depth) ! linear increase with depth
      trcmin5 = trcmin*5.
      bion = biota(1)
      biop = biota(2)
      bioz = biota(3)
      biod = biota(4)
      dndt(:)=0.0
      nppout = 0.0
      grazout = 0.0
      morpout = 0.0
      morzout = 0.0
      remiout = 0.0
      excrout = 0.0
      expoout = 0.0

      do l=1,nint
!    growth rate
        u_npz = min(avej,bion/(k1+bion))
!       u_npz = min(u_npz,LFe) ! iron limitation
        u_npz = jmax * u_npz
!    grazing function
        g_npz = gbio*epsbio*biop*biop/(gbio+epsbio*biop*biop)
        npp = u_npz*biop
        graz = g_npz*bioz
        morp = nup*biop
        morz = nuz*bioz*bioz
        remi = nud*biod
        excr = gamma2*bioz
        expo = wwd*biod/dzt
!    flags prevent negative values by setting outgoing fluxes to
!    zero if negative values would occur in next time step
        nflag = 0.5 + sign(0.5,bion-trcmin5)
        pflag = 0.5 + sign(0.5,biop-trcmin5)
        zflag = 0.5 + sign(0.5,bioz-trcmin5)
        dflag = 0.5 + sign(0.5,biod-trcmin5)
        npp = npp*nflag
        graz = graz*pflag
        morp = morp*pflag
        morz = morz*zflag
        remi = remi*dflag
        excr = excr*zflag
        expo = expo*dflag
        expoout = expoout + expo
        nppout = nppout + npp
        grazout = grazout + graz
        morpout = morpout + morp
        morzout = morzout + morz
        remiout = remiout + remi
        excrout = excrout + excr
!       nutrients equation
        dndt(1) = dndt(1) + remi + excr - npp
!       phytoplankton equation
        dndt(2) = dndt(2) + npp - morp - graz
!       zooplankton equation
        dndt(3) = dndt(3) + gamma1*graz - excr - morz
!       detritus equation
        dndt(4) = dndt(4) + (1.-gamma1)*graz + morp
     &          + morz - remi - expo + impo
        bion = bion + dtbio*(remi + excr - npp)
        biop = biop + dtbio*(npp - morp - graz)
        bioz = bioz + dtbio*(gamma1*graz - excr - morz)
        biod = biod + dtbio*((1.-gamma1)*graz + morp
     &       + morz - remi - expo + impo)
      enddo

!     convert from [mmol/(m^3 day)] to [mmol/(m^3 s)]
      do n=1,4
         dndt(n) = dndt(n)/(daylen*nint)
      enddo
      impo = impo/daylen
      expoout = expoout/(daylen*nint)
      nppout = nppout/(daylen*nint)
      remiout = remiout/(daylen*nint)
      excrout = excrout/(daylen*nint)
      grazout = grazout/(daylen*nint)
      morpout = morpout/(daylen*nint)
      morzout = morzout/(daylen*nint)
#endif
      return

      contains
       function phi(u)
       real phi,u
       phi = (0.555588*u+0.04926*u**2)/(1.+0.188721*u)
       end function phi
      end
